"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-solutions-constructs/core");
function deployLambdaToOpenSearch(stack) {
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain'
    };
    return new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
}
function getDefaultTestLambdaProps() {
    return {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
    };
}
test('Check pattern properties', () => {
    const stack = new cdk.Stack();
    const construct = deployLambdaToOpenSearch(stack);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.openSearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudWatchAlarms).toBeDefined();
    expect(construct.openSearchRole).toBeDefined();
});
test('Check properties with no CloudWatch alarms ', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.openSearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudWatchAlarms).toBeUndefined();
    expect(construct.openSearchRole).toBeDefined();
});
test('Check that TLS 1.2 is the default', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        createCloudWatchAlarms: false
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        DomainEndpointOptions: {
            EnforceHTTPS: true,
            TLSSecurityPolicy: "Policy-Min-TLS-1-2-2019-07"
        },
    });
});
test('Check for an existing Lambda object', () => {
    const stack = new cdk.Stack();
    const existingLambdaObj = defaults.buildLambdaFunction(stack, {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            functionName: 'test-function'
        }
    });
    const props = {
        openSearchDomainName: 'test-domain',
        existingLambdaObj
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: 'test-function'
    });
});
test('Check Lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        domainEndpointEnvironmentVariableName: 'CUSTOM_DOMAIN_ENDPOINT'
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DOMAIN_ENDPOINT: {
                    'Fn::GetAtt': [
                        'testlambdaopensearchstackOpenSearchDomain46D6A86E',
                        'DomainEndpoint'
                    ]
                }
            }
        }
    });
});
test('Check domain name', () => {
    const stack = new cdk.Stack();
    deployLambdaToOpenSearch(stack);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain",
        UserPoolId: {
            Ref: "testlambdaopensearchstackCognitoUserPoolF5169460"
        }
    });
    template.hasResourceProperties('AWS::OpenSearchService::Domain', {
        DomainName: "test-domain",
    });
});
test('Check cognito domain name override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        cognitoDomainName: 'test-cogn-domain'
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::Cognito::UserPoolDomain', {
        Domain: 'test-cogn-domain'
    });
});
test('Check engine version override', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps: {
            engineVersion: 'OpenSearch_1.0',
        }
    };
    new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::OpenSearchService::Domain', {
        EngineVersion: 'OpenSearch_1.0'
    });
});
test("Test minimal deployment that deploys a VPC in 2 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {});
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaopensearchstackReplaceDefaultSecurityGroupsecuritygroup293B90D7",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                }
            ],
        },
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test minimal deployment that deploys a VPC in 3 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        deployVpc: true,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaopensearchstackReplaceDefaultSecurityGroupsecuritygroup293B90D7",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D",
                },
            ],
        },
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test cluster deploy to 1 AZ when user set zoneAwarenessEnabled to false", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const openSearchDomainProps = {
        clusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            zoneAwarenessEnabled: false,
            instanceCount: 3
        }
    };
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 1
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false,
        }
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                }
            ]
        }
    });
});
test("Test cluster deploy to 2 AZ when user set availabilityZoneCount to 2", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const openSearchDomainProps = {
        clusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            instanceCount: 2,
            zoneAwarenessEnabled: true,
            zoneAwarenessConfig: {
                availabilityZoneCount: 2
            }
        }
    };
    new lib_1.LambdaToOpenSearch(stack, "lambda-opensearch-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: 'test-domain',
        openSearchDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 2
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        ClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 2,
            },
            ZoneAwarenessEnabled: true,
        }
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
});
test('Test minimal deployment with an existing isolated VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false, {
        vpcName: "existing-isolated-vpc-test"
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        existingVpc: vpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-isolated-vpc-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing private VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = (0, core_1.getTestVpc)(stack, true, {
        vpcName: "existing-private-vpc-test"
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        existingVpc: vpc
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-private-vpc-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcPrivateSubnet1Subnet536B997A"
                },
                {
                    Ref: "VpcPrivateSubnet2Subnet3788AAA1"
                },
                {
                    Ref: "VpcPrivateSubnet3SubnetF258B56E"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with VPC construct props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const construct = new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        openSearchDomainName: "test",
        deployVpc: true,
        vpcProps: {
            vpcName: "vpc-props-test"
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "vpc-props-test"
            }
        ]
    });
    template.hasResourceProperties("AWS::OpenSearchService::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test error for vpcProps and undefined deployVpc prop', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            openSearchDomainName: "test",
            vpcProps: {
                vpcName: "existing-vpc-test"
            }
        });
    };
    expect(app).toThrowError("Error - deployVpc must be true when defining vpcProps");
});
test('Test error for Lambda function VPC props', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: defaults.consolidateProps(getDefaultTestLambdaProps(), {}, { vpc }),
            openSearchDomainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Lambda function props");
});
test('Confirm CheckOpenSearchProps is called', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            openSearchDomainProps: {
                vpcOptions: {
                    subnetIds: ['fake-ids'],
                    securityGroupIds: ['fake-sgs']
                }
            },
            openSearchDomainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not the OpenSearch Service props");
});
test('Test error for missing existingLambdaObj or lambdaFunctionProps', () => {
    const stack = new cdk.Stack();
    const props = {
        openSearchDomainName: 'test-domain'
    };
    try {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch-stack', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Confirm CheckVpcProps is being called', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-lambda-opensearch', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            openSearchDomainName: "test",
            vpcProps: {
                vpcName: "existing-vpc-test"
            },
            deployVpc: true,
            existingVpc: vpc
        });
    };
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        openSearchDomainName: 'name',
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToOpenSearch(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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