"""
In this file, we analyse the action library generated by Teacher1 and Teacher2.

Through

    1. filtering out actions that decrease rho less effectively
    2. filtering out actions that occur less frequently
    3. filtering out "do nothing" action
    4. add your filtering rules...,

we obtain an action space in the form of numpy array.

Credit: The methods are the enhanced methods of the original code, see
@https://github.com/AsprinChina/L2RPN_NIPS_2020_a_PPO_Solution
"""
import logging
from pathlib import Path
from typing import Union, Collection, List

import defopt
import grid2op
import numpy as np
import pandas as pd
from grid2op.Action import BaseAction
from grid2op.Environment import BaseEnv

from curriculumagent.teacher.submodule.encoded_action import EncodedTopologyAction


def read_experience(experience_files: Union[Path, Collection[Path]]) -> pd.DataFrame:
    """Read experience data from csv file and concat it if needed.

    Args:
        experience_files:Either one file or multiple files with experience data.

    Returns: Dataframe with experience.

    """
    data = None
    for exp_file in experience_files:
        if data is None:
            data = pd.read_csv(exp_file, on_bad_lines='skip')
        else:
            data = pd.concat((data, pd.read_csv(exp_file, on_bad_lines='skip')))
    return data


def filter_good_experience(data: pd.DataFrame) -> pd.DataFrame:
    """Filter actions based on criteria.

    The actions are either filtered if:
        1. The action decreases the rho by less then 2%
        2. The action does nothing

    Args:
        data: Input of experience

    Returns: Reduced dataframe with filtered action/experience

    """
    good_samples = (data['rho_improvement'] > 0.02)
    return data[good_samples]


def rank_actions_simple(data: pd.DataFrame, env: BaseEnv, best_n: int = 416) -> List[BaseAction]:
    """Rank the given experience data by counting and sorting them and cutting of the best_n actions from the top.

    Args:
        data: Loaded experience data.
        env: The environment used to generate the experience of data.
        best_n: The cutoff point determining the maximum number of actions returned.

    Returns: The ranked actions as a list.
    """
    decode_action = np.vectorize(lambda act: EncodedTopologyAction.decode_action(act, env))

    # Resort actions
    best_action_freq = data['best_action'].value_counts()
    logging.info(f"There are {len(best_action_freq)} unique actions")

    best_actions = best_action_freq[:best_n]
    best_actions_array = decode_action(best_actions.index.to_numpy())

    return list(best_actions_array)



def save_actionspace_binbinchen(save_path: Path, actions: List[BaseAction]):
    """Save the given actionspace as a numpy array file(which was used by the binbinchen tutor).

    Args:
        save_path: The path where to save the file to.
        actions: The actionspace to save.
    """
    sample = actions[0].to_vect()
    action_space = np.zeros((len(actions), sample.shape[0]))
    for i, action in enumerate(actions):
        action_space[i] = action.to_vect()

    np.save(str(save_path), action_space)
    logging.info(f'Save an action space with the size of {action_space.shape[0]:d}')


def make_unitary_actionspace(action_space_file_path: Path,
                             experience_csv_files: List[Path],
                             env_name_path: Union[Path, str],
                             best_n: int):
    """
    Using the provided experience files generate a unitary space file for the tutor.

    Args:
        action_space_file_path: Where to save the action space file containing the best actions.
        experience_csv_files: The csv files containing the experience from the teachers.
        env_name_path: The name or path to the environment that is used for experience generation.
        best_n: The best_n actions to keep.

    Returns: None, but data is saved under action_space_file_path.
    """

    data = read_experience(experience_csv_files)
    logging.info(f"Read {len(data)} experience samples from {experience_csv_files}")
    data = filter_good_experience(data)
    logging.info(f"Kept {len(data)} after filtering")

    env = grid2op.make(env_name_path)

    actions = rank_actions_simple(data, env, best_n=best_n)
    save_actionspace_binbinchen(action_space_file_path, actions)
    logging.info(f"Saved {len(actions)} actions after ranking with best_n={best_n} to {action_space_file_path}")


if __name__ == "__main__":
    log_format = '(%(asctime)s) [%(name)-10s] %(levelname)8s: %(message)s [%(filename)s:%(lineno)s]'
    logging.basicConfig(level=logging.INFO, format=log_format)
    defopt.run({'unitary': make_unitary_actionspace})
