"""
In this file, a neural network is developed to fit the dataset generated by Tutor.
Depending on the observation space and action space, the tutor model can/has to be
adjusted.

The junior model returns a one-hot encoded output, based on the number of actions.

Credit: The junior is a more general approach of the original code, see
@https://github.com/AsprinChina/L2RPN_NIPS_2020_a_PPO_Solution
"""
import logging
import time
from pathlib import Path
from typing import Union, Optional, Tuple, List

import numpy as np
import pkg_resources
#  Due to some import troubles with ray, it is essential to define the correct tensorflow package or
# else the Callbacks are from keras v1
pkg_resources.require("tensorflow>=2.7.0")
pkg_resources.require("keras>=2.7.0")
import tensorflow as tf
from tensorflow.python.keras.callbacks import TensorBoard,ModelCheckpoint,Callback,EarlyStopping

class Junior():
    def __init__(self, trainset_size: int, epochs: Optional[int] = 1000,
                 num_actions: int = 208,
                 learning_rate: Optional[Union[float, tf.keras.optimizers.schedules.LearningRateSchedule]] = 5e-4,
                 activation: Optional[str] = "relu",
                 seed: Optional[int] = None):
        """ Constructor of the junior model

        Next to setting all variables, the init requires the size of the train set and optionally
        the number of epochs.

        Args:
            trainset_size: Size of the training data
            epochs: Number of epochs to run the training
            num_actions: Number of actions. Necessary for the action space
            learning_rate: Optional learning rate or LearningRateScheduler
            activation: Optional activation
            seed: Optional Seed to reproduce results
        """
        if isinstance(activation, str):
            if activation == "leaky_relu":
                activation = tf.keras.layers.LeakyReLU()
            elif activation == "relu":
                activation = tf.keras.layers.ReLU()
        else:
            logging.warning("Wrong input type of activation. Take relu instead")
            activation = tf.keras.layers.ReLU()

        # Seed:
        self.seed = seed
        if self.seed:
            np.random.seed(self.seed)
            tf.random.set_seed(self.seed)

        self.n_cell = 1000
        self.actions = num_actions
        self.lr = learning_rate
        self.activation = activation
        self.lf = tf.keras.losses.SparseCategoricalCrossentropy()
        self.initializer = tf.keras.initializers.Orthogonal()
        self.batch_size = 256
        self.trainset_size = trainset_size
        self.epochs = epochs
        self.model = self._build_model()
        self.callback = []

    def _build_model(self) -> tf.keras.models.Sequential:
        """ Build and return the junior network as a keras model

        Returns: Compiled Keras model

        """
        model = tf.keras.models.Sequential([
            tf.keras.layers.Dense(self.n_cell, activation=self.activation, kernel_initializer=self.initializer),
            tf.keras.layers.Dense(self.n_cell, activation=self.activation, kernel_initializer=self.initializer),
            tf.keras.layers.Dense(self.n_cell, activation=self.activation, kernel_initializer=self.initializer),
            tf.keras.layers.Dropout(0.25),
            tf.keras.layers.Dense(self.n_cell, activation=self.activation, kernel_initializer=self.initializer),
            tf.keras.layers.Dropout(0.25),
            tf.keras.layers.Dense(self.actions, activation='softmax')
        ])

        model.compile(optimizer=tf.keras.optimizers.Adam(learning_rate=self.lr),
                      loss=self.lf,
                      metrics=['accuracy'])
        return model

    def train(self, x_train: np.ndarray, y_train: np.ndarray,
              x_validate: np.ndarray, y_validate: np.ndarray,
              log_dir: Optional[Union[str, Path]] = None,
              ckpt_dir: Optional[Union[str, Path]] = None,
              patience: Optional[int] = None,
              epochs: Optional = None) -> tf.keras.callbacks.History:
        """ Train the junior model for given number of epochs.

        This method builds callbacks for the training and then executes the keras .fit() method
        to train the Junior model on the x_train,y_train data. Validation is recorded as well.

        Args:
            log_dir: directory for tensorboard callback
            ckpt_dir: directory for checkpoint callback
            x_train: training data containing the grid observations
            y_train: training actions of the tutor
            x_validate: validation data containing the grid observations
            y_validate: validation actions of the tutor
            epochs: number of epochs for the training
            patience: optional early stopping criterion.

        Returns: Returns training history.

        """
        callbacks = self.callback

        logging.warning(f"{tf.__version__}")
        logging.warning(f"{tf.keras.__version__}")
        if log_dir is not None:
            tensorboard_callback = TensorBoard(log_dir=log_dir)
            callbacks += [tensorboard_callback]
        if isinstance(ckpt_dir, (Path,str)):
            if isinstance(ckpt_dir, str):
                ckpt_path = ckpt_dir + "/" + "ckpt_{epoch}"
            else:
                ckpt_path = ckpt_dir / "ckpt_{epoch}"

            cp_callback = ModelCheckpoint(filepath=ckpt_path,
                                                             save_weights_only=False,
                                                             period=10,
                                                             verbose=1)
            callbacks += [cp_callback]

        if patience is not None:
            early_stopping = EarlyStopping(monitor="val_loss",
                                                              patience=patience,
                                                              verbose=1,
                                                              mode="auto",
                                                              restore_best_weights=True,
                                                              )
            callbacks += [early_stopping]

        for call in callbacks:
            assert isinstance(call, Callback)

        history = self.model.fit(x=x_train,
                                 y=y_train,
                                 epochs=epochs or self.epochs,
                                 validation_data=(x_validate, y_validate),
                                 batch_size=self.batch_size,
                                 callbacks=callbacks)
        return history

    def test(self, x: np.ndarray, y: np.ndarray, save_path: Optional[Path] = None) -> dict:
        """ Test the junior model with input dataset x and targets/actions y.

        The method predicts based on the x input and then computes a ranking, regarding the
        accuracy on the actions.

        The ranking collects, whether the action of the tutor was within the 1-20 actions

        Args:
            x: input with tutor observation for the prediction
            y: action of the tutor to validate with the prediction
            save_path: optional, path where the weights of the model are saved.
                If wanted, the weights are loaded by model.load_weights(save_path)

        Returns: The dictionary that contains the top values

        """
        if isinstance(save_path, Path):
            self.model = tf.keras.models.load_model(save_path)
            logging.info(f"Imported model from{save_path}")

        a_pred = self.model.predict(x, verbose=1)
        top_n = []
        for i in range(a_pred.shape[0]):
            top_n.append(a_pred[i, :].argsort()[-20:])

        # Added accuracy to record the prediction performance
        accuracy = {}

        for n in range(1, 21):
            correct = 0
            for i in range(a_pred.shape[0]):
                if y[i, 0] in top_n[i][-n:]:
                    correct += 1

            acc = correct / a_pred.shape[0] * 100
            logging.info(f'accuracy of top-{n} is {acc}')

            accuracy['accuracy of top-%d' % n] = correct / a_pred.shape[0] * 100
        return accuracy


def load_dataset(dataset_path: Union[str, Path], dataset_name: str) \
        -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray, np.ndarray, np.ndarray]:
    """ Function for loading the datasets of the Tutor run

    Note that the training, validation and testing datasets all have the same name but different
    endings.
    As example:
    dataset_name = data
    training: data_train.npz
    validation: data_val.npz
    test: data_test.npz

    Args:
        dataset_path: path where to find the Tutor results
        dataset_name: Name of the tutor results.

    Returns: Tuple, containing np.ndarrays with training, val and test data

    """
    train_path = dataset_path / f'{dataset_name}_train.npz'
    train_data = np.load(train_path)

    validation_path = dataset_path / f'{dataset_name}_val.npz'
    validation_data = np.load(validation_path)

    test_path = dataset_path / f'{dataset_name}_test.npz'
    test_data = np.load(test_path)

    return (train_data['s_train'], train_data['a_train'], validation_data['s_validate'], validation_data['a_validate'],
            test_data['s_test'], test_data['a_test'])


def train(run_name: str, dataset_path: Path,
          target_model_path: Path,
          action_space_file: Optional[Union[Path, List[Path]]] = None,
          dataset_name: str = 'junior_dataset',
          epochs: int = 1000, seed: Optional[int] = None) -> tf.keras.callbacks.History:
    """
    Loads the dataset and then trains the JuniorModel with the given dataset and hyper-parameters.

    Args:
        run_name: The name of the training run.
        dataset_path: Path to the dataset files.
        action_space_file: Optional action space file of the tutor.
        dataset_name: The name of the dataset in {dataset_name}_train.npz.
        epochs: The number of epochs to train.
        seed: Random seed to set for the training.

    Returns: Training history in Keras format

    """
    if not target_model_path.is_dir():
        logging.warning(f'{target_model_path} does not exists yet. Create directory')
        target_model_path.mkdir(parents=True, exist_ok=True)

    log_dir = target_model_path / 'tfboard' / time.strftime("%m-%d-%H-%M", time.localtime())
    ckpt_dir = target_model_path / f'ckpt-{run_name}'

    s_train, a_train, s_validate, a_validate, _, a_test = load_dataset(dataset_path, dataset_name)

    # Get maximum number of actions:
    if action_space_file is None:
        max_action_value = np.max([np.max(a_train), np.max(a_validate), np.max(a_test)]) + 1
    else:
        if isinstance(action_space_file, Path):
            assert action_space_file.is_file()
            actions = np.load(str(Path(action_space_file)))
            max_action_value = len(actions)

        elif isinstance(action_space_file, list):
            for act_path in action_space_file:
                assert act_path.is_file()
            actions = [np.load(str(act_path)) for act_path in action_space_file]
            max_action_value = 0
            for act in actions:
                max_action_value += len(act)

    logging.info(f"A total of {max_action_value} actions are assumed, based on the action_space_file input.")

    steps = (len(s_train) * epochs) / 256
    lr_schedule = tf.keras.optimizers.schedules.PolynomialDecay(
        5e-4, steps, end_learning_rate=1e-4, power=1.0,
        cycle=False, name=None
    )

    junior = Junior(trainset_size=len(s_train),
                    epochs=epochs,
                    num_actions=max_action_value,
                    learning_rate=lr_schedule,
                    activation="relu",
                    seed=seed)
    history = junior.train(x_train=s_train, y_train=a_train, x_validate=s_validate, y_validate=a_validate,
                           log_dir=log_dir,
                           ckpt_dir=ckpt_dir,
                           patience=None, epochs=epochs)
    # Save model
    junior.model.save(target_model_path)

    return history


def validate(checkpoint_path: Path, dataset_path: Path, dataset_name: str = 'junior_dataset',
             action_space_file: Optional[Union[Path, List[Path]]] = None) -> dict:
    """
    Test the given checkpoint against the test data set

    Args:
        checkpoint_path: The checkpoint file to use for the conversion/input.
        dataset_path: Path to the dataset used to train the checkpoint.
        dataset_name: The name of the dataset in {dataset_name}_test.npz.
        action_space_file: Optional action space file of the tutor. This is relevant if multiple
            action sets were used in the tutor optimization. If no action_space_file is provided, it is assumed
            that only one action file exists. Then the maximum action is taken from the
            training/val/test data.
    Returns: Accuracy dictionary of the testing dataset

    """

    assert checkpoint_path.is_dir()

    s_train, a_train, s_validate, a_validate, s_test, a_test = load_dataset(dataset_path, dataset_name)

    if action_space_file is None:
        max_action_value = np.max([np.max(a_train), np.max(a_validate), np.max(a_test)]) + 1
    else:
        if isinstance(action_space_file, Path):
            assert action_space_file.is_file()
            actions = np.load(str(Path(action_space_file)))
            max_action_value = len(actions)

        elif isinstance(action_space_file, list):
            for act_path in action_space_file:
                assert act_path.is_file()
            actions = np.concatenate([np.load(str(act_path)) for act_path in action_space_file],axis=0)
            max_action_value = len(actions)


    logging.info(f"A total of {max_action_value} are assumed, based on the action_space_file input.")

    junior = Junior(trainset_size=len(s_train),
                    epochs=0,
                    num_actions=max_action_value)
    accuracy = junior.test(x=s_test, y=a_test, save_path=checkpoint_path)
    return accuracy

if __name__ == '__main__':
    """
    If executed, this code runs the train, convert and test method for the junior model. 
    """
    logging.basicConfig(level=logging.INFO)
    # Don't use all the gpu memory from the beginning but grow it
    physical_devices = tf.config.list_physical_devices('GPU')
    for gpu_instance in physical_devices:
        tf.config.experimental.set_memory_growth(gpu_instance, True)

    import defopt

    defopt.run({
        'train': train,
        'test': validate,
    })
