
import os
import sqlite3
import json
import uuid
import logging
from flask import Flask, request, jsonify, send_file, render_template_string
import matplotlib.pyplot as plt
import io
import base64

class WebServer:
    def __init__(self, app_name="NeuraPy Web Server"):
        self.app = Flask(app_name)
        self.registered_routes = set()
        logging.basicConfig(level=logging.INFO, format="%(asctime)s [%(levelname)s] %(message)s")

# -------------------------------------------------------------------
# Helper to avoid duplicate route names
# -------------------------------------------------------------------
    def _unique_route_name(self, route):
        """Generate a unique internal endpoint name for each route."""
        base = route.strip("/").replace("/", "_") or "root"
        unique_name = f"{base}_{uuid.uuid4().hex[:6]}"
        return unique_name

# -------------------------------------------------------------------
# Basic Route Handler
# -------------------------------------------------------------------
    def simple_route(self, route='/', code="NeuraPy web app", html_file_path=""):
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, endpoint=endpoint_name)
        def webpage():
            if html_file_path:
                if os.path.exists(html_file_path):
                    with open(html_file_path, "r", encoding="utf-8") as f:
                        return f.read()
                return "No file found", 404
            return code or "No content provided"

        logging.info(f"Registered route: {route}")

# -------------------------------------------------------------------
# Error Handler
# -------------------------------------------------------------------
    def error_handler(self, error_code, code="NeuraPy Error Page", error_page_html=""):
        @self.app.errorhandler(error_code)
        def error(e):
            if error_page_html and os.path.exists(error_page_html):
                with open(error_page_html, "r", encoding="utf-8") as f:
                    return f.read(), error_code
            return code or f"Error {error_code}", error_code

# -------------------------------------------------------------------
# JSON Verification (for APIs)
# -------------------------------------------------------------------
    def verify_details(self, route='/', user_data=[], verify_data_from=[]):
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, methods=['POST'], endpoint=endpoint_name)
        def verify():
            data = request.get_json(force=True)
            if not data:
                return jsonify({"response": False, "error": "No JSON received"})
            if len(user_data) != len(verify_data_from):
                return jsonify({"response": False, "error": "Mismatch in data length"})

            validation = [data.get(k) == v for k, v in zip(user_data, verify_data_from)]
            return jsonify({"response": all(validation)})

# -------------------------------------------------------------------
# SQLite Database Setup & Operations
# -------------------------------------------------------------------
    def DataBase(self, db_path="neurapy.db", query=None):
        query = query or """
            CREATE TABLE IF NOT EXISTS NeuraPy (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                name TEXT NOT NULL,
                age INTEGER,
                email TEXT UNIQUE
            );
        """
        conn = sqlite3.connect(db_path)
        cursor = conn.cursor()
        cursor.execute(query)
        conn.commit()
        conn.close()
        logging.info(f"Database initialized at {db_path}")

    def insert_data(self, db_path, table, data: dict):
        with sqlite3.connect(db_path) as conn:
            cursor = conn.cursor()
            cols = ', '.join(data.keys())
            placeholders = ', '.join(['?'] * len(data))
            vals = tuple(data.values())
            cursor.execute(f"INSERT INTO {table} ({cols}) VALUES ({placeholders})", vals)
            conn.commit()
        return jsonify({"status": "Data inserted successfully"})

    def retrieve_data(self, db_path, query):
        with sqlite3.connect(db_path) as conn:
            cursor = conn.cursor()
            cursor.execute(query)
            data = cursor.fetchall()
        return jsonify(data)

    def delete_data(self, db_path, table, condition):
        with sqlite3.connect(db_path) as conn:
            cursor = conn.cursor()
            cursor.execute(f"DELETE FROM {table} WHERE {condition}")
            conn.commit()
        return jsonify({"status": "Record deleted"})

# -------------------------------------------------------------------
# File Upload Endpoint
# -------------------------------------------------------------------
    def file_upload(self, route='/upload', upload_folder='uploads'):
        os.makedirs(upload_folder, exist_ok=True)
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, methods=['POST'], endpoint=endpoint_name)
        def upload():
            if 'file' not in request.files:
                return jsonify({"error": "No file part"})
            file = request.files['file']
            if file.filename == '':
                return jsonify({"error": "No selected file"})
            path = os.path.join(upload_folder, file.filename)
            file.save(path)
            return jsonify({"success": True, "path": path})

# -------------------------------------------------------------------
# Serve Static or Dynamic Files
# -------------------------------------------------------------------
    def serve_file(self, route='/file', file_path=""):
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, endpoint=endpoint_name)
        def serve():
            if os.path.exists(file_path):
                return send_file(file_path)
            return "File not found", 404

# -------------------------------------------------------------------
# Vector Plotting API
# -------------------------------------------------------------------
    def vector_plot(self, route='/plot_vector'):
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, methods=['POST'], endpoint=endpoint_name)
        def plot_vector():
            data = request.get_json(force=True)
            vector = data.get("vector", [0, 0])
            origin = data.get("origin", [0, 0])
            color = data.get("color", "r")

            fig, ax = plt.subplots()
            ax.quiver(origin[0], origin[1], vector[0], vector[1],
                      angles='xy', scale_units='xy', scale=1, color=color)
            ax.set_xlim(-10, 10)
            ax.set_ylim(-10, 10)
            ax.grid(True)
            ax.axhline(0, color='black', linewidth=0.5)
            ax.axvline(0, color='black', linewidth=0.5)

            buf = io.BytesIO()
            plt.savefig(buf, format='png')
            buf.seek(0)
            encoded = base64.b64encode(buf.read()).decode('utf-8')
            plt.close(fig)
            return jsonify({"image_base64": encoded})

# -------------------------------------------------------------------
# Health Check Endpoint
# -------------------------------------------------------------------
    def health(self, route="/health"):
        endpoint_name = self._unique_route_name(route)

        @self.app.route(route, endpoint=endpoint_name)
        def health_check():
            return jsonify({"status": "online", "message": "Server is healthy"})

# -------------------------------------------------------------------
# Run Server
# -------------------------------------------------------------------
    def run(self, live_refresh=True, port=5200):
        logging.info(f"Server starting on port {port} ...")
        self.app.run(debug=live_refresh, host='0.0.0.0', port=port)
