import math
import matplotlib.pyplot as plt

class Vector2D:
    def __init__(self, x: float = 0, y: float = 0):
        self.x = x
        self.y = y

    def __repr__(self):
        return f"Vector2D({self.x}, {self.y})"
    def to_list(self):
        return [self.x,self.y]

    def add(self, *vectors):
        new_x, new_y = self.x, self.y
        for v in vectors:
            new_x += v.x
            new_y += v.y
        return Vector2D(new_x, new_y)

    def subtract(self, *vectors):
        new_x, new_y = self.x, self.y
        for v in vectors:
            new_x -= v.x
            new_y -= v.y
        return Vector2D(new_x, new_y)


    @staticmethod
    def dot_product(v1, v2):
        return v1.x * v2.x + v1.y * v2.y


    @staticmethod
    def cross_product(v1, v2):
        return v1.x * v2.y - v1.y * v2.x

    def magnitude(self):
        return math.sqrt(self.x**2 + self.y**2)

    def unit_vector(self):
        mag = self.magnitude()
        if mag == 0:
            raise ValueError("Cannot compute unit vector of zero vector.")
        return Vector2D(self.x / mag, self.y / mag)

    @staticmethod
    def angle_between(v1, v2 ,angle:bool=True):
        dot = Vector2D.dot_product(v1, v2)
        mag1 = v1.magnitude()
        mag2 = v2.magnitude()
        if mag1 == 0 or mag2 == 0:
            raise ValueError("Cannot compute angle with zero-length vector.")
        cos_theta = dot / (mag1 * mag2)
        cos_theta = max(-1, min(1, cos_theta))
        if angle:
            return math.degrees(math.acos(cos_theta))
        else:
            return math.acos(cos_theta)
    
    @classmethod
    def from_list(cls, data):
        if len(data) != 2:
            raise ValueError("List must contain exactly 2 values.")
        return cls(data[0], data[1])

    def plot(self, *others, color='r', show=True, title="2D Vector Plot"):
        """
        Plot this vector (and optionally others) from the origin.
        """
        plt.figure()
        ax = plt.gca()
        vectors = [self] + list(others)
        colors = ['r', 'b', 'g', 'm', 'c', 'y', 'k']

        for i, v in enumerate(vectors):
            col = colors[i % len(colors)] if len(vectors) > 1 else color
            ax.quiver(0, 0, v.x, v.y, angles='xy', scale_units='xy', scale=1, color=col, label=str(v))

        limit = max(max(abs(v.x), abs(v.y)) for v in vectors) + 1
        ax.set_xlim(-limit, limit)
        ax.set_ylim(-limit, limit)
        ax.axhline(0, color='black', linewidth=0.5)
        ax.axvline(0, color='black', linewidth=0.5)
        plt.grid(True)
        plt.title(title)
        plt.legend()
        if show:
            plt.show()


class Vector3D:
    def __init__(self, x: float = 0, y: float = 0, z: float = 0):
        self.x = x
        self.y = y
        self.z = z

    def __repr__(self):
        return f"Vector3D({self.x}, {self.y}, {self.z})"


    def add(self, *vectors):
        new_x, new_y, new_z = self.x, self.y, self.z
        for v in vectors:
            new_x += v.x
            new_y += v.y
            new_z += v.z
        return Vector3D(new_x, new_y, new_z)


    def subtract(self, *vectors):
        new_x, new_y, new_z = self.x, self.y, self.z
        for v in vectors:
            new_x -= v.x
            new_y -= v.y
            new_z -= v.z
        return Vector3D(new_x, new_y, new_z)

    @staticmethod
    def dot_product(v1, v2):
        return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z

    @staticmethod
    def cross_product(v1, v2):
        cx = v1.y * v2.z - v1.z * v2.y
        cy = v1.z * v2.x - v1.x * v2.z
        cz = v1.x * v2.y - v1.y * v2.x
        return Vector3D(cx, cy, cz)

    def magnitude(self):
        return math.sqrt(self.x**2 + self.y**2 + self.z**2)

    def unit_vector(self):
        mag = self.magnitude()
        if mag == 0:
            raise ValueError("Cannot compute unit vector of zero vector.")
        return Vector3D(self.x / mag, self.y / mag, self.z / mag)

    @staticmethod
    def angle_between(v1, v2, degrees=True):
        dot = Vector3D.dot_product(v1, v2)
        mag1 = v1.magnitude()
        mag2 = v2.magnitude()
        if mag1 == 0 or mag2 == 0:
            raise ValueError("Cannot compute angle with zero-length vector.")
        cos_theta = dot / (mag1 * mag2)
        cos_theta = max(-1, min(1, cos_theta))
        angle = math.acos(cos_theta)
        return math.degrees(angle) if degrees else angle

    def to_list(self):
        return [self.x, self.y, self.z]

    @classmethod
    def from_list(cls, data):
        if len(data) != 3:
            raise ValueError("List must contain exactly 3 values.")
        return cls(data[0], data[1], data[2])
    
    def plot(self, *others, color='r', show=True, title="3D Vector Plot"):
        """
        Plot this vector (and optionally others) in 3D space.
        """
        fig = plt.figure()
        ax = fig.add_subplot(111, projection='3d')
        vectors = [self] + list(others)
        colors = ['r', 'b', 'g', 'm', 'c', 'y', 'k']

        for i, v in enumerate(vectors):
            col = colors[i % len(colors)] if len(vectors) > 1 else color
            ax.quiver(0, 0, 0, v.x, v.y, v.z, color=col, label=str(v))

        limit = max(max(abs(v.x), abs(v.y), abs(v.z)) for v in vectors) + 1
        ax.set_xlim(-limit, limit)
        ax.set_ylim(-limit, limit)
        ax.set_zlim(-limit, limit)
        ax.set_xlabel('X')
        ax.set_ylabel('Y')
        ax.set_zlabel('Z')
        plt.title(title)
        plt.legend()
        if show:
            plt.show()
        