
#_____________________________________Sensors_______________________________________
import psutil
import sounddevice as sd
import numpy as np
import screen_brightness_control as sbc
import platform
import time
import cv2

class PhysicalSensors:
    def __init__(self):
        self.system = platform.system()
        print(f"🛰️ PhysicalSensors initialized on {self.system}")

    # ---------------- SOUND SENSOR ----------------
    def sound_level(self, duration=1):
        """Measure average microphone sound level (RMS)."""
        try:
            data = sd.rec(int(duration * 44100), samplerate=44100, channels=1, dtype='float64')
            sd.wait()
            rms = np.sqrt(np.mean(np.square(data)))
            return round(rms * 1000, 3)
        except Exception as e:
            return f"Sound sensor not accessible: {e}"

    # ---------------- LIGHT SENSOR ----------------
    def brightness(self):
        """Get display brightness (cross-platform)."""
        try:
            brightness = sbc.get_brightness(display=0)
            return brightness[0] if brightness else "Unknown"
        except Exception as e:
            return f"Brightness sensor not accessible: {e}"

    # ---------------- TEMPERATURE SENSOR ----------------
    def cpu_temperature(self):
        """Get CPU temperature if available."""
        try:
            temps = psutil.sensors_temperatures()
            if not temps:
                return "No temperature sensors found"
            # Return first available CPU temp
            for name, entries in temps.items():
                for entry in entries:
                    if "cpu" in name.lower() or "core" in name.lower():
                        label = entry.label if entry.label else name
                        return f"{label}: {entry.current:.1f}°C"
            return "CPU temperature sensor not found"
        except Exception as e:
            return f"Temperature sensor error: {e}"

    # ---------------- BATTERY SENSOR ----------------
    def battery_status(self):
        """Get battery percentage and charging status."""
        try:
            battery = psutil.sensors_battery()
            if not battery:
                return "Battery not found"
            time_left = battery.secsleft if battery.secsleft != psutil.POWER_TIME_UNLIMITED else "Unlimited"
            return {
                "percentage": battery.percent,
                "charging": battery.power_plugged,
                "time_left": time_left
            }
        except Exception as e:
            return f"Battery sensor error: {e}"

    # ---------------- SYSTEM INFO ----------------
    def system_info(self):
        """Return basic system info."""
        return {
            "os": self.system,
            "cpu_usage": psutil.cpu_percent(interval=1),
            "memory_usage": psutil.virtual_memory().percent
        }

    # ---------------- OPTIONAL MOTION DETECTION ----------------
    def motion_detected(self, camera_index=0, threshold=300000):
        """Detect movement using webcam frame difference (optional)."""
        try:
            cap = cv2.VideoCapture(camera_index)
            if not cap.isOpened():
                return "Camera not accessible"

            ret, frame1 = cap.read()
            ret, frame2 = cap.read()
            motion = False

            while ret:
                diff = cv2.absdiff(frame1, frame2)
                gray = cv2.cvtColor(diff, cv2.COLOR_BGR2GRAY)
                blur = cv2.GaussianBlur(gray, (5,5), 0)
                _, thresh = cv2.threshold(blur, 20, 255, cv2.THRESH_BINARY)
                motion_value = np.sum(thresh)

                if motion_value > threshold:
                    motion = True
                    break

                frame1 = frame2
                ret, frame2 = cap.read()

            cap.release()
            cv2.destroyAllWindows()
            return "Movement detected" if motion else "No significant motion"
        except Exception as e:
            return f"Motion detection error: {e}"


