
#_____________________________Quantum_______________________________
import numpy as np
import math
import cmath
from concurrent.futures import ThreadPoolExecutor, as_completed
from functools import reduce
import itertools
import random
import time
import scipy
# Constants
HBAR = 1.054571817e-34
ELECTRON_MASS = 9.10938356e-31
E_CHARGE = 1.602176634e-19
PI = math.pi


# Helper utilities
def kron(*mats):
    """Kronecker product of multiple matrices."""
    return reduce(np.kron, mats)

def normalize(vec):
    """Normalize complex vector."""
    norm = np.linalg.norm(vec)
    return vec / norm if norm > 0 else vec

def random_complex_state(n):
    """Generate random normalized quantum state of n qubits."""
    state = np.random.rand(2**n) + 1j*np.random.rand(2**n)
    return normalize(state)


# _______________________________Quantum Calculations (1.2.4)_______________________________ #
import numpy as np
import math, cmath, random, time
from functools import reduce
from concurrent.futures import ThreadPoolExecutor, as_completed
import itertools
import scipy.linalg

# Constants
HBAR = 1.054571817e-34
ELECTRON_MASS = 9.10938356e-31
E_CHARGE = 1.602176634e-19
PI = math.pi

# Utility helpers
def kron(*mats): return reduce(np.kron, mats)
def normalize(vec): return vec / np.linalg.norm(vec) if np.linalg.norm(vec) != 0 else vec
def dagger(M): return np.conjugate(M.T)
def random_state(n): return normalize(np.random.rand(2**n) + 1j*np.random.rand(2**n))

# _________________________________ Quantum Calculations _________________________________ #
class QuantumCalculation:
    """
    Handles theoretical physics and chemistry quantum calculations:
    - Potential wells
    - Quantum harmonic oscillators
    - Energy levels
    - Tunneling
    - Uncertainty, interference, probability
    - Wavepacket evolution
    """

    def __init__(self):
        self.history = {}

    # ---------- Fundamental Mechanics ----------
    def wkb_tunneling(self, E, V0, a, m=ELECTRON_MASS):
        if E >= V0: return 1.0
        kappa = np.sqrt(2*m*(V0 - E)) / HBAR
        T = np.exp(-2*kappa*a)
        return float(T)

    def infinite_square_well_energy(self, n, L, m=ELECTRON_MASS):
        return (n**2 * PI**2 * HBAR**2) / (2*m*L**2)

    def harmonic_oscillator_energy(self, n, omega):
        return HBAR * omega * (n + 0.5)

    def particle_in_box_wavefunc(self, n, L, x):
        return np.sqrt(2/L) * np.sin(n * PI * x / L)

    def uncertainty_relation(self, sigma_x, sigma_p):
        return sigma_x * sigma_p >= HBAR / 2

    def commutator(self, A, B):
        return A @ B - B @ A

    def expectation_value(self, psi, operator):
        return np.vdot(psi, operator @ psi).real

    def interference_pattern(self, amp1, amp2):
        return np.abs(amp1 + amp2)**2

    # ---------- Advanced Numerical ----------
    def eigenvalues_and_vectors(self, H):
        vals, vecs = np.linalg.eigh(H)
        return {'energies': vals, 'states': vecs}

    def potential_energy_curve(self, potential_func, x_range):
        return np.array([potential_func(x) for x in x_range])

    def wavepacket(self, x, x0, p0, sigma):
        pref = (1/(sigma * np.sqrt(2*PI)))
        return pref * np.exp(-(x-x0)**2/(2*sigma**2)) * np.exp(1j*p0*(x-x0)/HBAR)

    def propagate_wavepacket(self, psi, H, t):
        U = scipy.linalg.expm(-1j * H * t / HBAR)
        return U @ psi

    def probability_density(self, psi):
        return np.abs(psi)**2

    def expectation_potential(self, psi, V):
        return np.real(np.vdot(psi, V * psi))

    def density_matrix(self, psi):
        return np.outer(psi, np.conjugate(psi))

    def entanglement_entropy(self, rho):
        vals = np.linalg.eigvalsh(rho)
        vals = np.clip(vals, 1e-12, 1)
        return -np.sum(vals*np.log2(vals))

    # ---------- Quantum Chemistry ----------
    def hamiltonian_hydrogen_like(self, Z, n):
        """Simplified hydrogenic energy levels: E = -13.6 * Z² / n² eV"""
        return -13.6 * (Z**2) / (n**2)

    def molecular_overlap(self, psi1, psi2):
        return np.abs(np.vdot(psi1, psi2))**2

    def coulomb_potential(self, q1, q2, r):
        return (q1*q2) / (4 * PI * 8.854e-12 * r)

    def run_parallel(self, funcs):
        results = {}
        with ThreadPoolExecutor(max_workers=len(funcs)) as ex:
            futures = {ex.submit(func): name for name, func in funcs.items()}
            for f in as_completed(futures):
                results[futures[f]] = f.result()
        return results


# _________________________________ Quantum Computation _________________________________ #
class QuantumComputing:
    """
    Handles algorithmic and computational quantum logic operations:
    - Gate simulation
    - QFT, Grover, Teleportation
    - Measurement, density matrix, entropy
    - Variational circuits, entanglement detection
    """

    def __init__(self):
        self.I = np.eye(2, complex)
        self.X = np.array([[0,1],[1,0]], complex)
        self.Y = np.array([[0,-1j],[1j,0]], complex)
        self.Z = np.array([[1,0],[0,-1]], complex)
        self.H = (1/np.sqrt(2))*np.array([[1,1],[1,-1]], complex)
        self.S = np.array([[1,0],[0,1j]], complex)
        self.T = np.array([[1,0],[0, np.exp(1j*PI/4)]], complex)
        self.gates = {'I':self.I, 'X':self.X, 'Y':self.Y, 'Z':self.Z, 'H':self.H}

    def create_state(self, n):
        s = np.zeros(2**n, complex); s[0]=1; return s

    def apply_gate(self, state, gate, target, n):
        ops=[self.I]*n; ops[target]=gate
        return kron(*ops) @ state

    def controlled_gate(self, state, control, target, gate, n):
        P0=np.array([[1,0],[0,0]]); P1=np.array([[0,0],[0,1]])
        op0=[self.I]*n; op1=[self.I]*n
        op0[control]=P0; op1[control]=P1; op1[target]=gate
        U=kron(*op0)+kron(*op1)
        return U@state

    def qft(self, state, n):
        N=2**n
        F=np.array([[np.exp(2j*PI*j*k/N) for k in range(N)] for j in range(N)]) / np.sqrt(N)
        return F @ state

    def grover(self, n, marked):
        N=2**n; s=np.ones(N)/np.sqrt(N); s[marked]*=-1
        return 2*s*np.vdot(np.ones(N)/np.sqrt(N), s)-s

    def teleportation(self, psi):
        bell=(1/np.sqrt(2))*np.array([1,0,0,1], complex)
        state=kron(psi,bell)
        state=self.controlled_gate(state,0,1,self.X,3)
        state=self.apply_gate(state,self.H,0,3)
        return normalize(state)

    def measure(self, state):
        probs=np.abs(state)**2
        index=np.random.choice(len(state), p=probs/np.sum(probs))
        return format(index,'b').zfill(int(np.log2(len(state)))), probs[index]

    def density_matrix(self, state): return np.outer(state, state.conj())

    def partial_trace(self, rho, keep, dims):
        """Partial trace for subsystem reductions."""
        n = len(dims)
        keep_dims = np.prod([dims[i] for i in keep])
        trace_dims = np.prod([dims[i] for i in range(n) if i not in keep])
        rho = rho.reshape(dims + dims)
        traced = np.trace(rho, axis1=0, axis2=n)
        return traced.reshape((keep_dims, keep_dims))

    def entanglement_entropy(self, state, dims=[2,2]):
        rho = self.density_matrix(state)
        rhoA = self.partial_trace(rho, [0], dims)
        vals = np.real_if_close(np.linalg.eigvals(rhoA))
        vals = np.clip(vals, 1e-12, 1)
        return -np.sum(vals*np.log2(vals))

    def variational_ansatz(self, params):
        """Simple variational ansatz for demonstration."""
        theta, phi = params
        return np.array([math.cos(theta/2), np.exp(1j*phi)*math.sin(theta/2)], complex)

    def cost_function(self, psi, H):
        return np.real(np.vdot(psi, H @ psi))

    def variational_optimize(self, H):
        best = (None, float('inf'))
        for theta in np.linspace(0,2*PI,50):
            for phi in np.linspace(0,2*PI,50):
                psi=self.variational_ansatz((theta,phi))
                e=self.cost_function(psi,H)
                if e<best[1]: best=(psi,e)
        return best

    def decoherence(self, rho, p):
        d=len(rho)
        return (1-p)*rho + p*np.eye(d)/d

    def benchmark(self):
        start=time.time()
        tasks={'grover':lambda:self.grover(3,5),'qft':lambda:self.qft(self.create_state(3),3)}
        results={}
        with ThreadPoolExecutor(max_workers=2) as ex:
            futures={ex.submit(func):name for name,func in tasks.items()}
            for f in as_completed(futures):
                results[futures[f]]=f.result()
        return {"time":time.time()-start,"results":results}
