import math


class Physics:
    def __init__(self):
        print("⚛️ Physics module initialized — classical, relativity, and quantum toolkit.")

# -----------------------------------------------------------
# ⚙️ CONSTANTS
# -----------------------------------------------------------
    G = 6.67430e-11   # Gravitational constant (N·m²/kg²)
    g = 9.81          # Acceleration due to gravity (m/s²)
    c = 3.0e8         # Speed of light (m/s)
    R = 8.314         # Gas constant (J/mol·K)
    k = 1.380649e-23  # Boltzmann constant (J/K)
    e = 1.602176634e-19  # Elementary charge (C)
    h = 6.62607015e-34   # Planck constant (J·s)
    h_bar = h / (2 * math.pi)  # Reduced Planck’s constant (ħ)

# -----------------------------------------------------------
# 🧭 MECHANICS
# -----------------------------------------------------------
    def velocity(self, distance, time):
        return distance / time

    def acceleration(self, v_final, v_initial, time):
        return (v_final - v_initial) / time

    def force(self, mass, acceleration):
        return mass * acceleration

    def weight(self, mass):
        return mass * self.g

    def momentum(self, mass, velocity):
        return mass * velocity

    def kinetic_energy(self, mass, velocity):
        return 0.5 * mass * velocity ** 2

    def potential_energy(self, mass, height):
        return mass * self.g * height

    def power(self, work, time):
        return work / time

    def work_done(self, force, distance, angle=0):
        return force * distance * math.cos(math.radians(angle))

# -----------------------------------------------------------
# 🌌 RELATIVITY
# -----------------------------------------------------------
    def mass_energy_equivalence(self, mass):
        """E = mc²"""
        return mass * (self.c ** 2)

    def relativistic_mass(self, rest_mass, velocity):
        """m = m₀ / sqrt(1 - v²/c²)"""
        if velocity >= self.c:
            raise ValueError("Velocity cannot reach or exceed speed of light.")
        return rest_mass / math.sqrt(1 - (velocity ** 2 / self.c ** 2))

    def time_dilation(self, time_interval, velocity):
        """t' = t / sqrt(1 - v²/c²)"""
        if velocity >= self.c:
            raise ValueError("Velocity cannot reach or exceed speed of light.")
        return time_interval / math.sqrt(1 - (velocity ** 2 / self.c ** 2))

    def length_contraction(self, proper_length, velocity):
        """L = L₀ * sqrt(1 - v²/c²)"""
        if velocity >= self.c:
            raise ValueError("Velocity cannot reach or exceed speed of light.")
        return proper_length * math.sqrt(1 - (velocity ** 2 / self.c ** 2))

    def relativistic_momentum(self, mass, velocity):
        """p = γ * m * v"""
        gamma = 1 / math.sqrt(1 - (velocity ** 2 / self.c ** 2))
        return gamma * mass * velocity

    def lorentz_factor(self, velocity):
        """γ = 1 / sqrt(1 - v²/c²)"""
        if velocity >= self.c:
            raise ValueError("Velocity cannot reach or exceed speed of light.")
        return 1 / math.sqrt(1 - (velocity ** 2 / self.c ** 2))

# -----------------------------------------------------------
# ⚛️ QUANTUM PHYSICS
# -----------------------------------------------------------
    def photon_energy(self, frequency):
        """E = h * f"""
        return self.h * frequency

    def photon_energy_wavelength(self, wavelength):
        """E = h * c / λ"""
        return (self.h * self.c) / wavelength

    def de_broglie_wavelength(self, mass, velocity):
        """λ = h / (m * v)"""
        return self.h / (mass * velocity)

    def heisenberg_uncertainty(self, delta_x=None, delta_p=None):
        """
        Δx * Δp ≥ ħ / 2
        Provide one to calculate the other.
        """
        if delta_x is None and delta_p is None:
            raise ValueError("Provide at least one value (Δx or Δp).")
        if delta_x is not None:
            return self.h_bar / (2 * delta_x)
        elif delta_p is not None:
            return self.h_bar / (2 * delta_p)

    def energy_level_hydrogen(self, n):
        """Eₙ = -13.6 eV / n²"""
        return -13.6 / (n ** 2)

    def particle_energy(self, mass, velocity):
        """E = (γ - 1)mc²"""
        if velocity >= self.c:
            raise ValueError("Velocity cannot reach or exceed speed of light.")
        gamma = 1 / math.sqrt(1 - (velocity ** 2 / self.c ** 2))
        return (gamma - 1) * mass * (self.c ** 2)

# -----------------------------------------------------------
# ⚡ ELECTRICITY & MAGNETISM
# -----------------------------------------------------------
    def ohms_law(self, voltage=None, current=None, resistance=None):
        if voltage is None:
            return current * resistance
        elif current is None:
            return voltage / resistance
        elif resistance is None:
            return voltage / current
        else:
            raise ValueError("Provide only two parameters to find the third.")

    def electric_power(self, voltage, current):
        return voltage * current

    def charge(self, current, time):
        return current * time

    def coulomb_force(self, q1, q2, r):
        return (8.99e9 * q1 * q2) / (r ** 2)

# -----------------------------------------------------------
# 🌊 WAVES & LIGHT
# -----------------------------------------------------------
    def wave_speed(self, frequency, wavelength):
        return frequency * wavelength

    def frequency(self, wave_speed, wavelength):
        return wave_speed / wavelength

    def period(self, frequency):
        return 1 / frequency

# -----------------------------------------------------------
# 🌍 GRAVITATION
# -----------------------------------------------------------
    def gravitational_force(self, m1, m2, r):
        return self.G * m1 * m2 / (r ** 2)

# -----------------------------------------------------------
# 🔄 CONVERSIONS
# -----------------------------------------------------------
    def joule_to_electronvolt(self, joules):
        return joules / self.e

    def electronvolt_to_joule(self, ev):
        return ev * self.e

    def joule_to_calorie(self, joules):
        return joules / 4.184

    def calorie_to_joule(self, calories):
        return calories * 4.184