import os
import platform
import subprocess
import shutil
from pathlib import Path
import psutil
import socket
import sys

class OS:
    def __init__(self):
        self.name = platform.system().lower()

    # Detect operating system
    def get_os(self):
        return self.name

    # Shutdown device
    def shutdown(self):
        print(f"Detected OS: {self.name}")
        try:
            if "windows" in self.name:
                os.system("shutdown /s /t 1")
            elif "linux" in self.name or "darwin" in self.name:
                os.system("sudo shutdown now")
            else:
                print("Unsupported OS for shutdown.")
        except Exception as e:
            print(f"Error: {e}")

    # Create folder
    def create_folder(self, path):
        try:
            os.makedirs(path, exist_ok=True)
            print(f"Folder created: {path}")
        except Exception as e:
            print(f"Error creating folder: {e}")

    # Create file
    def create_file(self, path, content=""):
        try:
            with open(path, "w", encoding="utf-8") as f:
                f.write(content)
            print(f"File created: {path}")
        except Exception as e:
            print(f"Error creating file: {e}")

    # Delete folder
    def delete_folder(self, path):
        try:
            if os.path.exists(path):
                shutil.rmtree(path)
                print(f"Folder deleted: {path}")
            else:
                print("Folder does not exist.")
        except PermissionError:
            print("Admin privileges required to delete this folder.")
        except Exception as e:
            print(f"Error deleting folder: {e}")

    # Delete file
    def delete_file(self, path):
        try:
            if os.path.exists(path):
                os.remove(path)
                print(f"File deleted: {path}")
            else:
                print("File does not exist.")
        except PermissionError:
            print("Admin privileges required to delete this file.")
        except Exception as e:
            print(f"Error deleting file: {e}")

    # Check if path exists
    def path_exists(self, path):
        return Path(path).exists()

    # Run system command (cross-platform)
    def run_command(self, command):
        try:
            os.system(command)
        except Exception as e:
            print(f"Error executing command: {e}")

    # Open file or application
    def open_file(self, path):
        try:
            if not os.path.exists(path):
                print("Path does not exist.")
                return
            if "windows" in self.name:
                os.startfile(path)
            elif "darwin" in self.name:
                subprocess.run(["open", path])
            elif "linux" in self.name:
                subprocess.run(["xdg-open", path])
            else:
                print("Unsupported OS for file opening.")
        except Exception as e:
            print(f"Error opening file: {e}")

    # Fetch manufacturer (cross-platform)
    def get_manufacturer(self):
        try:
            if "windows" in self.name:
                # Try PowerShell first
                cmd = ['powershell', '-Command', "(Get-CimInstance Win32_ComputerSystem).Manufacturer"]
                result = subprocess.run(cmd, capture_output=True, text=True)
                if result.stdout.strip():
                    return result.stdout.strip()
                # Fallback to WMI if available
                try:
                    import wmi
                    c = wmi.WMI()
                    for system in c.Win32_ComputerSystem():
                        return system.Manufacturer
                except:
                    return None
            elif "linux" in self.name:
                if os.path.exists("/sys/class/dmi/id/sys_vendor"):
                    with open("/sys/class/dmi/id/sys_vendor") as f:
                        return f.read().strip()
            elif "darwin" in self.name:
                cmd = ["system_profiler", "SPHardwareDataType"]
                result = subprocess.run(cmd, capture_output=True, text=True)
                for line in result.stdout.splitlines():
                    if "Manufacturer" in line:
                        return line.split(":")[1].strip()
            return None
        except Exception:
            return None

    # Device information
    def device_info(self):
        try:
            info = {
                "os": platform.system(),
                "os_version": platform.version(),
                "os_release": platform.release(),
                "architecture": platform.machine(),
                "processor": platform.processor(),
                "cpu_usage_percent": psutil.cpu_percent(interval=1),
                "cpu_cores": psutil.cpu_count(logical=True),
                "memory_total_gb": round(psutil.virtual_memory().total / (1024**3), 2),
                "memory_used_gb": round(psutil.virtual_memory().used / (1024**3), 2),
                "memory_percent": psutil.virtual_memory().percent,
                "disk_total_gb": round(psutil.disk_usage('/').total / (1024**3), 2),
                "disk_used_gb": round(psutil.disk_usage('/').used / (1024**3), 2),
                "disk_percent": psutil.disk_usage('/').percent,
                "hostname": socket.gethostname(),
                "ip_address": socket.gethostbyname(socket.gethostname()),
                "battery_percent": psutil.sensors_battery().percent if psutil.sensors_battery() else None,
                "is_plugged": psutil.sensors_battery().power_plugged if psutil.sensors_battery() else None,
                "user": os.getenv("USERNAME") or os.getenv("USER"),
                "manufacturer": self.get_manufacturer()
            }
            return info
        except Exception as e:
            print(f"Error retrieving device info: {e}")
            return {}
