class Number_System:
    def __init__(self):
        print("🔢 Number System module initialized — supports Binary, Octal, Decimal, and Hexadecimal conversions.")

    # ---------------------------------------------------------
    # 🧮 BASIC CONVERSIONS
    # ---------------------------------------------------------
    def dec_to_bin(self, num: int) -> str:
        """Converts Decimal → Binary."""
        return bin(num)[2:]

    def dec_to_oct(self, num: int) -> str:
        """Converts Decimal → Octal."""
        return oct(num)[2:]

    def dec_to_hex(self, num: int) -> str:
        """Converts Decimal → Hexadecimal."""
        return hex(num)[2:].upper()

    def bin_to_dec(self, num: str) -> int:
        """Converts Binary → Decimal."""
        return int(num, 2)

    def oct_to_dec(self, num: str) -> int:
        """Converts Octal → Decimal."""
        return int(num, 8)

    def hex_to_dec(self, num: str) -> int:
        """Converts Hexadecimal → Decimal."""
        return int(num, 16)

    # ---------------------------------------------------------
    # 🔄 CROSS CONVERSIONS (BINARY, OCTAL, HEXADECIMAL)
    # ---------------------------------------------------------
    def bin_to_oct(self, num: str) -> str:
        """Converts Binary → Octal."""
        return oct(int(num, 2))[2:]

    def bin_to_hex(self, num: str) -> str:
        """Converts Binary → Hexadecimal."""
        return hex(int(num, 2))[2:].upper()

    def oct_to_bin(self, num: str) -> str:
        """Converts Octal → Binary."""
        return bin(int(num, 8))[2:]

    def oct_to_hex(self, num: str) -> str:
        """Converts Octal → Hexadecimal."""
        return hex(int(num, 8))[2:].upper()

    def hex_to_bin(self, num: str) -> str:
        """Converts Hexadecimal → Binary."""
        return bin(int(num, 16))[2:]

    def hex_to_oct(self, num: str) -> str:
        """Converts Hexadecimal → Octal."""
        return oct(int(num, 16))[2:]

    # ---------------------------------------------------------
    # 🧰 UNIVERSAL CONVERTER
    # ---------------------------------------------------------
    def convert(self, value: str, from_base: int, to_base: int) -> str:
        """
        Universal number converter.
        Supports bases between 2 and 36.
        Example: convert("1010", 2, 16) → 'A'
        """
        decimal_value = int(value, from_base)
        digits = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        if to_base == 10:
            return str(decimal_value)
        result = ""
        while decimal_value > 0:
            result = digits[decimal_value % to_base] + result
            decimal_value //= to_base
        return result or "0"

    # ---------------------------------------------------------
    # 🧪 VALIDATION
    # ---------------------------------------------------------
    def validate(self, value: str, base: int) -> bool:
        """Validates if a number string belongs to a base system."""
        valid_chars = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ"[:base]
        value = value.upper()
        return all(ch in valid_chars for ch in value)

    # ---------------------------------------------------------
    # 📦 BATCH CONVERSION
    # ---------------------------------------------------------
    def convert_list(self, values: list, from_base: int, to_base: int) -> list:
        """Converts a list of numbers between any two bases."""
        return [self.convert(v, from_base, to_base) for v in values]

    # ---------------------------------------------------------
    # 🧾 FORMAT DISPLAY
    # ---------------------------------------------------------
    def format_all(self, num: int) -> dict:
        """Returns a dictionary of all equivalent representations."""
        return {
            "Decimal": num,
            "Binary": self.dec_to_bin(num),
            "Octal": self.dec_to_oct(num),
            "Hexadecimal": self.dec_to_hex(num)
        }

    # ---------------------------------------------------------
    # 🧠 AUTO DETECT BASE
    # ---------------------------------------------------------
    def detect_base(self, value: str) -> str:
        """
        Detects number system based on prefix or pattern.
        """
        value = value.strip().lower()
        if value.startswith("0b"):
            return "Binary"
        elif value.startswith("0o"):
            return "Octal"
        elif value.startswith("0x"):
            return "Hexadecimal"
        elif all(c in "01" for c in value):
            return "Binary (No Prefix)"
        else:
            return "Decimal or Custom Base"
