import torch
import tensorflow as tf


class NeuralNetwork:
    def __init__(self, backend='torch', device=None):
        self.backend = backend.lower()
        self.device = device or ("cuda" if torch.cuda.is_available() else "cpu")
        self.model = None

        if self.backend not in ['torch', 'tf']:
            raise ValueError("Backend must be 'torch' or 'tf'")

        # dynamically attach all functions and submodules
        self._attach_backend()

    # -------------------------------------------------------------------------
    # Attach backend attributes dynamically
    # -------------------------------------------------------------------------
    def _attach_backend(self):
        """Attach TensorFlow or Torch modules as attributes for universal access."""
        if self.backend == 'torch':
            self.tensor_lib = torch
            self.nn = torch.nn
            self.optim = torch.optim
            self.utils = torch.utils
            self.vision = None
        else:
            self.tensor_lib = tf
            self.nn = tf.keras.layers
            self.optim = tf.keras.optimizers
            self.losses = tf.keras.losses
            self.metrics = tf.keras.metrics

    # -------------------------------------------------------------------------
    # Universal tensor creator
    # -------------------------------------------------------------------------
    def tensor(self, data, dtype=None, requires_grad=False):
        if self.backend == 'torch':
            return torch.tensor(data, dtype=dtype or torch.float32, requires_grad=requires_grad).to(self.device)
        elif self.backend == 'tf':
            return tf.convert_to_tensor(data, dtype=dtype or tf.float32)

    # -------------------------------------------------------------------------
    # Universal model creation
    # -------------------------------------------------------------------------
    def Sequential(self, layers_list):
        """Create a simple sequential model compatible with backend."""
        if self.backend == 'torch':
            seq_layers = []
            for i in range(len(layers_list) - 1):
                seq_layers.append(torch.nn.Linear(layers_list[i], layers_list[i + 1]))
                if i < len(layers_list) - 2:
                    seq_layers.append(torch.nn.ReLU())
            self.model = torch.nn.Sequential(*seq_layers).to(self.device)
        else:
            self.model = tf.keras.Sequential()
            for i in range(len(layers_list) - 1):
                self.model.add(tf.keras.layers.Dense(
                    layers_list[i + 1],
                    input_dim=layers_list[i] if i == 0 else None,
                    activation='relu' if i < len(layers_list) - 2 else None
                ))

    # -------------------------------------------------------------------------
    # Universal compile/train/predict
    # -------------------------------------------------------------------------
    def compile(self, optimizer='adam', loss='mse'):
        if self.backend == 'torch':
            self.loss_fn = torch.nn.MSELoss() if loss == 'mse' else loss
            self.optimizer = torch.optim.Adam(self.model.parameters()) if optimizer == 'adam' else optimizer
        else:
            self.model.compile(optimizer=getattr(tf.keras.optimizers, optimizer.capitalize())(),
                               loss=loss)

    def fit(self, X, y, epochs=5):
        if self.backend == 'torch':
            X = self.tensor(X)
            y = self.tensor(y)
            for epoch in range(epochs):
                self.optimizer.zero_grad()
                preds = self.model(X)
                loss = self.loss_fn(preds, y)
                loss.backward()
                self.optimizer.step()
                print(f"[Torch] Epoch {epoch+1}/{epochs}, Loss: {loss.item():.4f}")
        else:
            self.model.fit(X, y, epochs=epochs, verbose=1)

    def predict(self, X):
        if self.backend == 'torch':
            with torch.no_grad():
                preds = self.model(self.tensor(X))
            return preds.cpu().numpy()
        else:
            return self.model.predict(X)

    # -------------------------------------------------------------------------
    # Dynamic universal access
    # -------------------------------------------------------------------------
    def __getattr__(self, name):
        """Dynamic universal access to backend functions and submodules."""
        try:
            if hasattr(self.tensor_lib, name):
                return getattr(self.tensor_lib, name)
            elif self.backend == 'torch' and hasattr(torch.nn, name):
                return getattr(torch.nn, name)
            elif self.backend == 'tf':
                # check keras modules
                if hasattr(tf.keras.layers, name):
                    return getattr(tf.keras.layers, name)
                if hasattr(tf.keras.optimizers, name):
                    return getattr(tf.keras.optimizers, name)
                if hasattr(tf.keras.losses, name):
                    return getattr(tf.keras.losses, name)
        except Exception:
            pass
        raise AttributeError(f"'{self.backend}' backend has no attribute '{name}'")


