import numpy as np

class Matrices:
    def __init__(self, data: list):
        self.data = data
        self.rows = len(data)
        self.cols = len(data[0]) if data else 0

    def shape(self):
        return (self.rows, self.cols)

    def size(self):
        return self.rows * self.cols

    def copy(self):
        return Matrices([row[:] for row in self.data])

    def add(self, other):
        # Element-wise addition
        return Matrices([
            [self.data[i][j] + other.data[i][j] for j in range(self.cols)]
            for i in range(self.rows)
        ])

    def subtract(self, other):
        return Matrices([
            [self.data[i][j] - other.data[i][j] for j in range(self.cols)]
            for i in range(self.rows)
        ])

    def multiply(self, other):
        # Matrix multiplication (not element-wise)
        result = [
            [sum(self.data[i][k] * other.data[k][j] for k in range(self.cols))
             for j in range(other.cols)]
            for i in range(self.rows)
        ]
        return Matrices(result)

    def scalar_multiply(self, value):
        return Matrices([
            [self.data[i][j] * value for j in range(self.cols)]
            for i in range(self.rows)
        ])

    def transpose(self):
        return Matrices([
            [self.data[j][i] for j in range(self.rows)]
            for i in range(self.cols)
        ])

    def determinant(self):
        if self.rows != self.cols:
            raise ValueError("Determinant only defined for square matrices")
        if self.rows == 1:
            return self.data[0][0]
        if self.rows == 2:
            return self.data[0][0]*self.data[1][1] - self.data[0][1]*self.data[1][0]
        det = 0
        for c in range(self.cols):
            minor = [row[:c] + row[c+1:] for row in self.data[1:]]
            det += ((-1)**c) * self.data[0][c] * Matrices(minor).determinant()
        return det

    def inverse(self):
        det = self.determinant()
        if det == 0:
            raise ValueError("Matrix is singular and cannot be inverted")
        if self.rows == 2:
            a, b = self.data[0]
            c, d = self.data[1]
            inv = [[d, -b], [-c, a]]
            return Matrices(inv).scalar_multiply(1/det)
        cofactors = []
        for r in range(self.rows):
            cofactor_row = []
            for c in range(self.cols):
                minor = [row[:c] + row[c+1:] for i, row in enumerate(self.data) if i != r]
                cofactor_row.append(((-1)**(r+c)) * Matrices(minor).determinant())
            cofactors.append(cofactor_row)
        cofactors = Matrices(cofactors).transpose()
        return cofactors.scalar_multiply(1/det)

    def is_square(self):
        return self.rows == self.cols

    def flatten(self):
        return [x for row in self.data for x in row]

    def trace(self):
        return sum(self.data[i][i] for i in range(min(self.rows, self.cols)))

    def rank(self):
        import numpy as np
        return np.linalg.matrix_rank(self.data)

    def power(self, n):
        if not self.is_square():
            raise ValueError("Matrix must be square for power operation")
        result = Matrices.identity(self.rows)
        for _ in range(n):
            result = result.multiply(self)
        return result

    @staticmethod
    def identity(n):
        return Matrices([[1 if i == j else 0 for j in range(n)] for i in range(n)])

    def __str__(self):
        return '\n'.join(['\t'.join(map(str, row)) for row in self.data])