import customtkinter as ctk
import tkinter as tk
from tkinter import ttk, messagebox, simpledialog, filedialog
import threading, time

# --------------------------------------------------------------------------
# Initialization
# --------------------------------------------------------------------------
ctk.set_appearance_mode("System")
ctk.set_default_color_theme("blue")

# --------------------------------------------------------------------------
# COMPONENT WRAPPER
# --------------------------------------------------------------------------
class Component:
    """Wraps a widget and forwards .config() calls."""
    def __init__(self, widget):
        self.widget = widget

    def config(self, **kwargs):
        """Safely configure underlying widget."""
        for method in ("configure", "config"):
            if hasattr(self.widget, method):
                try:
                    getattr(self.widget, method)(**kwargs)
                    break
                except Exception:
                    continue


# --------------------------------------------------------------------------
# CONTAINER CLASS
# --------------------------------------------------------------------------
class Container:
    """A flexible container frame that supports both grid and pack layouts."""
    def __init__(self, master, **style):
        bg = style.get("bg", None)
        self.frame = ctk.CTkFrame(master, fg_color=bg)
        self.children = []

    def add(self, component, row=None, column=None, pack=False, padx=6, pady=6, sticky="nsew", **kwargs):
        """Add a component to this container."""
        w = component.widget if isinstance(component, Component) else component
        if pack:
            w.pack(padx=padx, pady=pady, **kwargs)
        else:
            row = len(self.children) if row is None else row
            column = 0 if column is None else column
            w.grid(row=row, column=column, padx=padx, pady=pady, sticky=sticky, **kwargs)
            self.frame.grid_rowconfigure(row, weight=1)
            self.frame.grid_columnconfigure(column, weight=1)
        self.children.append(w)
        return w

    def pack(self, **kwargs): self.frame.pack(**kwargs)
    def grid(self, **kwargs): self.frame.grid(**kwargs)
    def place(self, **kwargs): self.frame.place(**kwargs)


# --------------------------------------------------------------------------
# FLOOD GAUGE
# --------------------------------------------------------------------------
class FloodGauge(Component):
    """Custom vertical flood-style progress gauge."""
    def __init__(self, master, width=60, height=200, max_value=100, fg="#1E90FF", bg="#f5f5f5"):
        canvas = tk.Canvas(master, width=width, height=height, bg=bg, highlightthickness=1)
        super().__init__(canvas)
        self.width, self.height, self.max_value, self.value, self.fg, self.bg = width, height, max_value, 0, fg, bg
        self._draw()

    def _draw(self):
        c = self.widget
        c.delete("all")
        fill_height = (self.value / self.max_value) * (self.height - 10)
        y1 = self.height - 5 - fill_height
        c.create_rectangle(5, y1, self.width - 5, self.height - 5, fill=self.fg, outline="")
        c.create_rectangle(5, 5, self.width - 5, self.height - 5, outline="#999", width=1)
        c.create_text(self.width / 2, 12, text=f"{int(self.value / self.max_value * 100)}%", font=("Arial", 9, "bold"))

    def set(self, value):
        """Set the gauge value."""
        self.value = max(0, min(self.max_value, value))
        self._draw()

    def increment(self, step=1):
        """Increment gauge by step."""
        self.set(self.value + step)


# --------------------------------------------------------------------------
# MAIN GUI CLASS
# --------------------------------------------------------------------------
class GUI:
    """Simplified, powerful GUI builder for NeuraPy."""
    def __init__(self, title="NeuraPy GUI", size=(1000, 700), appearance="System", theme="blue"):
        ctk.set_appearance_mode(appearance)
        ctk.set_default_color_theme(theme)
        self.root = ctk.CTk()
        self.root.geometry(f"{size[0]}x{size[1]}")
        self.root.title(title)
        self.global_style = {}

        # Layout frames
        self.main = ctk.CTkFrame(self.root)
        self.main.pack(fill="both", expand=True)
        self.left = ctk.CTkFrame(self.main, width=200)
        self.center = ctk.CTkFrame(self.main)
        self.right = ctk.CTkFrame(self.main, width=240)
        self.left.pack(side="left", fill="y")
        self.center.pack(side="left", fill="both", expand=True)
        self.right.pack(side="right", fill="y")

        self.containers = {
            "left": Container(self.left),
            "center": Container(self.center),
            "right": Container(self.right),
            "root": Container(self.root)
        }

    # ----------------------------------------------------------------------
    # Basic Widgets
    # ----------------------------------------------------------------------
    def button(self, text="Button", command=None, **style):
        return Component(ctk.CTkButton(self.center, text=text, command=command, **{**self.global_style, **style}))

    def label(self, text="Label", **style):
        return Component(ctk.CTkLabel(self.center, text=text, **{**self.global_style, **style}))

    def textbox(self, placeholder="", multiline=False, **style):
        widget = ctk.CTkTextbox(self.center, **style) if multiline else \
                 ctk.CTkEntry(self.center, placeholder_text=placeholder, **style)
        return Component(widget)

    def slider(self, from_=0, to=100, value=0, orientation="horizontal", command=None, **style):
        if orientation == "horizontal":
            s = ctk.CTkSlider(self.center, from_=from_, to=to, **style)
            s.set(value)
            if command:
                s.configure(command=command)
            return Component(s)
        frame = ctk.CTkFrame(self.center)
        s = ttk.Scale(frame, from_=from_, to=to, orient="vertical")
        s.set(value)
        s.pack(fill="y", expand=True)
        return Component(s)

    def combo(self, values, default=None, **style):
        cb = ctk.CTkComboBox(self.center, values=values, **style)
        if default:
            cb.set(default)
        return Component(cb)

    def checkbox(self, text="Check", default=False, command=None, **style):
        var = tk.BooleanVar(value=default)
        chk = ctk.CTkCheckBox(self.center, text=text, variable=var, command=command, **style)
        chk.var = var
        return Component(chk)

    def radio(self, options, default=None, command=None, horizontal=False):
        var = tk.StringVar(value=default or options[0])
        frame = ctk.CTkFrame(self.center)
        for opt in options:
            rb = ctk.CTkRadioButton(frame, text=opt, variable=var, value=opt, command=command)
            rb.pack(side="left" if horizontal else "top", padx=4, pady=2, anchor="w")
        frame.var = var
        return Component(frame)

    def table(self, columns, height=8, **style):
        frame = ctk.CTkFrame(self.center, **style)
        tree = ttk.Treeview(frame, columns=columns, show="headings", height=height)
        vsb = ttk.Scrollbar(frame, orient="vertical", command=tree.yview)
        tree.configure(yscrollcommand=vsb.set)
        for c in columns:
            tree.heading(c, text=c)
            tree.column(c, anchor="center")
        tree.pack(side="left", fill="both", expand=True)
        vsb.pack(side="right", fill="y")
        frame.tree = tree
        return Component(frame)

    def flood_gauge(self, max_value=100, **style):
        frame = ctk.CTkFrame(self.center)
        gauge = FloodGauge(frame, max_value=max_value, **style)
        gauge.widget.pack()
        return gauge

    def loader(self, text="Loading...", duration=3):
        frame = ctk.CTkFrame(self.center)
        lbl = ctk.CTkLabel(frame, text=text)
        pb = ttk.Progressbar(frame, mode="indeterminate")
        lbl.pack(pady=4)
        pb.pack(pady=4)

        def spin():
            pb.start(10)
            time.sleep(duration)
            pb.stop()
        threading.Thread(target=spin, daemon=True).start()
        return Component(frame)

    # ----------------------------------------------------------------------
    # Dialogs and Message Boxes
    # ----------------------------------------------------------------------
    def alert(self, title="Alert", message="Something happened!"):
        messagebox.showinfo(title, message)

    def error(self, title="Error", message="An error occurred!"):
        messagebox.showerror(title, message)

    def ask_yes_no(self, title="Confirm", message="Are you sure?"):
        return messagebox.askyesno(title, message)

    def ask_input(self, title="Input", prompt="Enter value:"):
        return simpledialog.askstring(title, prompt)

    # ----------------------------------------------------------------------
    # File Dialogs
    # ----------------------------------------------------------------------
    def open_file(self, filetypes=(("All files", "*.*"),)):
        return filedialog.askopenfilename(filetypes=filetypes)

    def save_file(self, filetypes=(("All files", "*.*"),)):
        return filedialog.asksaveasfilename(filetypes=filetypes)

    def select_folder(self):
        return filedialog.askdirectory()

    # ----------------------------------------------------------------------
    # Utility Functions
    # ----------------------------------------------------------------------
    def toast(self, message, duration=2):
        """Temporary popup message."""
        popup = ctk.CTkToplevel(self.root)
        popup.overrideredirect(True)
        popup.geometry("+500+500")
        lbl = ctk.CTkLabel(popup, text=message, fg_color="#333", text_color="white", corner_radius=8)
        lbl.pack(padx=10, pady=10)
        self.root.after(duration * 1000, popup.destroy)

    def tooltip(self, widget, text):
        """Show tooltip when hovering over a widget."""
        tip = tk.Toplevel(widget)
        tip.withdraw()
        tip.overrideredirect(True)
        label = tk.Label(tip, text=text, bg="#333", fg="white", relief="solid", borderwidth=1)
        label.pack(ipadx=4)

        def enter(e):
            tip.geometry(f"+{e.x_root+10}+{e.y_root+10}")
            tip.deiconify()

        def leave(e):
            tip.withdraw()

        widget.bind("<Enter>", enter)
        widget.bind("<Leave>", leave)

    def set_style(self, **style):
        """Set global styling for all widgets."""
        self.global_style.update(style)

    # ----------------------------------------------------------------------
    # Window Controls
    # ----------------------------------------------------------------------
    def set_title(self, title):
        self.root.title(title)

    def resize(self, width, height):
        self.root.geometry(f"{width}x{height}")

    def fullscreen(self, enable=True):
        self.root.attributes("-fullscreen", enable)

    def close(self):
        self.root.destroy()

    # ----------------------------------------------------------------------
    # Event Binding System
    # ----------------------------------------------------------------------
    def bind_key(self, key, func):
        """Bind a keyboard key or combination globally to the window."""
        self.root.bind(key, func)

    def bind_mouse(self, event, func):
        """Bind a mouse event globally (e.g., '<Button-1>', '<Motion>')."""
        self.root.bind(event, func)

    def bind_to_widget(self, widget, event, func):
        """Bind an event to a specific widget."""
        target = widget.widget if isinstance(widget, Component) else widget
        target.bind(event, func)

    # ----------------------------------------------------------------------
    # Access & Run
    # ----------------------------------------------------------------------
    def container(self, name):
        return self.containers.get(name)

    def show(self):
        self.root.mainloop()