import os
from ftplib import FTP as FTPClient

class FTP:
    def __init__(self, host="", user="", password="", port=21, timeout=30):
        self.host = host
        self.user = user
        self.password = password
        self.port = port
        self.timeout = timeout
        self.ftp = None

    # -----------------------------------------------------------
    # 🔌 CONNECTION HANDLING
    # -----------------------------------------------------------
    def connect(self):
        """Connect to the FTP server."""
        try:
            self.ftp = FTPClient()
            self.ftp.connect(self.host, self.port, timeout=self.timeout)
            self.ftp.login(self.user, self.password)
            print(f"✅ Connected to {self.host}")
        except Exception as e:
            print(f"❌ Connection failed: {e}")

    def disconnect(self):
        """Close FTP connection."""
        if self.ftp:
            self.ftp.quit()
            print("🔌 Disconnected from FTP server.")

    # -----------------------------------------------------------
    # 📁 DIRECTORY MANAGEMENT
    # -----------------------------------------------------------
    def current_dir(self):
        """Return current working directory."""
        if self.ftp:
            return self.ftp.pwd()

    def change_dir(self, path):
        """Change working directory."""
        if self.ftp:
            self.ftp.cwd(path)
            print(f"📂 Changed directory to: {path}")

    def make_dir(self, dirname):
        """Create a new directory."""
        if self.ftp:
            self.ftp.mkd(dirname)
            print(f"📁 Directory created: {dirname}")

    def remove_dir(self, dirname):
        """Remove a directory."""
        if self.ftp:
            self.ftp.rmd(dirname)
            print(f"🗑️ Directory removed: {dirname}")

    # -----------------------------------------------------------
    # 📜 FILE LISTING & INFO
    # -----------------------------------------------------------
    def list_files(self):
        """List files in current directory."""
        if self.ftp:
            files = self.ftp.nlst()
            for f in files:
                print(f"📄 {f}")
            return files

    def file_info(self, filename):
        """Get file size and modification time."""
        if self.ftp:
            size = self.ftp.size(filename)
            print(f"📏 Size of {filename}: {size} bytes")
            return size

    # -----------------------------------------------------------
    # ⬆️ UPLOAD & ⬇️ DOWNLOAD
    # -----------------------------------------------------------
    def upload_file(self, local_path, remote_path=None):
        """Upload a local file to the FTP server."""
        if not remote_path:
            remote_path = os.path.basename(local_path)
        if self.ftp and os.path.exists(local_path):
            with open(local_path, "rb") as f:
                self.ftp.storbinary(f"STOR {remote_path}", f)
            print(f"⬆️ Uploaded: {local_path} → {remote_path}")
        else:
            print("⚠️ Local file not found or not connected.")

    def download_file(self, remote_path, local_path=None):
        """Download a file from the FTP server."""
        if not local_path:
            local_path = os.path.basename(remote_path)
        if self.ftp:
            with open(local_path, "wb") as f:
                self.ftp.retrbinary(f"RETR {remote_path}", f.write)
            print(f"⬇️ Downloaded: {remote_path} → {local_path}")

    # -----------------------------------------------------------
    # ⚙️ FILE OPERATIONS
    # -----------------------------------------------------------
    def rename_file(self, old_name, new_name):
        """Rename file on the FTP server."""
        if self.ftp:
            self.ftp.rename(old_name, new_name)
            print(f"✏️ Renamed: {old_name} → {new_name}")

    def delete_file(self, filename):
        """Delete a file from the FTP server."""
        if self.ftp:
            self.ftp.delete(filename)
            print(f"🗑️ Deleted: {filename}")

    # -----------------------------------------------------------
    # 🧩 UTILITY
    # -----------------------------------------------------------
    def is_connected(self):
        """Check if the FTP connection is active."""
        try:
            self.ftp.voidcmd("NOOP")
            return True
        except:
            return False