#____________________________________Encryption__________________________________________________
# encryption_suite.py
# Requires Python 3.8+. Optional packages: pycryptodome, bcrypt, argon2-cffi

import base64
import hashlib
import hmac
import secrets
from typing import Tuple

# Optional imports
try:
    from Crypto.Cipher import AES, PKCS1_OAEP
    from Crypto.PublicKey import RSA
    from Crypto.Util.Padding import pad, unpad
    HAS_PYCRYPTODOME = True
except Exception:
    HAS_PYCRYPTODOME = False

try:
    import bcrypt
    HAS_BCRYPT = True
except Exception:
    HAS_BCRYPT = False

try:
    from argon2 import PasswordHasher
    HAS_ARGON2 = True
except Exception:
    HAS_ARGON2 = False


class Encryption_Decryption:
    """
    Multi-tier encryption & hashing utilities.
    Methods are named for clarity: trivial -> intermediate -> strong.
    All encrypt/decrypt methods accept bytes or str (converted to utf-8).
    """

    def __init__(self):
        pass

    # --------------------------
    # Utilities
    # --------------------------
    @staticmethod
    def _to_bytes(data):
        return data if isinstance(data, (bytes, bytearray)) else str(data).encode('utf-8')

    @staticmethod
    def secure_random_bytes(n: int) -> bytes:
        return secrets.token_bytes(n)

    # --------------------------
    # Trivial / Educational (easy to crack)
    # --------------------------
    def caesar_encrypt(self, plaintext: str, shift: int = 3) -> str:
        b = self._to_bytes(plaintext)
        return bytes(( (c + shift) % 256 ) for c in b).hex()

    def caesar_decrypt(self, hex_cipher: str, shift: int = 3) -> str:
        data = bytes.fromhex(hex_cipher)
        return bytes(( (c - shift) % 256 ) for c in data).decode('utf-8', errors='ignore')

    def xor_encrypt(self, plaintext: str, key: bytes) -> str:
        b = self._to_bytes(plaintext)
        k = key if isinstance(key, (bytes,bytearray)) else self._to_bytes(key)
        out = bytes([b[i] ^ k[i % len(k)] for i in range(len(b))])
        return base64.b64encode(out).decode('ascii')

    def xor_decrypt(self, b64_cipher: str, key: bytes) -> str:
        data = base64.b64decode(b64_cipher)
        k = key if isinstance(key, (bytes,bytearray)) else self._to_bytes(key)
        out = bytes([data[i] ^ k[i % len(k)] for i in range(len(data))])
        return out.decode('utf-8', errors='ignore')

    def rot13(self, text: str) -> str:
        return text.translate(str.maketrans(
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz",
            "NOPQRSTUVWXYZABCDEFGHIJKLMnopqrstuvwxyzabcdefghijklm"))

    # --------------------------
    # Moderate (symmetric) - requires pycryptodome
    # --------------------------
    def aes_cbc_encrypt(self, plaintext: str, key: bytes) -> Tuple[str, str]:
        """
        AES-256-CBC with PKCS7 padding.
        Returns (iv_hex, cipher_hex).
        """
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for AES functions. pip install pycryptodome")
        key = hashlib.sha256(self._to_bytes(key)).digest()  # derive 32 bytes key
        iv = self.secure_random_bytes(16)
        cipher = AES.new(key, AES.MODE_CBC, iv)
        ct = cipher.encrypt(pad(self._to_bytes(plaintext), AES.block_size))
        return iv.hex(), ct.hex()

    def aes_cbc_decrypt(self, iv_hex: str, cipher_hex: str, key: bytes) -> str:
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for AES functions. pip install pycryptodome")
        key = hashlib.sha256(self._to_bytes(key)).digest()
        iv = bytes.fromhex(iv_hex)
        ct = bytes.fromhex(cipher_hex)
        cipher = AES.new(key, AES.MODE_CBC, iv)
        pt = unpad(cipher.decrypt(ct), AES.block_size)
        return pt.decode('utf-8', errors='ignore')

    def aes_gcm_encrypt(self, plaintext: str, key: bytes) -> Tuple[str, str, str]:
        """
        AES-256-GCM. Returns (nonce_hex, cipher_hex, tag_hex).
        """
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for AES functions. pip install pycryptodome")
        key = hashlib.sha256(self._to_bytes(key)).digest()
        nonce = self.secure_random_bytes(12)
        cipher = AES.new(key, AES.MODE_GCM, nonce=nonce)
        ct, tag = cipher.encrypt_and_digest(self._to_bytes(plaintext))
        return nonce.hex(), ct.hex(), tag.hex()

    def aes_gcm_decrypt(self, nonce_hex: str, cipher_hex: str, tag_hex: str, key: bytes) -> str:
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for AES functions. pip install pycryptodome")
        key = hashlib.sha256(self._to_bytes(key)).digest()
        nonce = bytes.fromhex(nonce_hex)
        ct = bytes.fromhex(cipher_hex)
        tag = bytes.fromhex(tag_hex)
        cipher = AES.new(key, AES.MODE_GCM, nonce=nonce)
        pt = cipher.decrypt_and_verify(ct, tag)
        return pt.decode('utf-8', errors='ignore')

    # --------------------------
    # Asymmetric (RSA) - requires pycryptodome
    # --------------------------
    def rsa_generate_keys(self, bits: int = 2048) -> Tuple[str, str]:
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for RSA. pip install pycryptodome")
        key = RSA.generate(bits)
        private = key.export_key().decode('utf-8')
        public = key.publickey().export_key().decode('utf-8')
        return private, public

    def rsa_encrypt(self, plaintext: str, public_key_pem: str) -> str:
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for RSA. pip install pycryptodome")
        pub = RSA.import_key(public_key_pem)
        cipher = PKCS1_OAEP.new(pub)
        ct = cipher.encrypt(self._to_bytes(plaintext))
        return base64.b64encode(ct).decode('ascii')

    def rsa_decrypt(self, b64_cipher: str, private_key_pem: str) -> str:
        if not HAS_PYCRYPTODOME:
            raise RuntimeError("pycryptodome required for RSA. pip install pycryptodome")
        priv = RSA.import_key(private_key_pem)
        cipher = PKCS1_OAEP.new(priv)
        pt = cipher.decrypt(base64.b64decode(b64_cipher))
        return pt.decode('utf-8', errors='ignore')

    # --------------------------
    # Hashing (one-way) - fast -> slow adaptive
    # --------------------------
    def hash_md5(self, data: str) -> str:
        return hashlib.md5(self._to_bytes(data)).hexdigest()

    def hash_sha1(self, data: str) -> str:
        return hashlib.sha1(self._to_bytes(data)).hexdigest()

    def hash_sha256(self, data: str) -> str:
        return hashlib.sha256(self._to_bytes(data)).hexdigest()

    def hash_sha3_512(self, data: str) -> str:
        return hashlib.sha3_512(self._to_bytes(data)).hexdigest()

    def hmac_sha256(self, key: str, message: str) -> str:
        return hmac.new(self._to_bytes(key), self._to_bytes(message), hashlib.sha256).hexdigest()

    # PBKDF2
    def pbkdf2_hash(self, password: str, salt: bytes = None, iterations: int = 200_000) -> Tuple[str, str, int]:
        salt = salt or self.secure_random_bytes(16)
        dk = hashlib.pbkdf2_hmac('sha256', self._to_bytes(password), salt, iterations)
        return dk.hex(), salt.hex(), iterations

    # scrypt (built-in)
    def scrypt_hash(self, password: str, salt: bytes = None, n: int = 2**14, r: int = 8, p: int = 1) -> Tuple[str, str]:
        salt = salt or self.secure_random_bytes(16)
        dk = hashlib.scrypt(self._to_bytes(password), salt=salt, n=n, r=r, p=p, dklen=64)
        return dk.hex(), salt.hex()

    # bcrypt (external)
    def bcrypt_hash(self, password: str, rounds: int = 12) -> str:
        if not HAS_BCRYPT:
            raise RuntimeError("bcrypt required. pip install bcrypt")
        return bcrypt.hashpw(self._to_bytes(password), bcrypt.gensalt(rounds)).decode('utf-8')

    def bcrypt_check(self, password: str, hashed: str) -> bool:
        if not HAS_BCRYPT:
            raise RuntimeError("bcrypt required. pip install bcrypt")
        return bcrypt.checkpw(self._to_bytes(password), hashed.encode('utf-8'))

    # argon2 (external)
    def argon2_hash(self, password: str) -> str:
        if not HAS_ARGON2:
            raise RuntimeError("argon2-cffi required. pip install argon2-cffi")
        ph = PasswordHasher()
        return ph.hash(password)

    def argon2_verify(self, password: str, hashed: str) -> bool:
        if not HAS_ARGON2:
            raise RuntimeError("argon2-cffi required. pip install argon2-cffi")
        ph = PasswordHasher()
        try:
            return ph.verify(hashed, password)
        except Exception:
            return False

    # --------------------------
    # Signed HMAC + Verify
    # --------------------------
    def sign_hmac(self, key: str, message: str) -> str:
        return hmac.new(self._to_bytes(key), self._to_bytes(message), hashlib.sha256).hexdigest()

    def verify_hmac(self, key: str, message: str, signature_hex: str) -> bool:
        expected = self.sign_hmac(key, message)
        return hmac.compare_digest(expected, signature_hex)
