import base64
import urllib.parse
import codecs

class Encoding_Decoding:
    def __init__(self):
        print("🔐 Encoding-Decoding module initialized — supports Base64, Hex, Binary, URL, and Unicode operations.")

    # ---------------------------------------------------------
    # 🧩 BASE64 ENCODING / DECODING
    # ---------------------------------------------------------
    def base64_encode(self, text: str) -> str:
        """Encodes text into Base64."""
        return base64.b64encode(text.encode('utf-8')).decode('utf-8')

    def base64_decode(self, encoded_text: str) -> str:
        """Decodes Base64 text into normal string."""
        return base64.b64decode(encoded_text.encode('utf-8')).decode('utf-8')

    # ---------------------------------------------------------
    # 🧱 HEX ENCODING / DECODING
    # ---------------------------------------------------------
    def hex_encode(self, text: str) -> str:
        """Encodes text into hexadecimal representation."""
        return text.encode('utf-8').hex()

    def hex_decode(self, hex_text: str) -> str:
        """Decodes hexadecimal text into normal string."""
        return bytes.fromhex(hex_text).decode('utf-8')

    # ---------------------------------------------------------
    # 💾 BINARY ENCODING / DECODING
    # ---------------------------------------------------------
    def binary_encode(self, text: str) -> str:
        """Encodes text into binary representation."""
        return ' '.join(format(ord(char), '08b') for char in text)

    def binary_decode(self, binary_text: str) -> str:
        """Decodes binary string into normal text."""
        return ''.join(chr(int(b, 2)) for b in binary_text.split())

    # ---------------------------------------------------------
    # 🌐 URL ENCODING / DECODING
    # ---------------------------------------------------------
    def url_encode(self, text: str) -> str:
        """Encodes URL or text safely for web transmission."""
        return urllib.parse.quote(text)

    def url_decode(self, encoded_text: str) -> str:
        """Decodes URL-encoded text back to original."""
        return urllib.parse.unquote(encoded_text)

    # ---------------------------------------------------------
    # 🔡 UTF ENCODING / DECODING
    # ---------------------------------------------------------
    def utf_encode(self, text: str, encoding='utf-8') -> bytes:
        """Encodes text using given UTF format (utf-8, utf-16, utf-32)."""
        return text.encode(encoding)

    def utf_decode(self, data: bytes, encoding='utf-8') -> str:
        """Decodes bytes using given UTF format."""
        return data.decode(encoding)

    # ---------------------------------------------------------
    # 📁 FILE ENCODING / DECODING
    # ---------------------------------------------------------
    def file_to_base64(self, file_path: str) -> str:
        """Encodes a file into Base64 string."""
        with open(file_path, 'rb') as f:
            return base64.b64encode(f.read()).decode('utf-8')

    def base64_to_file(self, encoded_data: str, output_path: str):
        """Decodes Base64 string and writes binary file."""
        with open(output_path, 'wb') as f:
            f.write(base64.b64decode(encoded_data))

    # ---------------------------------------------------------
    # 🧮 ASCII CONVERSION
    # ---------------------------------------------------------
    def text_to_ascii(self, text: str) -> list:
        """Converts text to list of ASCII codes."""
        return [ord(c) for c in text]

    def ascii_to_text(self, ascii_list: list) -> str:
        """Converts list of ASCII codes to text."""
        return ''.join(chr(i) for i in ascii_list)

    # ---------------------------------------------------------
    # 🔄 UNICODE CONVERSION
    # ---------------------------------------------------------
    def text_to_unicode(self, text: str) -> list:
        """Converts text to list of Unicode code points."""
        return [f"U+{ord(c):04X}" for c in text]

    def unicode_to_text(self, unicode_list: list) -> str:
        """Converts list of Unicode code points to text."""
        return ''.join(chr(int(u.replace('U+', ''), 16)) for u in unicode_list)

    # ---------------------------------------------------------
    # 🔏 ROT13 ENCODING
    # ---------------------------------------------------------
    def rot13_encode(self, text: str) -> str:
        """Encodes/decodes text using ROT13 cipher."""
        return codecs.encode(text, 'rot_13')

    # ---------------------------------------------------------
    # 🧬 MORSE CODE ENCODING / DECODING
    # ---------------------------------------------------------
    def morse_encode(self, text: str) -> str:
        """Encodes text into Morse code."""
        morse = {
            'A': '.-', 'B': '-...', 'C': '-.-.', 'D': '-..',
            'E': '.', 'F': '..-.', 'G': '--.', 'H': '....',
            'I': '..', 'J': '.---', 'K': '-.-', 'L': '.-..',
            'M': '--', 'N': '-.', 'O': '---', 'P': '.--.',
            'Q': '--.-', 'R': '.-.', 'S': '...', 'T': '-',
            'U': '..-', 'V': '...-', 'W': '.--', 'X': '-..-',
            'Y': '-.--', 'Z': '--..', '1': '.----', '2': '..---',
            '3': '...--', '4': '....-', '5': '.....', '6': '-....',
            '7': '--...', '8': '---..', '9': '----.', '0': '-----',
            ' ': '/'
        }
        return ' '.join(morse.get(c.upper(), '?') for c in text)

    def morse_decode(self, code: str) -> str:
        """Decodes Morse code into text."""
        morse = {
            '.-': 'A', '-...': 'B', '-.-.': 'C', '-..': 'D',
            '.': 'E', '..-.': 'F', '--.': 'G', '....': 'H',
            '..': 'I', '.---': 'J', '-.-': 'K', '.-..': 'L',
            '--': 'M', '-.': 'N', '---': 'O', '.--.': 'P',
            '--.-': 'Q', '.-.': 'R', '...': 'S', '-': 'T',
            '..-': 'U', '...-': 'V', '.--': 'W', '-..-': 'X',
            '-.--': 'Y', '--..': 'Z', '.----': '1', '..---': '2',
            '...--': '3', '....-': '4', '.....': '5', '-....': '6',
            '--...': '7', '---..': '8', '----.': '9', '-----': '0',
            '/': ' '
        }
        return ''.join(morse.get(c, '?') for c in code.split())
