import matplotlib.pyplot as plt
import math
class ComplexNumber:
    def __init__(self, real=0.0, imag=0.0):
        self.real = real
        self.imag = imag

    def __str__(self):
        if self.imag >= 0:
            return f"{self.real} + {self.imag}i"
        else:
            return f"{self.real} - {abs(self.imag)}i"

    # =========================
    #   Basic Operations
    # =========================
    def add(self, other):
        return ComplexNumber(self.real + other.real, self.imag + other.imag)

    def subtract(self, other):
        return ComplexNumber(self.real - other.real, self.imag - other.imag)

    def multiply(self, other):
        r = self.real * other.real - self.imag * other.imag
        i = self.real * other.imag + self.imag * other.real
        return ComplexNumber(r, i)

    def divide(self, other):
        denominator = other.real**2 + other.imag**2
        if denominator == 0:
            raise ZeroDivisionError("Cannot divide by zero complex number")
        r = (self.real * other.real + self.imag * other.imag) / denominator
        i = (self.imag * other.real - self.real * other.imag) / denominator
        return ComplexNumber(r, i)

    # =========================
    #   Properties
    # =========================
    def conjugate(self):
        return ComplexNumber(self.real, -self.imag)

    def magnitude(self):
        return math.sqrt(self.real**2 + self.imag**2)

    def phase(self):
        return math.atan2(self.imag, self.real)

    def to_polar(self):
        return (self.magnitude(), self.phase())

    @staticmethod
    def from_polar(r, theta):
        return ComplexNumber(r * math.cos(theta), r * math.sin(theta))

    # =========================
    #   Utility Functions
    # =========================
    def equals(self, other):
        return self.real == other.real and self.imag == other.imag

    def is_real(self):
        return self.imag == 0

    def is_imaginary(self):
        return self.real == 0 and self.imag != 0

    # =========================
    #   Plotting Function
    # =========================
    def plot(self):
        plt.figure(figsize=(5, 5))
        plt.axhline(0, color='gray', linewidth=0.8)
        plt.axvline(0, color='gray', linewidth=0.8)
        plt.scatter(self.real, self.imag, color='red', s=100, label=str(self))
        plt.quiver(0, 0, self.real, self.imag, angles='xy', scale_units='xy', scale=1, color='blue')
        plt.title(f"Complex Number: {self}")
        plt.xlabel("Real Axis")
        plt.ylabel("Imaginary Axis")
        plt.grid(True)
        plt.legend()
        plt.xlim(-max(5, abs(self.real)+1), max(5, abs(self.real)+1))
        plt.ylim(-max(5, abs(self.imag)+1), max(5, abs(self.imag)+1))
        plt.show()


