import math
class Chemistry:
    def __init__(self):
        print("⚗️ Chemistry module initialized — includes classical and quantum functions.")

    # -----------------------------------------------------------
    # 🌍 CONSTANTS
    # -----------------------------------------------------------
    R = 8.314           # Gas constant (J/mol·K)
    NA = 6.022e23       # Avogadro's number (1/mol)
    h = 6.626e-34       # Planck constant (J·s)
    c = 3.0e8           # Speed of light (m/s)
    e = 1.602e-19       # Charge of electron (C)
    me = 9.109e-31      # Mass of electron (kg)
    kB = 1.38e-23       # Boltzmann constant (J/K)
    _elements = [
    {
        "number": 1,
        "symbol": "H",
        "name": "Hydrogen",
        "mass": 1.008,
        "group": 1,
        "period": 1,
        "type": "Nonmetal"
    },
    {
        "number": 2,
        "symbol": "He",
        "name": "Helium",
        "mass": 4.0026,
        "group": 18,
        "period": 1,
        "type": "Noble gas"
    },
    {
        "number": 3,
        "symbol": "Li",
        "name": "Lithium",
        "mass": 6.94,
        "group": 1,
        "period": 2,
        "type": "Alkali metal"
    },
    {
        "number": 4,
        "symbol": "Be",
        "name": "Beryllium",
        "mass": 9.0122,
        "group": 2,
        "period": 2,
        "type": "Alkaline earth metal"
    },
    {
        "number": 5,
        "symbol": "B",
        "name": "Boron",
        "mass": 10.81,
        "group": 13,
        "period": 2,
        "type": "Metalloid"
    },
    {
        "number": 6,
        "symbol": "C",
        "name": "Carbon",
        "mass": 12.011,
        "group": 14,
        "period": 2,
        "type": "Nonmetal"
    },
    {
        "number": 7,
        "symbol": "N",
        "name": "Nitrogen",
        "mass": 14.007,
        "group": 15,
        "period": 2,
        "type": "Nonmetal"
    },
    {
        "number": 8,
        "symbol": "O",
        "name": "Oxygen",
        "mass": 15.999,
        "group": 16,
        "period": 2,
        "type": "Nonmetal"
    },
    {
        "number": 9,
        "symbol": "F",
        "name": "Fluorine",
        "mass": 18.998,
        "group": 17,
        "period": 2,
        "type": "Halogen"
    },
    {
        "number": 10,
        "symbol": "Ne",
        "name": "Neon",
        "mass": 20.180,
        "group": 18,
        "period": 2,
        "type": "Noble gas"
    },
    {
        "number": 11,
        "symbol": "Na",
        "name": "Sodium",
        "mass": 22.990,
        "group": 1,
        "period": 3,
        "type": "Alkali metal"
    },
    {
        "number": 12,
        "symbol": "Mg",
        "name": "Magnesium",
        "mass": 24.305,
        "group": 2,
        "period": 3,
        "type": "Alkaline earth metal"
    },
    {
        "number": 13,
        "symbol": "Al",
        "name": "Aluminum",
        "mass": 26.982,
        "group": 13,
        "period": 3,
        "type": "Post-transition metal"
    },
    {
        "number": 14,
        "symbol": "Si",
        "name": "Silicon",
        "mass": 28.085,
        "group": 14,
        "period": 3,
        "type": "Metalloid"
    },
    {
        "number": 15,
        "symbol": "P",
        "name": "Phosphorus",
        "mass": 30.974,
        "group": 15,
        "period": 3,
        "type": "Nonmetal"
    },
    {
        "number": 16,
        "symbol": "S",
        "name": "Sulfur",
        "mass": 32.06,
        "group": 16,
        "period": 3,
        "type": "Nonmetal"
    },
    {
        "number": 17,
        "symbol": "Cl",
        "name": "Chlorine",
        "mass": 35.45,
        "group": 17,
        "period": 3,
        "type": "Halogen"
    },
    {
        "number": 18,
        "symbol": "Ar",
        "name": "Argon",
        "mass": 39.948,
        "group": 18,
        "period": 3,
        "type": "Noble gas"
    },
    {
        "number": 19,
        "symbol": "K",
        "name": "Potassium",
        "mass": 39.098,
        "group": 1,
        "period": 4,
        "type": "Alkali metal"
    },
    {
        "number": 20,
        "symbol": "Ca",
        "name": "Calcium",
        "mass": 40.078,
        "group": 2,
        "period": 4,
        "type": "Alkaline earth metal"
    },
    {
        "number": 21,
        "symbol": "Sc",
        "name": "Scandium",
        "mass": 44.956,
        "group": 3,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 22,
        "symbol": "Ti",
        "name": "Titanium",
        "mass": 47.867,
        "group": 4,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 23,
        "symbol": "V",
        "name": "Vanadium",
        "mass": 50.942,
        "group": 5,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 24,
        "symbol": "Cr",
        "name": "Chromium",
        "mass": 51.996,
        "group": 6,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 25,
        "symbol": "Mn",
        "name": "Manganese",
        "mass": 54.938,
        "group": 7,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 26,
        "symbol": "Fe",
        "name": "Iron",
        "mass": 55.845,
        "group": 8,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 27,
        "symbol": "Co",
        "name": "Cobalt",
        "mass": 58.933,
        "group": 9,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 28,
        "symbol": "Ni",
        "name": "Nickel",
        "mass": 58.693,
        "group": 10,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 29,
        "symbol": "Cu",
        "name": "Copper",
        "mass": 63.546,
        "group": 11,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 30,
        "symbol": "Zn",
        "name": "Zinc",
        "mass": 65.38,
        "group": 12,
        "period": 4,
        "type": "Transition metal"
    },
    {
        "number": 31,
        "symbol": "Ga",
        "name": "Gallium",
        "mass": 69.723,
        "group": 13,
        "period": 4,
        "type": "Post-transition metal"
    },
    {
        "number": 32,
        "symbol": "Ge",
        "name": "Germanium",
        "mass": 72.630,
        "group": 14,
        "period": 4,
        "type": "Metalloid"
    },
    {
        "number": 33,
        "symbol": "As",
        "name": "Arsenic",
        "mass": 74.922,
        "group": 15,
        "period": 4,
        "type": "Metalloid"
    },
    {
        "number": 34,
        "symbol": "Se",
        "name": "Selenium",
        "mass": 78.971,
        "group": 16,
        "period": 4,
        "type": "Nonmetal"
    },
    {
        "number": 35,
        "symbol": "Br",
        "name": "Bromine",
        "mass": 79.904,
        "group": 17,
        "period": 4,
        "type": "Halogen"
    },
    {
        "number": 36,
        "symbol": "Kr",
        "name": "Krypton",
        "mass": 83.798,
        "group": 18,
        "period": 4,
        "type": "Noble gas"
    },
    {
        "number": 37,
        "symbol": "Rb",
        "name": "Rubidium",
        "mass": 85.468,
        "group": 1,
        "period": 5,
        "type": "Alkali metal"
    },
    {
        "number": 38,
        "symbol": "Sr",
        "name": "Strontium",
        "mass": 87.62,
        "group": 2,
        "period": 5,
        "type": "Alkaline earth metal"
    },
    {
        "number": 39,
        "symbol": "Y",
        "name": "Yttrium",
        "mass": 88.906,
        "group": 3,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 40,
        "symbol": "Zr",
        "name": "Zirconium",
        "mass": 91.224,
        "group": 4,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 41,
        "symbol": "Nb",
        "name": "Niobium",
        "mass": 92.906,
        "group": 5,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 42,
        "symbol": "Mo",
        "name": "Molybdenum",
        "mass": 95.95,
        "group": 6,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 43,
        "symbol": "Tc",
        "name": "Technetium",
        "mass": 98,
        "group": 7,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 44,
        "symbol": "Ru",
        "name": "Ruthenium",
        "mass": 101.07,
        "group": 8,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 45,
        "symbol": "Rh",
        "name": "Rhodium",
        "mass": 102.91,
        "group": 9,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 46,
        "symbol": "Pd",
        "name": "Palladium",
        "mass": 106.42,
        "group": 10,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 47,
        "symbol": "Ag",
        "name": "Silver",
        "mass": 107.87,
        "group": 11,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 48,
        "symbol": "Cd",
        "name": "Cadmium",
        "mass": 112.41,
        "group": 12,
        "period": 5,
        "type": "Transition metal"
    },
    {
        "number": 49,
        "symbol": "In",
        "name": "Indium",
        "mass": 114.82,
        "group": 13,
        "period": 5,
        "type": "Post-transition metal"
    },
    {
        "number": 50,
        "symbol": "Sn",
        "name": "Tin",
        "mass": 118.71,
        "group": 14,
        "period": 5,
        "type": "Post-transition metal"
    },
    {
        "number": 51,
        "symbol": "Sb",
        "name": "Antimony",
        "mass": 121.76,
        "group": 15,
        "period": 5,
        "type": "Metalloid"
    },
    {
        "number": 52,
        "symbol": "Te",
        "name": "Tellurium",
        "mass": 127.60,
        "group": 16,
        "period": 5,
        "type": "Metalloid"
    },
    {
        "number": 53,
        "symbol": "I",
        "name": "Iodine",
        "mass": 126.90,
        "group": 17,
        "period": 5,
        "type": "Halogen"
    },
    {
        "number": 54,
        "symbol": "Xe",
        "name": "Xenon",
        "mass": 131.29,
        "group": 18,
        "period": 5,
        "type": "Noble gas"
    },
    {
        "number": 55,
        "symbol": "Cs",
        "name": "Cesium",
        "mass": 132.91,
        "group": 1,
        "period": 6,
        "type": "Alkali metal"
    },
    {
        "number": 56,
        "symbol": "Ba",
        "name": "Barium",
        "mass": 137.33,
        "group": 2,
        "period": 6,
        "type": "Alkaline earth metal"
    },
    {
        "number": 57,
        "symbol": "La",
        "name": "Lanthanum",
        "mass": 138.91,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 58,
        "symbol": "Ce",
        "name": "Cerium",
        "mass": 140.12,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 59,
        "symbol": "Pr",
        "name": "Praseodymium",
        "mass": 140.91,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 60,
        "symbol": "Nd",
        "name": "Neodymium",
        "mass": 144.24,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 61,
        "symbol": "Pm",
        "name": "Promethium",
        "mass": 145,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 62,
        "symbol": "Sm",
        "name": "Samarium",
        "mass": 150.36,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 63,
        "symbol": "Eu",
        "name": "Europium",
        "mass": 151.96,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 64,
        "symbol": "Gd",
        "name": "Gadolinium",
        "mass": 157.25,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 65,
        "symbol": "Tb",
        "name": "Terbium",
        "mass": 158.93,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 66,
        "symbol": "Dy",
        "name": "Dysprosium",
        "mass": 162.50,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 67,
        "symbol": "Ho",
        "name": "Holmium",
        "mass": 164.93,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 68,
        "symbol": "Er",
        "name": "Erbium",
        "mass": 167.26,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 69,
        "symbol": "Tm",
        "name": "Thulium",
        "mass": 168.93,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 70,
        "symbol": "Yb",
        "name": "Ytterbium",
        "mass": 173.05,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 71,
        "symbol": "Lu",
        "name": "Lutetium",
        "mass": 174.97,
        "group": 3,
        "period": 6,
        "type": "Lanthanide"
    },
    {
        "number": 72,
        "symbol": "Hf",
        "name": "Hafnium",
        "mass": 178.49,
        "group": 4,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 73,
        "symbol": "Ta",
        "name": "Tantalum",
        "mass": 180.95,
        "group": 5,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 74,
        "symbol": "W",
        "name": "Tungsten",
        "mass": 183.84,
        "group": 6,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 75,
        "symbol": "Re",
        "name": "Rhenium",
        "mass": 186.21,
        "group": 7,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 76,
        "symbol": "Os",
        "name": "Osmium",
        "mass": 190.23,
        "group": 8,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 77,
        "symbol": "Ir",
        "name": "Iridium",
        "mass": 192.22,
        "group": 9,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 78,
        "symbol": "Pt",
        "name": "Platinum",
        "mass": 195.08,
        "group": 10,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 79,
        "symbol": "Au",
        "name": "Gold",
        "mass": 196.97,
        "group": 11,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 80,
        "symbol": "Hg",
        "name": "Mercury",
        "mass": 200.59,
        "group": 12,
        "period": 6,
        "type": "Transition metal"
    },
    {
        "number": 81,
        "symbol": "Tl",
        "name": "Thallium",
        "mass": 204.38,
        "group": 13,
        "period": 6,
        "type": "Post-transition metal"
    },
    {
        "number": 82,
        "symbol": "Pb",
        "name": "Lead",
        "mass": 207.2,
        "group": 14,
        "period": 6,
        "type": "Post-transition metal"
    },
    {
        "number": 83,
        "symbol": "Bi",
        "name": "Bismuth",
        "mass": 208.98,
        "group": 15,
        "period": 6,
        "type": "Post-transition metal"
    },
    {
        "number": 84,
        "symbol": "Po",
        "name": "Polonium",
        "mass": 209,
        "group": 16,
        "period": 6,
        "type": "Metalloid"
    },
    {
        "number": 85,
        "symbol": "At",
        "name": "Astatine",
        "mass": 210,
        "group": 17,
        "period": 6,
        "type": "Halogen"
    },
    {
        "number": 86,
        "symbol": "Rn",
        "name": "Radon",
        "mass": 222,
        "group": 18,
        "period": 6,
        "type": "Noble gas"
    },
    {
        "number": 87,
        "symbol": "Fr",
        "name": "Francium",
        "mass": 223,
        "group": 1,
        "period": 7,
        "type": "Alkali metal"
    },
    {
        "number": 88,
        "symbol": "Ra",
        "name": "Radium",
        "mass": 226,
        "group": 2,
        "period": 7,
        "type": "Alkaline earth metal"
    },
    {
        "number": 89,
        "symbol": "Ac",
        "name": "Actinium",
        "mass": 227,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 90,
        "symbol": "Th",
        "name": "Thorium",
        "mass": 232.04,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 91,
        "symbol": "Pa",
        "name": "Protactinium",
        "mass": 231.04,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 92,
        "symbol": "U",
        "name": "Uranium",
        "mass": 238.03,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 93,
        "symbol": "Np",
        "name": "Neptunium",
        "mass": 237,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 94,
        "symbol": "Pu",
        "name": "Plutonium",
        "mass": 244,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 95,
        "symbol": "Am",
        "name": "Americium",
        "mass": 243,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 96,
        "symbol": "Cm",
        "name": "Curium",
        "mass": 247,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 97,
        "symbol": "Bk",
        "name": "Berkelium",
        "mass": 247,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 98,
        "symbol": "Cf",
        "name": "Californium",
        "mass": 251,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 99,
        "symbol": "Es",
        "name": "Einsteinium",
        "mass": 252,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 100,
        "symbol": "Fm",
        "name": "Fermium",
        "mass": 257,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 101,
        "symbol": "Md",
        "name": "Mendelevium",
        "mass": 258,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 102,
        "symbol": "No",
        "name": "Nobelium",
        "mass": 259,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 103,
        "symbol": "Lr",
        "name": "Lawrencium",
        "mass": 266,
        "group": 3,
        "period": 7,
        "type": "Actinide"
    },
    {
        "number": 104,
        "symbol": "Rf",
        "name": "Rutherfordium",
        "mass": 267,
        "group": 4,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 105,
        "symbol": "Db",
        "name": "Dubnium",
        "mass": 268,
        "group": 5,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 106,
        "symbol": "Sg",
        "name": "Seaborgium",
        "mass": 269,
        "group": 6,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 107,
        "symbol": "Bh",
        "name": "Bohrium",
        "mass": 270,
        "group": 7,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 108,
        "symbol": "Hs",
        "name": "Hassium",
        "mass": 277,
        "group": 8,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 109,
        "symbol": "Mt",
        "name": "Meitnerium",
        "mass": 278,
        "group": 9,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 110,
        "symbol": "Ds",
        "name": "Darmstadtium",
        "mass": 281,
        "group": 10,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 111,
        "symbol": "Rg",
        "name": "Roentgenium",
        "mass": 282,
        "group": 11,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 112,
        "symbol": "Cn",
        "name": "Copernicium",
        "mass": 285,
        "group": 12,
        "period": 7,
        "type": "Transition metal"
    },
    {
        "number": 113,
        "symbol": "Nh",
        "name": "Nihonium",
        "mass": 286,
        "group": 13,
        "period": 7,
        "type": "Post-transition metal"
    },
    {
        "number": 114,
        "symbol": "Fl",
        "name": "Flerovium",
        "mass": 289,
        "group": 14,
        "period": 7,
        "type": "Post-transition metal"
    },
    {
        "number": 115,
        "symbol": "Mc",
        "name": "Moscovium",
        "mass": 290,
        "group": 15,
        "period": 7,
        "type": "Post-transition metal"
    },
    {
        "number": 116,
        "symbol": "Lv",
        "name": "Livermorium",
        "mass": 293,
        "group": 16,
        "period": 7,
        "type": "Post-transition metal"
    },
    {
        "number": 117,
        "symbol": "Ts",
        "name": "Tennessine",
        "mass": 294,
        "group": 17,
        "period": 7,
        "type": "Halogen"
    },
    {
        "number": 118,
        "symbol": "Og",
        "name": "Oganesson",
        "mass": 294,
        "group": 18,
        "period": 7,
        "type": "Noble gas"
    }
]      # Blank element list (can be filled later)

    # -----------------------------------------------------------
    # 🧪 BASIC CHEMISTRY
    # -----------------------------------------------------------
    def molar_mass(self, mass, moles):
        """M = m / n"""
        return mass / moles if moles != 0 else float('inf')

    def moles_from_mass(self, mass, molar_mass):
        """n = m / M"""
        return mass / molar_mass if molar_mass != 0 else float('inf')

    def mass_from_moles(self, moles, molar_mass):
        """m = n * M"""
        return moles * molar_mass

    def concentration(self, moles, volume):
        """C = n / V (mol/L)"""
        return moles / volume if volume != 0 else float('inf')

    def percent_yield(self, actual, theoretical):
        """% yield = (actual / theoretical) * 100"""
        return (actual / theoretical) * 100 if theoretical != 0 else 0

    # -----------------------------------------------------------
    # 🌡️ GAS LAWS & THERMODYNAMICS
    # -----------------------------------------------------------
    def ideal_gas_pressure(self, n, V, T):
        """P = (nRT) / V"""
        return (n * self.R * T) / V if V != 0 else float('inf')

    def ideal_gas_volume(self, n, P, T):
        """V = (nRT) / P"""
        return (n * self.R * T) / P if P != 0 else float('inf')

    def ideal_gas_temperature(self, P, V, n):
        """T = (PV) / (nR)"""
        return (P * V) / (n * self.R) if n != 0 else float('inf')

    def heat_energy(self, mass, specific_heat, delta_T):
        """Q = m * c * ΔT"""
        return mass * specific_heat * delta_T

    def gibbs_free_energy(self, delta_H, delta_S, T):
        """ΔG = ΔH - TΔS"""
        return delta_H - T * delta_S

    def equilibrium_constant(self, delta_G, T):
        """K = e^(-ΔG / RT)"""
        return math.exp(-delta_G / (self.R * T))

    # -----------------------------------------------------------
    # ⚛️ ATOMIC & QUANTUM CHEMISTRY
    # -----------------------------------------------------------
    def energy_level_hydrogen(self, n):
        """Eₙ = -13.6 eV / n²"""
        return -13.6 / (n ** 2)

    def photon_energy(self, wavelength):
        """E = h * c / λ"""
        return (self.h * self.c) / wavelength

    def wavelength_from_energy(self, energy):
        """λ = h * c / E"""
        return (self.h * self.c) / energy

    def de_broglie_wavelength(self, mass, velocity):
        """λ = h / (m * v)"""
        return self.h / (mass * velocity)

    def uncertainty(self, delta_x):
        """Δp ≥ h / (4πΔx)"""
        return self.h / (4 * math.pi * delta_x)

    def particle_energy(self, frequency):
        """E = h * f"""
        return self.h * frequency

    def bohr_radius(self, n=1):
        """rₙ = (ε₀ * h² * n²) / (π * me * e²)"""
        ε0 = 8.854e-12
        return (ε0 * self.h**2 * n**2) / (math.pi * self.me * self.e**2)

    def ionization_energy(self, n1, n2):
        """ΔE (eV) = 13.6 * (1/n1² - 1/n2²)"""
        return 13.6 * ((1 / (n1 ** 2)) - (1 / (n2 ** 2)))

    # -----------------------------------------------------------
    # 🧮 STOICHIOMETRY
    # -----------------------------------------------------------
    def limiting_reactant(self, moles_A, ratio_A, moles_B, ratio_B):
        """Return limiting reactant based on stoichiometric ratios"""
        if (moles_A / ratio_A) < (moles_B / ratio_B):
            return "Reactant A is limiting"
        elif (moles_B / ratio_B) < (moles_A / ratio_A):
            return "Reactant B is limiting"
        else:
            return "Reactants are in perfect ratio"

    def percent_purity(self, pure_mass, impure_mass):
        """% Purity = (pure_mass / impure_mass) * 100"""
        return (pure_mass / impure_mass) * 100 if impure_mass != 0 else 0

    # -----------------------------------------------------------
    # 🔥 SPECTROSCOPY & ENERGY TRANSITIONS
    # -----------------------------------------------------------
    def frequency_from_wavelength(self, wavelength):
        """f = c / λ"""
        return self.c / wavelength

    def wavelength_from_frequency(self, frequency):
        """λ = c / f"""
        return self.c / frequency

    def photon_wavenumber(self, wavelength):
        """ν̃ = 1 / λ (in m⁻¹)"""
        return 1 / wavelength

    # -----------------------------------------------------------
    # 🔄 CONVERSIONS
    # -----------------------------------------------------------
    def joule_to_ev(self, joules):
        """Convert Joules to electronvolts"""
        return joules / self.e

    def ev_to_joule(self, ev):
        """Convert electronvolts to Joules"""
        return ev * self.e

    # -----------------------------------------------------------
    # 🧫 ELEMENT INFORMATION
    # -----------------------------------------------------------
    def atom_info(self, query):
        """
        Get atomic details by number, symbol, or name.
        Example:
            atom_info(8)
            atom_info('O')
            atom_info('Oxygen')
        """
        query_str = str(query).capitalize().strip()
        for element in self._elements:
            if (
                str(element["number"]) == str(query)
                or element["symbol"].lower() == str(query).lower()
                or element["name"].lower() == str(query).lower()
            ):
                return {
                    "Atomic Number": element["number"],
                    "Symbol": element["symbol"],
                    "Name": element["name"],
                    "Atomic Mass (u)": element["mass"],
                    "Group": element["group"],
                    "Period": element["period"],
                    "Type": element["type"]
                }
        return f"No element found for '{query}'"

