

from typing import Tuple, List, Dict, Optional, Union,Any
import mmap
import ctypes
import os
import platform
import sys
import tempfile
import struct
import logging

# --- Optional libs (robust imports) ---
KEYSTONE_AVAILABLE = False
CAPSTONE_AVAILABLE = False
UC_AVAILABLE = False

Ks = None
KsError = None
KS_ARCH_X86 = KS_ARCH_ARM = KS_ARCH_ARM64 = KS_ARCH_MIPS = KS_ARCH_PPC = KS_ARCH_SPARC = KS_ARCH_SYSTEMZ = None
KS_ARCH_HEXAGON = None
KS_MODE_16 = KS_MODE_32 = KS_MODE_64 = KS_MODE_ARM = KS_MODE_THUMB = None
KS_OPT_SYNTAX_INTEL = KS_OPT_SYNTAX_ATT = None

try:
    import keystone
    # bring commonly-used names, but tolerate missing attrs
    Ks = getattr(keystone, "Ks", None)
    KsError = getattr(keystone, "KsError", Exception)
    KS_ARCH_X86 = getattr(keystone, "KS_ARCH_X86", None)
    KS_ARCH_ARM = getattr(keystone, "KS_ARCH_ARM", None)
    KS_ARCH_ARM64 = getattr(keystone, "KS_ARCH_ARM64", None)
    KS_ARCH_MIPS = getattr(keystone, "KS_ARCH_MIPS", None)
    KS_ARCH_PPC = getattr(keystone, "KS_ARCH_PPC", None)
    KS_ARCH_SPARC = getattr(keystone, "KS_ARCH_SPARC", None)
    KS_ARCH_SYSTEMZ = getattr(keystone, "KS_ARCH_SYSTEMZ", None)
    KS_ARCH_HEXAGON = getattr(keystone, "KS_ARCH_HEXAGON", None)
    KS_MODE_16 = getattr(keystone, "KS_MODE_16", None)
    KS_MODE_32 = getattr(keystone, "KS_MODE_32", None)
    KS_MODE_64 = getattr(keystone, "KS_MODE_64", None)
    KS_MODE_ARM = getattr(keystone, "KS_MODE_ARM", None)
    KS_MODE_THUMB = getattr(keystone, "KS_MODE_THUMB", None)
    KS_OPT_SYNTAX_INTEL = getattr(keystone, "KS_OPT_SYNTAX_INTEL", None)
    KS_OPT_SYNTAX_ATT = getattr(keystone, "KS_OPT_SYNTAX_ATT", None)
    if Ks is not None:
        KEYSTONE_AVAILABLE = True
except Exception:
    # leave variables as None
    pass

try:
    import capstone
    Cs = getattr(capstone, "Cs", None)
    CS_ARCH_X86 = getattr(capstone, "CS_ARCH_X86", None)
    CS_ARCH_ARM = getattr(capstone, "CS_ARCH_ARM", None)
    CS_ARCH_ARM64 = getattr(capstone, "CS_ARCH_ARM64", None)
    CS_ARCH_MIPS = getattr(capstone, "CS_ARCH_MIPS", None)
    CS_ARCH_PPC = getattr(capstone, "CS_ARCH_PPC", None)
    CS_MODE_16 = getattr(capstone, "CS_MODE_16", None)
    CS_MODE_32 = getattr(capstone, "CS_MODE_32", None)
    CS_MODE_64 = getattr(capstone, "CS_MODE_64", None)
    CS_MODE_THUMB = getattr(capstone, "CS_MODE_THUMB", None)
    if Cs is not None:
        CAPSTONE_AVAILABLE = True
except Exception:
    Cs = None

try:
    import unicorn
    Uc = getattr(unicorn, "Uc", None)
    UC_ARCH_X86 = getattr(unicorn, "UC_ARCH_X86", None)
    UC_ARCH_ARM = getattr(unicorn, "UC_ARCH_ARM", None)
    UC_ARCH_ARM64 = getattr(unicorn, "UC_ARCH_ARM64", None)
    UC_ARCH_MIPS = getattr(unicorn, "UC_ARCH_MIPS", None)
    UC_MODE_16 = getattr(unicorn, "UC_MODE_16", None)
    UC_MODE_32 = getattr(unicorn, "UC_MODE_32", None)
    UC_MODE_64 = getattr(unicorn, "UC_MODE_64", None)
    UC_MODE_ARM = getattr(unicorn, "UC_MODE_ARM", None)
    UC_MODE_THUMB = getattr(unicorn, "UC_MODE_THUMB", None)
    # const modules
    try:
        import unicorn.x86_const as uc_x86_const
    except Exception:
        uc_x86_const = None
    try:
        import unicorn.arm_const as uc_arm_const
    except Exception:
        uc_arm_const = None
    try:
        import unicorn.arm64_const as uc_arm64_const
    except Exception:
        uc_arm64_const = None
    if Uc is not None:
        UC_AVAILABLE = True
except Exception:
    Uc = None
    uc_x86_const = None
    uc_arm_const = None
    uc_arm64_const = None

# For type clarity
BytesLike = Union[bytes, bytearray, str]  # str when hex string is allowed

# Setup logging
logger = logging.getLogger("Assembly")
if not logger.handlers:
    logging.basicConfig(level=logging.INFO)


class AssemblyError(Exception):
    pass


class Assembly:
    """
    Assembly helper using Keystone (assembler), Capstone (disassembler),
    Unicorn (emulator), and native mmap/ctypes for direct execution.
    """

    def __init__(self, arch: str = "x86", mode: Union[int, str] = 64, syntax: str = "intel"):
        self.arch = arch.lower()
        self.mode = mode
        self.syntax = syntax.lower() if isinstance(syntax, str) else "intel"
        self.ks = None
        self._init_keystone()
        # emulator (Unicorn) instance placeholder
        self.uc: Optional[Any] = None
        self.emu_mapped: Dict[int, Union[int, tuple]] = {}  # addr -> size (UC) or (tmpfile, size) simulated
        self.emu_hooks = []
        logger.info(f"Assembly initialized ({self.arch}, {self.mode}, {self.syntax})")

    # ---------------- Keyston initialization ----------------
    def _init_keystone(self):
        if not KEYSTONE_AVAILABLE or Ks is None:
            self.ks = None
            logger.warning("Keystone engine not available. Assembly will be disabled.")
            return

        arch_map = {
            "x86": KS_ARCH_X86,
            "arm": KS_ARCH_ARM,
            "arm64": KS_ARCH_ARM64,
            "mips": KS_ARCH_MIPS,
            "ppc": KS_ARCH_PPC,
            "sparc": KS_ARCH_SPARC,
            "systemz": KS_ARCH_SYSTEMZ,
            "hexagon": KS_ARCH_HEXAGON,
        }
        mode_map = {
            16: KS_MODE_16,
            32: KS_MODE_32,
            64: KS_MODE_64,
            "arm": KS_MODE_ARM,
            "thumb": KS_MODE_THUMB,
        }

        arch_val = arch_map.get(self.arch)
        if arch_val is None:
            raise AssemblyError(f"Keystone does not support architecture '{self.arch}' in this wrapper.")
        mode_val = mode_map.get(self.mode)
        if mode_val is None:
            mode_val = KS_MODE_64 if self.mode == 64 else KS_MODE_32

        try:
            self.ks = Ks(arch_val, mode_val)
            # try setting syntax via attribute or option
            if self.arch == "x86" and KS_OPT_SYNTAX_INTEL is not None and KS_OPT_SYNTAX_ATT is not None:
                try:
                    # prefer option API if available
                    if hasattr(self.ks, "option"):
                        desired = KS_OPT_SYNTAX_INTEL if self.syntax == "intel" else KS_OPT_SYNTAX_ATT
                        try:
                            # some ks versions require integer key Ks.OPT_SYNTAX or attribute 'OPT_SYNTAX'
                            opt_key = getattr(Ks, "OPT_SYNTAX", None)
                            if opt_key is not None:
                                self.ks.option(opt_key, desired)
                            else:
                                # try attribute assignment
                                self.ks.syntax = desired
                        except Exception:
                            # fallback to attribute
                            try:
                                self.ks.syntax = desired
                            except Exception:
                                pass
                except Exception:
                    pass
        except Exception as e:
            # map to AssemblyError
            raise AssemblyError(f"Keystone initialization failed: {e}")

    # ---------------- Assembly ----------------
    def assemble(self, code: str, addr: int = 0x1000, as_hex: bool = True) -> Tuple[Union[str, bytes], int]:
        """
        Assemble the provided assembly text to bytes or hex string.
        Returns (data, count) where data is bytes or hex string depending on as_hex.
        """
        if not self.ks:
            raise AssemblyError("Keystone not available. Install keystone-engine to assemble.")
        try:
            encoding, count = self.ks.asm(code, addr)
            data = bytes(encoding)
            return (data.hex(), count) if as_hex else (data, count)
        except Exception as e:
            raise AssemblyError(f"Assembly failed: {e}")

    def batch_assemble(self, blocks: List[Dict], base_addr: int = 0x1000, as_hex: bool = True) -> List[Dict]:
        """
        Assemble multiple blocks.
        blocks: List of dicts: [{'code': 'mov eax, 1', 'addr': 0x1000}, ...]
        Returns list with {'addr', 'count', 'len', 'data'(hex or bytes)}
        """
        results = []
        for b in blocks:
            code = b.get("code", "")
            addr = b.get("addr", base_addr)
            data, count = self.assemble(code, addr, as_hex=False)
            results.append({
                "addr": addr,
                "count": count,
                "len": len(data),
                "data": data.hex() if as_hex else data
            })
        return results

    # ---------------- Disassembly ----------------
    def disassemble(self, code_bytes: BytesLike, addr: int = 0x1000) -> List[Tuple[int, str, str]]:
        """
        Disassemble bytes -> list of (addr, mnemonic, op_str)
        Accepts bytes or hex string.
        """
        if isinstance(code_bytes, str):
            s = code_bytes.strip()
            if s == "":
                return []
            # guess hex if only hex chars
            if all(c in "0123456789abcdefABCDEF" for c in s):
                code_bytes = bytes.fromhex(s)
            else:
                code_bytes = s.encode("latin1")
        if not CAPSTONE_AVAILABLE or Cs is None:
            raise AssemblyError("Capstone not available. Install capstone to disassemble.")

        arch_map = {
            "x86": CS_ARCH_X86,
            "arm": CS_ARCH_ARM,
            "arm64": CS_ARCH_ARM64,
            "mips": CS_ARCH_MIPS,
            "ppc": CS_ARCH_PPC,
        }
        mode_map = {
            16: CS_MODE_16,
            32: CS_MODE_32,
            64: CS_MODE_64,
            "thumb": CS_MODE_THUMB,
        }

        cs_arch = arch_map.get(self.arch)
        if cs_arch is None:
            raise AssemblyError(f"Disassembly: unsupported arch {self.arch}")

        cs_mode = mode_map.get(self.mode, CS_MODE_64 if self.mode == 64 else CS_MODE_32)
        cs = Cs(cs_arch, cs_mode)
        out = []
        for i in cs.disasm(code_bytes, addr):
            out.append((i.address, i.mnemonic, i.op_str))
        return out

    # ---------------- Native execution (dangerous) ----------------
    def execute_native(self, code_bytes: BytesLike, as_hex: bool = True):
        """
        Execute raw machine code within the current process memory.
        WARNING: Extremely dangerous. Use only on trusted code and in controlled environment.
        Works on POSIX via mmap and on Windows via VirtualAlloc.
        """
        if isinstance(code_bytes, str) and as_hex:
            code_bytes = bytes.fromhex(code_bytes.strip())
        elif isinstance(code_bytes, str) and not as_hex:
            code_bytes = code_bytes.encode('latin1')

        size = len(code_bytes)
        if size == 0:
            raise AssemblyError("No code to execute.")

        if os.name == "nt":
            # Windows: use VirtualAlloc to get executable RWX memory
            PAGE_EXECUTE_READWRITE = 0x40
            MEM_COMMIT = 0x1000
            MEM_RESERVE = 0x2000
            kernel32 = ctypes.windll.kernel32
            kernel32.VirtualAlloc.restype = ctypes.c_void_p
            addr = kernel32.VirtualAlloc(None, ctypes.c_size_t(size), MEM_COMMIT | MEM_RESERVE, PAGE_EXECUTE_READWRITE)
            if not addr:
                raise AssemblyError("VirtualAlloc failed.")
            # copy bytes to addr
            ctypes.memmove(addr, code_bytes, size)
            functype = ctypes.CFUNCTYPE(ctypes.c_void_p)
            func = functype(addr)
            logger.warning("Executing native code on Windows — this may crash your process.")
            try:
                res = func()
                return res
            finally:
                kernel32.VirtualFree(ctypes.c_void_p(addr), 0, 0x8000)  # MEM_RELEASE
        else:
            # POSIX: use mmap with PROT_EXEC if available
            PROT_READ = getattr(mmap, "PROT_READ", 1)
            PROT_WRITE = getattr(mmap, "PROT_WRITE", 2)
            PROT_EXEC = getattr(mmap, "PROT_EXEC", 4)
            prot = PROT_READ | PROT_WRITE | PROT_EXEC
            anon_flag = getattr(mmap, "MAP_ANONYMOUS", getattr(mmap, "MAP_ANON", 0))
            flags = mmap.MAP_PRIVATE | anon_flag if anon_flag != 0 else mmap.MAP_PRIVATE
            # create mapping
            mm = mmap.mmap(-1, size, prot=prot, flags=flags)
            try:
                mm.write(code_bytes)
                mm.seek(0)
                address = ctypes.addressof(ctypes.c_char.from_buffer(mm))
                functype = ctypes.CFUNCTYPE(ctypes.c_void_p)
                func = functype(address)
                logger.warning("Executing native code — this may crash your process. Proceed only if you know what you are doing.")
                try:
                    res = func()
                    return res
                finally:
                    pass
            finally:
                mm.close()

    # ---------------- Memory management for emulator ----------------
    def memory_alloc(self, size: int, addr: Optional[int] = None) -> int:
        """
        Allocate space inside the emulator address space. Returns mapped address or handle.
        If Unicorn is not available, simulate using a temporary file-backed mapping and return a handle.
        """
        if UC_AVAILABLE and self.uc is not None:
            # choose an address if not provided
            if addr is None:
                addr = 0x1000000 + (len(self.emu_mapped) * 0x10000)
            page_size = 0x1000
            alloc_size = ((size + page_size - 1) // page_size) * page_size
            self.uc.mem_map(addr, alloc_size)
            self.emu_mapped[addr] = alloc_size
            logger.info(f"Unicorn mapped {alloc_size} bytes at 0x{addr:X}")
            return addr
        else:
            tmp = tempfile.TemporaryFile()
            tmp.truncate(size)
            handle = id(tmp)
            # store object to keep it alive + size
            self.emu_mapped[handle] = (tmp, size)
            logger.info(f"Simulated allocation: handle {handle}, size {size}")
            return handle

    def memory_free(self, addr):
        if UC_AVAILABLE and self.uc is not None and isinstance(addr, int) and addr in self.emu_mapped:
            size = self.emu_mapped.pop(addr)
            self.uc.mem_unmap(addr, size)
            logger.info(f"Unmapped emulator memory at 0x{addr:X} size {size}")
            return True
        else:
            entry = self.emu_mapped.pop(addr, None)
            if entry:
                if isinstance(entry, tuple):
                    entry[0].close()
                logger.info(f"Freed simulated allocation {addr}")
                return True
        return False

    def memory_write(self, addr: int, data: BytesLike, offset: int = 0):
        """
        Write data to mapped memory. For simulated mappings, address is the handle returned by memory_alloc.
        Supports optional offset for simulated mappings.
        """
        if isinstance(data, str):
            s = data.strip()
            if all(c in "0123456789abcdefABCDEF" for c in s) and s != "":
                data = bytes.fromhex(s)
            else:
                data = data.encode('latin1')

        if UC_AVAILABLE and self.uc is not None and isinstance(addr, int) and addr in self.emu_mapped:
            self.uc.mem_write(addr + offset, data)
            logger.info(f"Wrote {len(data)} bytes to 0x{addr+offset:X} in emulator")
            return True
        else:
            entry = self.emu_mapped.get(addr)
            if entry and isinstance(entry, tuple):
                tmp, size = entry
                if offset < 0 or offset + len(data) > size:
                    raise AssemblyError("Write exceeds simulated mapping bounds.")
                tmp.seek(offset)
                tmp.write(data)
                tmp.flush()
                logger.info(f"Wrote {len(data)} bytes to simulated handle {addr} at offset {offset}")
                return True
        raise AssemblyError("Emulator not running and no suitable simulated mapping found.")

    def memory_read(self, addr: int, size: int, offset: int = 0) -> bytes:
        """
        Read from mapped memory. For simulated mappings, address is the handle returned by memory_alloc.
        """
        if UC_AVAILABLE and self.uc is not None and isinstance(addr, int) and addr in self.emu_mapped:
            return self.uc.mem_read(addr + offset, size)
        else:
            entry = self.emu_mapped.get(addr)
            if entry and isinstance(entry, tuple):
                tmp, sz = entry
                if offset < 0 or offset + size > sz:
                    raise AssemblyError("Read exceeds simulated mapping bounds.")
                tmp.seek(offset)
                return tmp.read(size)
        raise AssemblyError("Emulator not running and no suitable simulated mapping found.")

    # ---------------- Emulator (Unicorn) ----------------
    def emu_init(self):
        if not UC_AVAILABLE or Uc is None:
            raise AssemblyError("Unicorn engine not available. Install 'unicorn' package to emulate.")
        arch_map = {
            "x86": UC_ARCH_X86,
            "arm": UC_ARCH_ARM,
            "arm64": UC_ARCH_ARM64,
            "mips": UC_ARCH_MIPS,
        }
        mode_map = {
            16: UC_MODE_16,
            32: UC_MODE_32,
            64: UC_MODE_64,
            "arm": UC_MODE_ARM,
            "thumb": UC_MODE_THUMB,
        }
        uc_arch = arch_map.get(self.arch)
        if uc_arch is None:
            raise AssemblyError(f"Emulation: unsupported architecture {self.arch}")
        uc_mode = mode_map.get(self.mode, UC_MODE_64 if self.mode == 64 else UC_MODE_32)
        self.uc = Uc(uc_arch, uc_mode)
        self.emu_mapped = {}
        self.emu_hooks = []
        logger.info("Unicorn emulator initialized")

    def emu_map_and_write(self, addr: int, code: BytesLike):
        if isinstance(code, str):
            s = code.strip()
            if all(c in "0123456789abcdefABCDEF" for c in s) and s != "":
                code = bytes.fromhex(s)
            else:
                code = code.encode('latin1')
        if self.uc is None:
            raise AssemblyError("Unicorn emulator not initialized. Call emu_init() first.")
        size = len(code)
        page_size = 0x1000
        alloc_size = ((size + page_size - 1) // page_size) * page_size
        self.uc.mem_map(addr, alloc_size)
        self.uc.mem_write(addr, code)
        self.emu_mapped[addr] = alloc_size
        logger.info(f"Mapped and wrote {size} bytes to 0x{addr:X}")

    def emu_run(self, start_addr: int, size: int, timeout: int = 0, regs_init: Optional[Dict[str, int]] = None):
        """
        Run code in emulator. Returns dict with final registers snapshot.
        regs_init: mapping of architecture register names -> values
        timeout: microseconds
        """
        if not UC_AVAILABLE or self.uc is None:
            raise AssemblyError("Unicorn not available or not initialized.")
        if regs_init:
            self._emu_set_regs_from_map(regs_init)
        try:
            end_addr = start_addr + size
            # call with keywords first (modern unicorn), otherwise try positional
            try:
                if timeout:
                    self.uc.emu_start(start_addr, end_addr, timeout=timeout, count=0)
                else:
                    self.uc.emu_start(start_addr, end_addr)
            except TypeError:
                # try positional fallback (timeout, count)
                if timeout:
                    self.uc.emu_start(start_addr, end_addr, timeout, 0)
                else:
                    self.uc.emu_start(start_addr, end_addr)
        except Exception as e:
            logger.warning(f"Emulation stopped with exception: {e}")
        finally:
            return self.regs_snapshot()

    # ---------------- Registers (emulator) ----------------
    def _get_reg_const(self, name: str) -> Optional[int]:
        """Map common register name strings to Unicorn constants depending on arch."""
        if not isinstance(name, str) or name == "":
            return None
        n = name.lower()
        if self.arch == "x86" and uc_x86_const is not None:
            reg_map = {
                "rax": getattr(uc_x86_const, "UC_X86_REG_RAX", None),
                "rbx": getattr(uc_x86_const, "UC_X86_REG_RBX", None),
                "rcx": getattr(uc_x86_const, "UC_X86_REG_RCX", None),
                "rdx": getattr(uc_x86_const, "UC_X86_REG_RDX", None),
                "rsp": getattr(uc_x86_const, "UC_X86_REG_RSP", None),
                "rbp": getattr(uc_x86_const, "UC_X86_REG_RBP", None),
                "rsi": getattr(uc_x86_const, "UC_X86_REG_RSI", None),
                "rdi": getattr(uc_x86_const, "UC_X86_REG_RDI", None),
                "rip": getattr(uc_x86_const, "UC_X86_REG_RIP", None),
                "eax": getattr(uc_x86_const, "UC_X86_REG_EAX", None),
                "ebx": getattr(uc_x86_const, "UC_X86_REG_EBX", None),
                "ecx": getattr(uc_x86_const, "UC_X86_REG_ECX", None),
                "edx": getattr(uc_x86_const, "UC_X86_REG_EDX", None),
                "esp": getattr(uc_x86_const, "UC_X86_REG_ESP", None),
                "ebp": getattr(uc_x86_const, "UC_X86_REG_EBP", None),
                "esi": getattr(uc_x86_const, "UC_X86_REG_ESI", None),
                "edi": getattr(uc_x86_const, "UC_X86_REG_EDI", None),
            }
            return reg_map.get(n)
        elif self.arch == "arm" and uc_arm_const is not None:
            # r0..r12
            if n.startswith("r") and len(n) > 1 and n[1:].isdigit():
                idx = int(n[1:])
                return getattr(uc_arm_const, f"UC_ARM_REG_R{idx}", None)
            map_simple = {
                "sp": getattr(uc_arm_const, "UC_ARM_REG_SP", None),
                "lr": getattr(uc_arm_const, "UC_ARM_REG_LR", None),
                "pc": getattr(uc_arm_const, "UC_ARM_REG_PC", None),
            }
            if n in map_simple:
                return map_simple[n]
            try:
                return getattr(uc_arm_const, n.upper(), None)
            except Exception:
                return None
        elif self.arch == "arm64" and uc_arm64_const is not None:
            if n.startswith("x") and len(n) > 1 and n[1:].isdigit():
                idx = int(n[1:])
                return getattr(uc_arm64_const, f"UC_ARM64_REG_X{idx}", None)
            if n == "sp":
                return getattr(uc_arm64_const, "UC_ARM64_REG_SP", None)
            if n == "pc":
                return getattr(uc_arm64_const, "UC_ARM64_REG_PC", None)
            return None
        else:
            return None

    def _emu_set_regs_from_map(self, regs: Dict[str, int]):
        """Set multiple registers in the running emulator."""
        for name, val in regs.items():
            rc = self._get_reg_const(name)
            if rc is None:
                logger.warning(f"Unknown register '{name}' for arch {self.arch}")
                continue
            self.uc.reg_write(rc, val)

    def regs_snapshot(self) -> Dict[str, int]:
        """Return a snapshot of commonly useful registers depending on arch."""
        snapshot = {}
        if not UC_AVAILABLE or self.uc is None:
            raise AssemblyError("Unicorn emulator not available/running.")
        if self.arch == "x86" and uc_x86_const is not None:
            regs = ["RAX", "RBX", "RCX", "RDX", "RSP", "RBP", "RSI", "RDI", "RIP"]
            for r in regs:
                const = getattr(uc_x86_const, f"UC_X86_REG_{r}", None)
                if const is not None:
                    snapshot[r.lower()] = self.uc.reg_read(const)
        elif self.arch == "arm" and uc_arm_const is not None:
            for i in range(13):
                const = getattr(uc_arm_const, f"UC_ARM_REG_R{i}", None)
                if const is not None:
                    snapshot[f"r{i}"] = self.uc.reg_read(const)
            if getattr(uc_arm_const, "UC_ARM_REG_SP", None):
                snapshot["sp"] = self.uc.reg_read(uc_arm_const.UC_ARM_REG_SP)
            if getattr(uc_arm_const, "UC_ARM_REG_LR", None):
                snapshot["lr"] = self.uc.reg_read(uc_arm_const.UC_ARM_REG_LR)
            if getattr(uc_arm_const, "UC_ARM_REG_PC", None):
                snapshot["pc"] = self.uc.reg_read(uc_arm_const.UC_ARM_REG_PC)
        elif self.arch == "arm64" and uc_arm64_const is not None:
            for i in range(31):
                const = getattr(uc_arm64_const, f"UC_ARM64_REG_X{i}", None)
                if const is not None:
                    snapshot[f"x{i}"] = self.uc.reg_read(const)
            if getattr(uc_arm64_const, "UC_ARM64_REG_SP", None):
                snapshot["sp"] = self.uc.reg_read(uc_arm64_const.UC_ARM64_REG_SP)
            if getattr(uc_arm64_const, "UC_ARM64_REG_PC", None):
                snapshot["pc"] = self.uc.reg_read(uc_arm64_const.UC_ARM64_REG_PC)
        else:
            logger.warning("regs_snapshot: limited register set for arch " + self.arch)
        return snapshot

    def reg_read(self, name: str) -> int:
        """Read a single register value using emulator."""
        if not UC_AVAILABLE or self.uc is None:
            raise AssemblyError("Unicorn emulator not available/running.")
        rc = self._get_reg_const(name)
        if rc is None:
            raise AssemblyError(f"Unknown register '{name}' for arch {self.arch}")
        return self.uc.reg_read(rc)

    def reg_write(self, name: str, value: int):
        """Write a single register value using emulator."""
        if not UC_AVAILABLE or self.uc is None:
            raise AssemblyError("Unicorn emulator not available/running.")
        rc = self._get_reg_const(name)
        if rc is None:
            raise AssemblyError(f"Unknown register '{name}' for arch {self.arch}")
        self.uc.reg_write(rc, value)
        return True

    # ---------------- Hardware IO (simulated where not allowed) ----------------
    def write_port(self, port: int, value: int):
        """Simulated port I/O. Real port I/O needs platform-specific drivers and privileges."""
        logger.info(f"Simulated write to port 0x{port:X} with value 0x{value:X}")
        return True

    def read_port(self, port: int) -> int:
        logger.info(f"Simulated read from port 0x{port:X} -> returning 0")
        return 0

    # ---------------- Utilities ----------------
    @staticmethod
    def hex_dump(byte_data: BytesLike, width: int = 16) -> str:
        if isinstance(byte_data, str):
            s = byte_data.strip()
            if s == "":
                data = b""
            elif all(c in "0123456789abcdefABCDEF" for c in s):
                data = bytes.fromhex(s)
            else:
                data = s.encode('latin1')
        else:
            data = bytes(byte_data)
        lines = []
        for i in range(0, len(data), width):
            chunk = data[i:i + width]
            hex_chunk = ' '.join(f"{b:02X}" for b in chunk)
            ascii_chunk = ''.join(chr(b) if 32 <= b <= 126 else '.' for b in chunk)
            lines.append(f"{i:08X}: {hex_chunk:<{width*3}} {ascii_chunk}")
        return "\n".join(lines)

    def save_binary(self, filename: str, data: BytesLike):
        if isinstance(data, str):
            s = data.strip()
            if s != "" and all(c in "0123456789abcdefABCDEF" for c in s):
                data = bytes.fromhex(s)
            else:
                data = data.encode('latin1')
        with open(filename, "wb") as f:
            f.write(data)
        logger.info(f"Wrote {len(data)} bytes to {filename}")
        return filename

    def load_binary(self, filename: str) -> bytes:
        with open(filename, "rb") as f:
            data = f.read()
        logger.info(f"Loaded {len(data)} bytes from {filename}")
        return data

    # ---------------- High-level helper ----------------
    def emulate_code(self, asm_code: str, start_addr: int = 0x1000, regs_init: Optional[Dict[str, int]] = None) -> Dict[str, int]:
        """
        Assemble and emulate a small code snippet and return register snapshot.
        """
        code_bytes, cnt = self.assemble(asm_code, addr=start_addr, as_hex=False)
        if isinstance(code_bytes, str):
            code_bytes = bytes.fromhex(code_bytes)
        if not UC_AVAILABLE:
            raise AssemblyError("Unicorn not available. Cannot emulate code.")
        if self.uc is None:
            self.emu_init()
        self.emu_map_and_write(start_addr, code_bytes)
        snapshot = self.emu_run(start_addr, len(code_bytes), timeout=0, regs_init=regs_init)
        return snapshot

    