# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import os
import subprocess
import sys
try:
    # Needed for Python < 3.3, works up to 3.8
    import xml.etree.cElementTree as etree
except ImportError:
    # Python 3.9 onwards
    import xml.etree.ElementTree as etree
from subprocess import Popen
from textwrap import dedent

import mock

import six
from diff_cover.violationsreporters import base

from diff_cover.command_runner import CommandError, run_command_for_code
import unittest
from diff_cover.violationsreporters.base import QualityReporter
from diff_cover.violationsreporters.violations_reporter import (
    XmlCoverageReporter,
    Violation,
    pycodestyle_driver,
    pyflakes_driver,
    flake8_driver,
    PylintDriver,
    jshint_driver,
    eslint_driver,
    pydocstyle_driver,
    CppcheckDriver,
)
from mock import Mock, patch, MagicMock
from six import BytesIO, StringIO


def _patch_so_all_files_exist():
    _mock_exists = patch.object(base.os.path, 'exists').start()
    _mock_exists.returnvalue = True

def _setup_patch(return_value, status_code=0):
    mocked_process = mock.Mock()
    mocked_process.returncode = status_code
    mocked_process.communicate.return_value = return_value
    mocked_subprocess = mock.patch('diff_cover.command_runner.subprocess').start()
    mocked_subprocess.Popen.return_value = mocked_process
    return mocked_process


class XmlCoverageReporterTest(unittest.TestCase):

    MANY_VIOLATIONS = {Violation(3, None), Violation(7, None),
                           Violation(11, None), Violation(13, None)}
    FEW_MEASURED = {2, 3, 5, 7, 11, 13}

    FEW_VIOLATIONS = {Violation(3, None), Violation(11, None)}
    MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17}

    ONE_VIOLATION = {Violation(11, None)}
    VERY_MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17, 23, 24, 25, 26, 26, 27}

    def setUp(self):
        # Paths generated by git_path are always the given argument
        _git_path_mock = patch('diff_cover.violationsreporters.violations_reporter.GitPathTool').start()
        _git_path_mock.relative_path = lambda path: path
        _git_path_mock.absolute_path = lambda path: path
        self.addCleanup(patch.stopall)

    def test_violations(self):

        # Construct the XML report
        file_paths = ['file1.py', 'subdir/file2.py']
        violations = self.MANY_VIOLATIONS
        measured = self.FEW_MEASURED
        xml = self._coverage_xml(file_paths, violations, measured)

        # Parse the report
        coverage = XmlCoverageReporter(xml)

        # Expect that the name is set
        self.assertEqual(coverage.name(), "XML")

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(violations, coverage.violations('file1.py'))
        self.assertEqual(measured, coverage.measured_lines('file1.py'))

        # Try getting a smaller range
        result = coverage.violations('subdir/file2.py')
        self.assertEqual(result, violations)

        # Once more on the first file (for caching)
        result = coverage.violations('file1.py')
        self.assertEqual(result, violations)

    def test_non_python_violations(self):
        """
        Non python projects often just have a file name specified while
         the full path can be acquired from a sources tag in the XML.

         This test checks that flow by requesting violation info from a path
         that can only be constructed by using the path provided in the sources
         tag
        """
        fancy_path = 'superFancyPath'
        file_paths = ['file1.java']
        source_paths = [fancy_path]
        violations = self.MANY_VIOLATIONS
        measured = self.FEW_MEASURED

        xml = self._coverage_xml(
            file_paths,
            violations,
            measured,
            source_paths=source_paths
        )
        coverage = XmlCoverageReporter([xml])

        self.assertEqual(
            violations,
            coverage.violations(
                '{}/{}'.format(fancy_path, file_paths[0])
            )
        )

        self.assertEqual(
            measured,
            coverage.measured_lines(
                '{}/{}'.format(fancy_path, file_paths[0])
            )
        )

    def test_non_python_violations_empty_path(self):
        """
        In the wild empty sources can happen. See https://github.com/Bachmann1234/diff-cover/issues/88
        Best I can tell its mostly irrelevant but I mostly dont want it crashing
        """

        xml = etree.fromstring('''
        <coverage line-rate="0.178" branch-rate="0.348" version="1.9" timestamp="1545037553" lines-covered="675" lines-valid="3787" branches-covered="260" branches-valid="747">
        <sources>
        <source></source>
        </sources>
        </coverage>
        ''')

        coverage = XmlCoverageReporter([xml])

        self.assertEqual(
            set(),
            coverage.violations('')
        )

        self.assertEqual(
            set(),
            coverage.measured_lines('')
        )

    def test_two_inputs_first_violate(self):

        # Construct the XML report
        file_paths = ['file1.py']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml, xml2])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.py')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.py')
        )

    def test_two_inputs_second_violate(self):

        # Construct the XML report
        file_paths = ['file1.py']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.py')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.py')
        )

    def test_three_inputs(self):

        # Construct the XML report
        file_paths = ['file1.py']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS
        violations3 = self.ONE_VIOLATION

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED
        measured3 = self.VERY_MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)
        xml3 = self._coverage_xml(file_paths, violations3, measured3)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml, xml3])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2 & violations3,
            coverage.violations('file1.py')
        )

        self.assertEqual(
            measured1 | measured2 | measured3,
            coverage.measured_lines('file1.py')
        )

    def test_different_files_in_inputs(self):

        # Construct the XML report
        xml_roots = [
            self._coverage_xml(['file.py'], self.MANY_VIOLATIONS, self.FEW_MEASURED),
            self._coverage_xml(['other_file.py'], self.FEW_VIOLATIONS, self.MANY_MEASURED)
        ]

        # Parse the report
        coverage = XmlCoverageReporter(xml_roots)

        self.assertEqual(self.MANY_VIOLATIONS, coverage.violations('file.py'))
        self.assertEqual(self.FEW_VIOLATIONS, coverage.violations('other_file.py'))

    def test_empty_violations(self):
        """
        Test that an empty violations report is handled properly
        """

        # Construct the XML report
        file_paths = ['file1.py']

        violations1 = self.MANY_VIOLATIONS
        violations2 = set()

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.py')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.py')
        )

    def test_no_such_file(self):

        # Construct the XML report with no source files
        xml = self._coverage_xml([], [], [])

        # Parse the report
        coverage = XmlCoverageReporter(xml)

        # Expect that we get no results
        result = coverage.violations('file.py')
        self.assertEqual(result, set())

    def _coverage_xml(
            self,
            file_paths,
            violations,
            measured,
            source_paths=None
    ):
        """
        Build an XML tree with source files specified by `file_paths`.
        Each source fill will have the same set of covered and
        uncovered lines.

        `file_paths` is a list of path strings
        `line_dict` is a dictionary with keys that are line numbers
        and values that are True/False indicating whether the line
        is covered

        This leaves out some attributes of the Cobertura format,
        but includes all the elements.
        """
        root = etree.Element('coverage')
        if source_paths:
            sources = etree.SubElement(root, 'sources')
            for path in source_paths:
                source = etree.SubElement(sources, 'source')
                source.text = path

        packages = etree.SubElement(root, 'packages')
        classes = etree.SubElement(packages, 'classes')

        violation_lines = {violation.line for violation in violations}

        for path in file_paths:

            src_node = etree.SubElement(classes, 'class')
            src_node.set('filename', path)

            etree.SubElement(src_node, 'methods')
            lines_node = etree.SubElement(src_node, 'lines')

            # Create a node for each line in measured
            for line_num in measured:
                is_covered = line_num not in violation_lines
                line = etree.SubElement(lines_node, 'line')

                hits = 1 if is_covered else 0
                line.set('hits', str(hits))
                line.set('number', str(line_num))

        return root

    def test_to_unix_path(self):
        """
        Ensure the _to_unix_path static function handles paths properly.
        """
        to_unix_path = XmlCoverageReporter._to_unix_path
        self.assertEqual(to_unix_path('foo/bar'), 'foo/bar')
        self.assertEqual(to_unix_path('foo\\bar'), 'foo/bar')
        if sys.platform.startswith('win'):
            self.assertEqual(to_unix_path('FOO\\bar'), 'foo/bar')


class CloverXmlCoverageReporterTest(unittest.TestCase):

    MANY_VIOLATIONS = {Violation(3, None), Violation(7, None),
                           Violation(11, None), Violation(13, None)}
    FEW_MEASURED = {2, 3, 5, 7, 11, 13}

    FEW_VIOLATIONS = {Violation(3, None), Violation(11, None)}
    MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17}

    ONE_VIOLATION = {Violation(11, None)}
    VERY_MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17, 23, 24, 25, 26, 26, 27}

    def setUp(self):
        # Paths generated by git_path are always the given argument
        _git_path_mock = patch('diff_cover.violationsreporters.violations_reporter.GitPathTool').start()
        _git_path_mock.relative_path = lambda path: path
        _git_path_mock.absolute_path = lambda path: path

    def tearDown(self):
        patch.stopall()

    def test_violations(self):

        # Construct the XML report
        file_paths = ['file1.java', 'subdir/file2.java']
        violations = self.MANY_VIOLATIONS
        measured = self.FEW_MEASURED
        xml = self._coverage_xml(file_paths, violations, measured)

        # Parse the report
        coverage = XmlCoverageReporter([xml])

        # Expect that the name is set
        self.assertEqual(coverage.name(), "XML")

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(violations, coverage.violations('file1.java'))
        self.assertEqual(measured, coverage.measured_lines('file1.java'))

        # Try getting a smaller range
        result = coverage.violations('subdir/file2.java')
        self.assertEqual(result, violations)

        # Once more on the first file (for caching)
        result = coverage.violations('file1.java')
        self.assertEqual(result, violations)

    def test_two_inputs_first_violate(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml, xml2])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_two_inputs_second_violate(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_three_inputs(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS
        violations3 = self.ONE_VIOLATION

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED
        measured3 = self.VERY_MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)
        xml3 = self._coverage_xml(file_paths, violations3, measured3)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml, xml3])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2 & violations3,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2 | measured3,
            coverage.measured_lines('file1.java')
        )

    def test_different_files_in_inputs(self):

        # Construct the XML report
        xml_roots = [
            self._coverage_xml(['file.java'], self.MANY_VIOLATIONS, self.FEW_MEASURED),
            self._coverage_xml(['other_file.java'], self.FEW_VIOLATIONS, self.MANY_MEASURED)
        ]

        # Parse the report
        coverage = XmlCoverageReporter(xml_roots)

        self.assertEqual(self.MANY_VIOLATIONS, coverage.violations('file.java'))
        self.assertEqual(self.FEW_VIOLATIONS, coverage.violations('other_file.java'))

    def test_empty_violations(self):
        """
        Test that an empty violations report is handled properly
        """

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = set()

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_no_such_file(self):

        # Construct the XML report with no source files
        xml = self._coverage_xml([], [], [])

        # Parse the report
        coverage = XmlCoverageReporter(xml)

        # Expect that we get no results
        result = coverage.violations('file.java')
        self.assertEqual(result, set())

    def _coverage_xml(
            self,
            file_paths,
            violations,
            measured
    ):
        """
        Build an XML tree with source files specified by `file_paths`.
        Each source fill will have the same set of covered and
        uncovered lines.

        `file_paths` is a list of path strings
        `line_dict` is a dictionary with keys that are line numbers
        and values that are True/False indicating whether the line
        is covered

        This leaves out some attributes of the Cobertura format,
        but includes all the elements.
        """
        root = etree.Element('coverage')
        root.set('clover', '4.2.0')
        project = etree.SubElement(root, 'project')
        package = etree.SubElement(project, 'package')

        violation_lines = {violation.line for violation in violations}

        for path in file_paths:

            src_node = etree.SubElement(package, 'file')
            src_node.set('path', path)

            # Create a node for each line in measured
            for line_num in measured:
                is_covered = line_num not in violation_lines
                line = etree.SubElement(src_node, 'line')

                hits = 1 if is_covered else 0
                line.set('count', str(hits))
                line.set('num', str(line_num))
                line.set('type', 'stmt')
        return root


class JacocoXmlCoverageReporterTest(unittest.TestCase):

    MANY_VIOLATIONS = {Violation(3, None), Violation(7, None),
                           Violation(11, None), Violation(13, None)}
    FEW_MEASURED = {2, 3, 5, 7, 11, 13}

    FEW_VIOLATIONS = {Violation(3, None), Violation(11, None)}
    MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17}

    ONE_VIOLATION = {Violation(11, None)}
    VERY_MANY_MEASURED = {2, 3, 5, 7, 11, 13, 17, 23, 24, 25, 26, 26, 27}

    def setUp(self):
        # Paths generated by git_path are always the given argument
        _git_path_mock = patch('diff_cover.violationsreporters.violations_reporter.GitPathTool').start()
        _git_path_mock.relative_path = lambda path: path
        _git_path_mock.absolute_path = lambda path: path

    def tearDown(self):
        patch.stopall()

    def test_violations(self):

        # Construct the XML report
        file_paths = ['file1.java', 'subdir/file2.java']
        violations = self.MANY_VIOLATIONS
        measured = self.FEW_MEASURED
        xml = self._coverage_xml(file_paths, violations, measured)

        # Parse the report
        coverage = XmlCoverageReporter([xml])

        # Expect that the name is set
        self.assertEqual(coverage.name(), "XML")

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(violations, coverage.violations('file1.java'))
        self.assertEqual(measured, coverage.measured_lines('file1.java'))

        # Try getting a smaller range
        result = coverage.violations('subdir/file2.java')
        self.assertEqual(result, violations)

        # Once more on the first file (for caching)
        result = coverage.violations('file1.java')
        self.assertEqual(result, violations)

    def test_two_inputs_first_violate(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml, xml2])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_two_inputs_second_violate(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_three_inputs(self):

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = self.FEW_VIOLATIONS
        violations3 = self.ONE_VIOLATION

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED
        measured3 = self.VERY_MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)
        xml3 = self._coverage_xml(file_paths, violations3, measured3)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml, xml3])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2 & violations3,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2 | measured3,
            coverage.measured_lines('file1.java')
        )

    def test_different_files_in_inputs(self):

        # Construct the XML report
        xml_roots = [
            self._coverage_xml(['file.java'], self.MANY_VIOLATIONS, self.FEW_MEASURED),
            self._coverage_xml(['other_file.java'], self.FEW_VIOLATIONS, self.MANY_MEASURED)
        ]

        # Parse the report
        coverage = XmlCoverageReporter(xml_roots)

        self.assertEqual(self.MANY_VIOLATIONS, coverage.violations('file.java'))
        self.assertEqual(self.FEW_VIOLATIONS, coverage.violations('other_file.java'))

    def test_empty_violations(self):
        """
        Test that an empty violations report is handled properly
        """

        # Construct the XML report
        file_paths = ['file1.java']

        violations1 = self.MANY_VIOLATIONS
        violations2 = set()

        measured1 = self.FEW_MEASURED
        measured2 = self.MANY_MEASURED

        xml = self._coverage_xml(file_paths, violations1, measured1)
        xml2 = self._coverage_xml(file_paths, violations2, measured2)

        # Parse the report
        coverage = XmlCoverageReporter([xml2, xml])

        # By construction, each file has the same set
        # of covered/uncovered lines
        self.assertEqual(
            violations1 & violations2,
            coverage.violations('file1.java')
        )

        self.assertEqual(
            measured1 | measured2,
            coverage.measured_lines('file1.java')
        )

    def test_no_such_file(self):

        # Construct the XML report with no source files
        xml = self._coverage_xml([], [], [])

        # Parse the report
        coverage = XmlCoverageReporter(xml)

        # Expect that we get no results
        result = coverage.violations('file.java')
        self.assertEqual(result, set())

    def _coverage_xml(
            self,
            file_paths,
            violations,
            measured
    ):
        """
        Build an XML tree with source files specified by `file_paths`.
        Each source fill will have the same set of covered and
        uncovered lines.

        `file_paths` is a list of path strings
        `line_dict` is a dictionary with keys that are line numbers
        and values that are True/False indicating whether the line
        is covered

        This leaves out some attributes of the Cobertura format,
        but includes all the elements.
        """
        root = etree.Element('report')
        root.set('name', 'diff-cover')
        sessioninfo = etree.SubElement(root, 'sessioninfo')
        sessioninfo.set('id', 'C13WQ1WFHTEE-83e2bc9b')


        violation_lines = {violation.line for violation in violations}

        for path in file_paths:

            package = etree.SubElement(root, 'package')
            package.set('name', os.path.dirname(path))
            src_node = etree.SubElement(package, 'sourcefile')
            src_node.set('name', os.path.basename(path))

            # Create a node for each line in measured
            for line_num in measured:
                is_covered = line_num not in violation_lines
                line = etree.SubElement(src_node, 'line')

                hits = 1 if is_covered else 0
                line.set('ci', str(hits))
                line.set('nr', str(line_num))
        return root


class pycodestyleQualityReporterTest(unittest.TestCase):

    def setUp(self):
        _patch_so_all_files_exist()

    def tearDown(self):
        """
        Undo all patches
        """
        patch.stopall()

    def test_quality(self):

        # Patch the output of `pycodestyle`
        _mock_communicate = patch.object(Popen, 'communicate').start()
        return_string = '\n' + dedent("""
                ../new_file.py:1:17: E231 whitespace
                ../new_file.py:3:13: E225 whitespace
                ../new_file.py:7:1: E302 blank lines
            """).strip() + '\n'
        _setup_patch((return_string.encode('utf-8'), b''))

        # Parse the report
        quality = QualityReporter(pycodestyle_driver)

        # Expect that the name is set
        self.assertEqual(quality.name(), 'pycodestyle')

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('../new_file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, 'E231 whitespace'),
            Violation(3, 'E225 whitespace'),
            Violation(7, 'E302 blank lines')
        ]

        self.assertEqual(expected_violations, quality.violations('../new_file.py'))

    def test_no_quality_issues_newline(self):

        # Patch the output of `pycodestyle`
        _setup_patch((b'\n', b''))

        # Parse the report
        quality = QualityReporter(pycodestyle_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_no_quality_issues_emptystring(self):

        # Patch the output of `pycodestyle`
        _setup_patch((b'', b''))

        # Parse the report
        quality = QualityReporter(pycodestyle_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_quality_error(self):

        # Patch the output of `pycodestyle`
        _setup_patch((b"", 'whoops Ƕئ'.encode('utf-8')), status_code=255)
        with patch('diff_cover.violationsreporters.base.run_command_for_code') as code:
            code.return_value = 0
            # Parse the report
            quality = QualityReporter(pycodestyle_driver)

            # Expect that the name is set
            self.assertEqual(quality.name(), 'pycodestyle')
            with self.assertRaises(CommandError) as ex:
                quality.violations('file1.py')
            self.assertEqual(six.text_type(ex.exception), 'whoops Ƕئ')

    def test_no_such_file(self):
        quality = QualityReporter(pycodestyle_driver)

        # Expect that we get no results
        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_python_file(self):
        quality = QualityReporter(pycodestyle_driver)
        file_paths = ['file1.coffee', 'subdir/file2.js']
        # Expect that we get no results because no Python files
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality_pregenerated_report(self):

        # When the user provides us with a pre-generated pycodestyle report
        # then use that instead of calling pycodestyle directly.
        pycodestyle_reports = [
            BytesIO(('\n' + dedent("""
                path/to/file.py:1:17: E231 whitespace
                path/to/file.py:3:13: E225 whitespace
                another/file.py:7:1: E302 blank lines
            """).strip() + '\n').encode('utf-8')),

            BytesIO(('\n' + dedent("""
                path/to/file.py:24:2: W123 \u9134\u1912
                another/file.py:50:1: E302 blank lines
            """).strip() + '\n').encode('utf-8')),
        ]

        # Parse the report
        quality = QualityReporter(pycodestyle_driver, reports=pycodestyle_reports)

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('path/to/file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, 'E231 whitespace'),
            Violation(3, 'E225 whitespace'),
            Violation(24, 'W123 \u9134\u1912')
        ]

        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('path/to/file.py')

        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)


class PyflakesQualityReporterTest(unittest.TestCase):
    """
    Tests for Pyflakes quality violations
    """

    def tearDown(self):
        """
        Undo all patches
        """
        patch.stopall()

    def test_quality(self):
        _patch_so_all_files_exist()

        # Patch the output of `pyflakes`
        return_string = '\n' + dedent("""
                ../new_file.py:328: undefined name '_thing'
                ../new_file.py:418: 'random' imported but unused
            """).strip() + '\n'
        _setup_patch((return_string.encode('utf-8'), b''))

        # Parse the report
        quality = QualityReporter(pyflakes_driver)

        # Expect that the name is set
        self.assertEqual(quality.name(), 'pyflakes')

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('../new_file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(328, "undefined name '_thing'"),
            Violation(418, "'random' imported but unused")
        ]

        self.assertEqual(
            expected_violations,
            quality.violations('../new_file.py'))

    def test_no_quality_issues_newline(self):

        _setup_patch((b'\n', b''))
        # Parse the report
        quality = QualityReporter(pyflakes_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_no_quality_issues_emptystring(self):

        # Patch the output of `pyflakes`
        _setup_patch((b'', b''))
        # Parse the report
        quality = QualityReporter(pyflakes_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_quality_error(self):
        _patch_so_all_files_exist()

        # Patch the output of `pyflakes`
        _setup_patch((b"", b'whoops'), status_code=255)

        with patch('diff_cover.violationsreporters.base.run_command_for_code') as code:
            code.return_value = 0
            quality = QualityReporter(pyflakes_driver)

            # Expect that the name is set
            self.assertEqual(quality.name(), 'pyflakes')

            self.assertRaises(CommandError, quality.violations, 'file1.py')

    def test_no_such_file(self):
        quality = QualityReporter(pyflakes_driver)

        # Expect that we get no results
        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_python_file(self):
        quality = QualityReporter(pyflakes_driver)
        file_paths = ['file1.coffee', 'subdir/file2.js']
        # Expect that we get no results because no Python files
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality_pregenerated_report(self):

        # When the user provides us with a pre-generated pyflakes report
        # then use that instead of calling pyflakes directly.
        pyflakes_reports = [
            BytesIO(('\n' + dedent("""
                path/to/file.py:1: undefined name 'this'
                path/to/file.py:3: 'random' imported but unused
                another/file.py:7: 'os' imported but unused
            """).strip() + '\n').encode('utf-8')),

            BytesIO(('\n' + dedent("""
                path/to/file.py:24: undefined name 'that'
                another/file.py:50: undefined name 'another'
            """).strip() + '\n').encode('utf-8')),
        ]

        # Parse the report
        quality = QualityReporter(pyflakes_driver, reports=pyflakes_reports)

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('path/to/file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, "undefined name 'this'"),
            Violation(3, "'random' imported but unused"),
            Violation(24, "undefined name 'that'")
        ]

        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('path/to/file.py')

        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)


class Flake8QualityReporterTest(unittest.TestCase):

    def tearDown(self):
        """
        Undo all patches
        """
        patch.stopall()

    def test_quality(self):
        _patch_so_all_files_exist()

        # Patch the output of `flake8`
        return_string = '\n' + dedent("""
                ../new_file.py:1:17: E231 whitespace
                ../new_file.py:3:13: E225 whitespace
                ../new_file.py:7:1: E302 blank lines
                ../new_file.py:8:1: W191 indentation contains tabs
                ../new_file.py:10:1: F841 local variable name is assigned to but never used
                ../new_file.py:20:1: C901 'MyModel.mymethod' is too complex (14)
                ../new_file.py:50:1: N801 class names should use CapWords convention
                ../new_file.py:60:10: T000 Todo note found.
                ../new_file.py:70:0: I100 statements are in the wrong order.
                ../new_file.py:80:0: B901 blind except: statement
                ../new_file.py:90:0: D207 Docstring is under-indented
                ../new_file.py:100:0: S100 Snippet found
                ../new_file.py:110:0: Q000 Remove Single quotes
            """).strip() + '\n'
        _setup_patch((return_string.encode('utf-8'), b''))

        # Parse the report
        quality = QualityReporter(flake8_driver)

        # Expect that the name is set
        self.assertEqual(quality.name(), 'flake8')

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('../new_file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, 'E231 whitespace'),
            Violation(3, 'E225 whitespace'),
            Violation(7, 'E302 blank lines'),
            Violation(8, 'W191 indentation contains tabs'),
            Violation(10, 'F841 local variable name is assigned to but never used'),
            Violation(20, "C901 'MyModel.mymethod' is too complex (14)"),
            Violation(50, 'N801 class names should use CapWords convention'),
            Violation(60, 'T000 Todo note found.'),
            Violation(70, 'I100 statements are in the wrong order.'),
            Violation(80, 'B901 blind except: statement'),
            Violation(90, 'D207 Docstring is under-indented'),
            Violation(100, 'S100 Snippet found'),
            Violation(110, 'Q000 Remove Single quotes'),
        ]

        self.assertEqual(expected_violations, quality.violations('../new_file.py'))

    def test_no_quality_issues_newline(self):

        _setup_patch((b'\n', b''), 0)

        quality = QualityReporter(flake8_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_no_quality_issues_emptystring(self):

        # Patch the output of `flake8`
        _setup_patch((b'', b''), 0)

        # Parse the report
        quality = QualityReporter(flake8_driver)
        self.assertEqual([], quality.violations('file1.py'))

    def test_quality_error(self):
        _patch_so_all_files_exist()

        # Patch the output of `flake8`
        _setup_patch((b"", 'whoops Ƕئ'.encode('utf-8')), status_code=255)

        # Parse the report
        with patch('diff_cover.violationsreporters.base.run_command_for_code') as code:
            code.return_value = 0
            quality = QualityReporter(flake8_driver)

            # Expect that the name is set
            self.assertEqual(quality.name(), 'flake8')
            with self.assertRaises(CommandError) as ex:
                quality.violations('file1.py')
            self.assertEqual(six.text_type(ex.exception), 'whoops Ƕئ')

    def test_no_such_file(self):
        quality = QualityReporter(flake8_driver)

        # Expect that we get no results
        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_python_file(self):
        quality = QualityReporter(flake8_driver)
        file_paths = ['file1.coffee', 'subdir/file2.js']
        # Expect that we get no results because no Python files
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_file_does_not_exist(self):
        quality = QualityReporter(flake8_driver)
        file_paths = ['ajshdjlasdhajksdh.py']
        # Expect that we get no results because that file does not exist
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality_pregenerated_report(self):

        # When the user provides us with a pre-generated flake8 report
        # then use that instead of calling flake8 directly.
        flake8_reports = [
            BytesIO(('\n' + dedent("""
                path/to/file.py:1:17: E231 whitespace
                path/to/file.py:3:13: E225 whitespace
                another/file.py:7:1: E302 blank lines
            """).strip() + '\n').encode('utf-8')),

            BytesIO(('\n' + dedent("""
                path/to/file.py:24:2: W123 \u9134\u1912
                another/file.py:50:1: E302 blank lines
            """).strip() + '\n').encode('utf-8')),
        ]

        # Parse the report
        quality = QualityReporter(flake8_driver, reports=flake8_reports)

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('path/to/file.py'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, 'E231 whitespace'),
            Violation(3, 'E225 whitespace'),
            Violation(24, 'W123 \u9134\u1912')
        ]

        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('path/to/file.py')

        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)


class PydocstlyeQualityReporterTest(unittest.TestCase):
    """Tests for pydocstyle quality violations."""

    def setUp(self):
        """Set up required files."""
        _patch_so_all_files_exist()

    def tearDown(self):
        """Undo all patches."""
        patch.stopall()

    def test_no_such_file(self):
        """Expect that we get no results."""
        quality = QualityReporter(pydocstyle_driver)

        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_python_file(self):
        """Expect that we get no results because no Python files."""
        quality = QualityReporter(pydocstyle_driver)
        file_paths = ['file1.coffee', 'subdir/file2.js']
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality(self):
        """Integration test."""
        # Patch the output of `pydocstye`
        _setup_patch((
            dedent("""
            ../new_file.py:1 at module level:
                    D100: Missing docstring in public module
            ../new_file.py:13 in public function `gather`:
                    D103: Missing docstring in public function
            """).strip().encode('ascii'), ''
        ))

        expected_violations = [
            Violation(1, 'D100: Missing docstring in public module'),
            Violation(13, "D103: Missing docstring in public function"),
        ]

        # Parse the report
        quality = QualityReporter(pydocstyle_driver)

        # Expect that the name is set
        self.assertEqual(quality.name(), 'pydocstyle')

        # Measured_lines is undefined for a
        # quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('../new_file.py'))

        # Expect that we get violations for file1.py only
        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('../new_file.py')
        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)

class PylintQualityReporterTest(unittest.TestCase):

    def setUp(self):
        _patch_so_all_files_exist()

    def tearDown(self):
        """
        Undo all patches.
        """
        patch.stopall()

    def test_no_such_file(self):
        quality = QualityReporter(PylintDriver())

        # Expect that we get no results
        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_python_file(self):
        quality = QualityReporter(PylintDriver())
        file_paths = ['file1.coffee', 'subdir/file2.js']
        # Expect that we get no results because no Python files
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality(self):
        # Patch the output of `pylint`
        _setup_patch((
            dedent("""
            file1.py:1: [C0111] Missing docstring
            file1.py:1: [C0111, func_1] Missing docstring
            file1.py:2: [W0612, cls_name.func] Unused variable 'd'
            file1.py:2: [W0511] TODO: Not the real way we'll store usages!
            file1.py:579: [F0401] Unable to import 'rooted_paths'
            file1.py:113: [W0613, cache_relation.clear_pk] Unused argument 'cls'
            file1.py:150: [F0010] error while code parsing ([Errno 2] No such file or directory)
            file1.py:149: [C0324, Foo.__dict__] Comma not followed by a space
                self.peer_grading._find_corresponding_module_for_location(Location('i4x','a','b','c','d'))
            file1.py:162: [R0801] Similar lines in 2 files
            ==file1:162
            ==student.views:4
            import json
            import logging
            import random
            path/to/file2.py:100: [W0212, openid_login_complete] Access to a protected member
            """).strip().encode('ascii'), ''
        ))

        expected_violations = [
            Violation(1, 'C0111: Missing docstring'),
            Violation(1, 'C0111: func_1: Missing docstring'),
            Violation(2, "W0612: cls_name.func: Unused variable 'd'"),
            Violation(2, "W0511: TODO: Not the real way we'll store usages!"),
            Violation(579, "F0401: Unable to import 'rooted_paths'"),
            Violation(150, "F0010: error while code parsing ([Errno 2] No such file or directory)"),
            Violation(149, "C0324: Foo.__dict__: Comma not followed by a space"),
            Violation(162, "R0801: Similar lines in 2 files"),
            Violation(113, "W0613: cache_relation.clear_pk: Unused argument 'cls'")
        ]

        # Parse the report
        quality = QualityReporter(PylintDriver())

        # Expect that the name is set
        self.assertEqual(quality.name(), 'pylint')

        # Measured_lines is undefined for a
        # quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('file1.py'))

        # Expect that we get violations for file1.py only
        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('file1.py')
        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)

    def test_unicode(self):
        _setup_patch(
            (dedent("""
            file_\u6729.py:616: [W1401] Anomalous backslash in string: '\u5922'. String constant might be missing an r prefix.
            file.py:2: [W0612, cls_name.func_\u9492] Unused variable '\u2920'
            """).encode('utf-8'),
             ''),
            0
        )
        quality = QualityReporter(PylintDriver())
        violations = quality.violations('file_\u6729.py')
        self.assertEqual(violations, [
            Violation(616, "W1401: Anomalous backslash in string: '\u5922'. String constant might be missing an r prefix."),
        ])

        violations = quality.violations('file.py')
        self.assertEqual(violations, [Violation(2, "W0612: cls_name.func_\u9492: Unused variable '\u2920'")])

    def test_unicode_continuation_char(self):
        _setup_patch(
            (b"file.py:2: [W1401]"b" Invalid char '\xc3'", ''),
            0
        )
        # Since we are replacing characters we can't interpet, this should
        # return a valid string with the char replaced with '?'
        quality = QualityReporter(PylintDriver())
        violations = quality.violations('file.py')
        self.assertEqual(violations, [Violation(2, "W1401: Invalid char '\ufffd'")])

    def test_non_integer_line_num(self):
        _setup_patch((dedent("""
            file.py:not_a_number: C0111: Missing docstring
            file.py:\u8911: C0111: Missing docstring
        """).encode('utf-8'), ''), 0)

        # None of the violations have a valid line number, so they should all be skipped
        violations = QualityReporter(PylintDriver()).violations('file.py')
        self.assertEqual(violations, [])

    def test_quality_deprecation_warning(self):

        # Patch the output stderr/stdout and returncode of `pylint`
        _setup_patch((
            b'file1.py:1: [C0111] Missing docstring\n'
            b'file1.py:1: [C0111, func_1] Missing docstring',
            b'Foobar: pylintrc deprecation warning'
        ), 0)

        # Parse the report
        quality = QualityReporter(PylintDriver())
        actual_violations = quality.violations('file1.py')

        # Assert that pylint successfully runs and finds 2 violations
        self.assertEqual(len(actual_violations), 2)

    def test_quality_error(self):
        # Patch the output stderr/stdout and returncode of `pylint`
        _setup_patch((b'file1.py:1: [C0111] Missing docstring', b'oops'), status_code=1)

        # Parse the report
        with patch('diff_cover.violationsreporters.violations_reporter.run_command_for_code') as code:
            code.return_value = 0
            quality = QualityReporter(PylintDriver())

            # Expect an error
            self.assertRaises(CommandError, quality.violations, 'file1.py')

    def test_no_quality_issues_newline(self):

        _setup_patch((b'\n', b''), 0)

        # Parse the report
        quality = QualityReporter(PylintDriver())
        self.assertEqual([], quality.violations('file1.py'))

    def test_no_quality_issues_emptystring(self):

        # Patch the output of `pylint`
        _setup_patch((b'', b''), 0)

        # Parse the report
        quality = QualityReporter(PylintDriver())
        self.assertEqual([], quality.violations('file1.py'))

    def test_quality_pregenerated_report(self):

        # When the user provides us with a pre-generated pylint report
        # then use that instead of calling pylint directly.
        pylint_reports = [
            BytesIO(dedent("""
                path/to/file.py:1: [C0111] Missing docstring
                path/to/file.py:57: [W0511] TODO the name of this method is a little bit confusing
                another/file.py:41: [W1201, assign_default_role] Specify string format arguments as logging function parameters
                another/file.py:175: [C0322, Foo.bar] Operator not preceded by a space
                        x=2+3
                          ^
                        Unicode: \u9404 \u1239
                another/file.py:259: [C0103, bar] Invalid name "\u4920" for type variable (should match [a-z_][a-z0-9_]{2,30}$)
            """).strip().encode('utf-8')),

            BytesIO(dedent("""
            path/to/file.py:183: [C0103, Foo.bar.gettag] Invalid name "\u3240" for type argument (should match [a-z_][a-z0-9_]{2,30}$)
            another/file.py:183: [C0111, Foo.bar.gettag] Missing docstring
            """).strip().encode('utf-8'))
        ]

        # Generate the violation report
        quality = QualityReporter(PylintDriver(), reports=pylint_reports)

        # Expect that we get the right violations
        expected_violations = [
            Violation(1, 'C0111: Missing docstring'),
            Violation(57, 'W0511: TODO the name of this method is a little bit confusing'),
            Violation(183, 'C0103: Foo.bar.gettag: Invalid name "\u3240" for type argument (should match [a-z_][a-z0-9_]{2,30}$)')
        ]

        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('path/to/file.py')
        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)

    def test_quality_pregenerated_report_continuation_char(self):

        # The report contains a non-ASCII continuation char
        pylint_reports = [BytesIO(b"file.py:2: [W1401] Invalid char '\xc3'")]

        # Generate the violation report
        quality = QualityReporter(PylintDriver(), reports=pylint_reports)
        violations = quality.violations('file.py')

        # Expect that the char is replaced
        self.assertEqual(violations, [Violation(2, "W1401: Invalid char '\ufffd'")])


class JsQualityBaseReporterMixin(object):
    """
    Generic JS linter tests. Assumes the linter is not available as a python
    library, but is available on the commandline.
    """

    def setUp(self):
        # Mock patch the installation of the linter
        self._mock_command_simple = patch('diff_cover.violationsreporters.violations_reporter.run_command_for_code').start()
        self._mock_command_simple.return_value = 0
        # Mock patch the linter results
        self._mock_communicate = patch.object(subprocess, 'Popen').start()
        self.subproc_mock = MagicMock()
        self.subproc_mock.returncode = 0

    def tearDown(self):
        """
        Undo all patches
        """
        patch.stopall()

    def _get_out(self):
        """
        get Object Under Test
        """
        return None  # pragma: no cover

    def test_quality(self):
        """
        Test basic scenarios, including special characters that would appear in JavaScript and mixed quotation marks
        """
        _patch_so_all_files_exist()
        # Patch the output of the linter cmd
        return_string = '\n' + dedent("""
                ../test_file.js: line 3, col 9, Missing "use strict" statement.
                ../test_file.js: line 10, col 17, '$hi' is defined but never used.
            """).strip() + '\n'
        self.subproc_mock.communicate.return_value = (
            (return_string.encode('utf-8'), b''))
        self._mock_communicate.return_value = self.subproc_mock

        # Parse the report
        quality = QualityReporter(self._get_out())

        # Expect that the name is set
        self.assertEqual(quality.name(), self.quality_name)

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('../blah.js'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(3, 'Missing "use strict" statement.'),
            Violation(10, "'$hi' is defined but never used."),
        ]

        self.assertEqual(expected_violations, quality.violations('../test_file.js'))

    def test_no_quality_issues_newline(self):

        # Patch the output of the linter cmd
        self.subproc_mock.communicate.return_value = (b'\n', b'')
        self._mock_communicate.return_value = self.subproc_mock

        # Parse the report
        quality = QualityReporter(self._get_out())
        self.assertEqual([], quality.violations('test-file.js'))

    def test_no_quality_issues_emptystring(self):

        # Patch the output of the linter cmd
        self.subproc_mock.communicate.return_value = (b'', b'')
        self._mock_communicate.return_value = self.subproc_mock

        # Parse the report
        quality = QualityReporter(self._get_out())
        self.assertEqual([], quality.violations('file1.js'))

    def test_quality_error(self):
        _patch_so_all_files_exist()
        _setup_patch((b"", 'whoops Ƕئ'.encode('utf-8')), status_code=1)
        with patch('diff_cover.violationsreporters.base.run_command_for_code') as code:
            code.return_value = 0
            # Parse the report
            quality = QualityReporter(self._get_out())

            # Expect that the name is set
            self.assertEqual(quality.name(), self.quality_name)
            with self.assertRaises(CommandError) as ex:
                quality.violations('file1.js')
            self.assertEqual(six.text_type(ex.exception), 'whoops Ƕئ')

    def test_no_such_file(self):
        quality = QualityReporter(self._get_out())

        # Expect that we get no results
        result = quality.violations('')
        self.assertEqual(result, [])

    def test_no_js_file(self):
        quality = QualityReporter(self._get_out())
        file_paths = ['file1.py', 'subdir/file2.java']
        # Expect that we get no results because no JS files
        for path in file_paths:
            result = quality.violations(path)
            self.assertEqual(result, [])

    def test_quality_pregenerated_report(self):

        # When the user provides us with a pre-generated linter report
        # then use that instead of calling linter directly.
        reports = [
            BytesIO(('\n' + dedent("""
                path/to/file.js: line 3, col 9, Missing "use strict" statement.
                path/to/file.js: line 10, col 130, Line is too long.
                another/file.js: line 1, col 1, 'require' is not defined.
            """).strip() + '\n').encode('utf-8')),

            BytesIO(('\n' + dedent("""
                path/to/file.js: line 12, col 14, \u9134\u1912
                path/to/file.js: line 10, col 17, '$hi' is defined but never used.
            """).strip() + '\n').encode('utf-8')),
        ]

        # Parse the report
        quality = QualityReporter(self._get_out(), reports=reports)

        # Measured_lines is undefined for
        # a quality reporter since all lines are measured
        self.assertIsNone(quality.measured_lines('path/to/file.js'))

        # Expect that we get the right violations
        expected_violations = [
            Violation(3, 'Missing "use strict" statement.'),
            Violation(10, "Line is too long."),
            Violation(10, "'$hi' is defined but never used."),
            Violation(12, "\u9134\u1912")
        ]

        # We're not guaranteed that the violations are returned
        # in any particular order.
        actual_violations = quality.violations('path/to/file.js')

        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)

    def test_not_installed(self):
        """
        If linter is not available via commandline, it should raise
        an EnvironmentError
        """
        self._mock_command_simple = patch('diff_cover.violationsreporters.violations_reporter.run_command_for_code').start()
        self._mock_command_simple.return_value = 1
        with self.assertRaises(EnvironmentError):
            QualityReporter(self._get_out()).violations('test.js')


class JsHintQualityReporterTest(JsQualityBaseReporterMixin, unittest.TestCase):
    """
    JsHintQualityReporter tests. Assumes JsHint is not available as a python
    library, but is available on the commandline.
    """

    quality_name = 'jshint'

    def _get_out(self):
        return jshint_driver


class ESLintQualityReporterTest(JsQualityBaseReporterMixin, unittest.TestCase):
    """
    ESLintQualityReporter tests. Assumes ESLint is not available as a python
    library, but is available on the commandline.
    """

    quality_name = 'eslint'

    def _get_out(self):
        return eslint_driver


class SimpleCommandTestCase(unittest.TestCase):
    """
    Tests that the exit code detected by the method is passed as the return value of the method.
    """

    def setUp(self):
        self._mock_communicate = patch.object(subprocess, 'Popen').start()
        self.subproc_mock = MagicMock()

    def test_run_simple_failure(self):
        # command_simple should fail
        self.subproc_mock.returncode = 127
        self._mock_communicate.return_value = self.subproc_mock
        # Create an implementation of BaseQualityReporter and explicitly call _run_command_simple
        bad_command = run_command_for_code('foo')  # pylint: disable=protected-access
        self.assertEqual(bad_command, 127)

    def test_run_simple_success(self):
        self.subproc_mock.returncode = 0
        self._mock_communicate.return_value = self.subproc_mock
        # Create an implementation of BaseQualityReporter and explicitly call _run_command_simple
        good_command = run_command_for_code('foo')  # pylint: disable=protected-access
        self.assertEqual(good_command, 0)


class SubprocessErrorTestCase(unittest.TestCase):
    """
    Error in subprocess call(s)
    """
    def setUp(self):
        # when you create a new subprocess.Popen() object and call .communicate()
        # on it, raise an OSError
        _mock_Popen = Mock()
        _mock_Popen.return_value.communicate.side_effect = OSError
        patcher = patch("diff_cover.command_runner.subprocess.Popen", _mock_Popen)
        patcher.start()
        self.addCleanup(patcher.stop)

    def tearDown(self):
        patch.stopall()

    @patch('sys.stderr', new_callable=StringIO)
    def test_quality_reporter(self, mock_stderr):
        _patch_so_all_files_exist()
        with patch('diff_cover.violationsreporters.base.run_command_for_code') as code:
            code.return_value = 0
            reporter = QualityReporter(pycodestyle_driver)
            with self.assertRaises(OSError):
                reporter.violations("path/to/file.py")

            self.assertEqual(mock_stderr.getvalue(), "pycodestyle path/to/file.py")

class CppcheckQualityDriverTest(unittest.TestCase):
    """
    Tests for cppcheck quality driver.
    """
    def test_parse_report(self):
        """Basic report test parse"""
        expected_violations = {
            "src/foo.c": Violation(123, "(error) Array 'yolo[4]' accessed at index 4, which is out of bounds."),
        }
        report = "[src/foo.c:123]: (error) Array 'yolo[4]' accessed at index 4, which is out of bounds."

        driver = CppcheckDriver()
        actual_violations = driver.parse_reports([report])
        self.assertEqual(len(actual_violations), len(expected_violations))
        for expected in expected_violations:
            self.assertIn(expected, actual_violations)
