-- This file is part of the Grid LSC User Environment (GLUE)
-- 
-- GLUE is free software: you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation, either version 3 of the License, or (at your option) any later
-- version.
-- 
-- This program is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
-- FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
-- details.
-- 
-- You should have received a copy of the GNU General Public License along
-- with this program.  If not, see <http://www.gnu.org/licenses/>.

CREATE TABLE gds_trigger
(
-- Triggers generated by data monitoring programs (on-line or off-line).

-- Database which created this entry
      creator_db         INTEGER NOT NULL WITH DEFAULT 1,

-- INFORMATION ABOUT THE PROGRAM AND FILTER WHICH GENERATED THIS TRIGGER
-- Process ID
      process_id         CHAR(13) FOR BIT DATA NOT NULL,
-- Filter identifier (indicates type of filter, plus parameters).  May be null
      filter_id          CHAR(13) FOR BIT DATA,
-- Unique descriptive trigger name.
      name               VARCHAR(32) NOT NULL,
-- Sub-type of the trigger (e.g. a particular channel name)
      subtype            VARCHAR(64) NOT NULL,

-- INFORMATION ABOUT THIS PARTICULAR TRIGGER INSTANCE
-- Site or interferometer to which this trigger applies (H0, H1, H2, L0, L1)
      ifo                CHAR(2) NOT NULL,

-- Time when the trigger detects a transient (GPS seconds and nanoseconds)
      start_time         INTEGER NOT NULL,
      start_time_ns      INTEGER NOT NULL,

-- Duration of the transient (seconds)
      duration           REAL NOT NULL,

-- Priority of the trigger (meaning has yet to be defined)
      priority           INTEGER NOT NULL,

-- Bitmask indicating what was done with this trigger (sent to operator
-- console, logged in database, etc.)
      disposition        INTEGER,

-- Peak amplitude of the triggered effect (i.e. filtered signal amplitude at
--   time tPeak). May be omitted (set to zero) in cases where this can't be
--   measured (F-domain or wavelets), see significance.
      size               REAL NOT NULL WITH DEFAULT 0,

-- Number of sigmas represented by the peak amplitude (i.e. SIZE divided 
--   by the sigma of the noise the in the appropriate frequency band).
--   Zero indicates that the significance is unknown.
      significance       REAL NOT NULL WITH DEFAULT 0,

-- Low frequency of the (t, F) tile or prefilter response.
      frequency          REAL NOT NULL WITH DEFAULT 0,

-- Width (fmax-fmin) of the (t,F) tile or t-domain filter response. 
--   BANDWIDTH=0 indicates that the frequency limits are not known.
      bandwidth          REAL NOT NULL WITH DEFAULT 0,

-- Offset from the start time to center of the peak amplitude pixel (peak 
--   sample time in time-domain monitors).
      time_peak          REAL,

-- The signal power weighted average time in seconds relative to the start 
--   time,  e.g., in frequency or wavelet domain monitors:
--
--        tAvg = Sum(i; (P_i - B_i)*(t_i - tStart)) / Sum(i; P_i - B_i)
--
--   where i is a pixel(wavelet) index and P_i, B_i, and t_i are the Total 
--   Power, expected Background noise power and average time of the ith pixel. 
--   Obviously, only pixels with P_i > B_i should be included in the sum.
--   In time domain monitors, individual samples are used instead of pixels.
      time_average       REAL,

-- Gaussian duration estimate of the signal in seconds, i.e.
--
--        tSig = Sum(i; (P_i - B_i)*(t_i - tAvg)2) / Sum(i; P_i - B_i)
--
--   Note that the pixel time granularity is ignored here. This is may be
--   reviewed. especially if some algorithm is created with a single time
--   pixel.
      time_sigma        REAL NOT NULL WITH DEFAULT 0,

-- Center frequency of peak amplitude pixel (0.0 in time-domain monitors?).
      freq_peak         REAL NOT NULL WITH DEFAULT 0,

-- Frequency average of the signal power, i.e.
--
--        fAvg = Sum(i; (P_i - B_i)*(f_i) / Sum(i; P_i - B_i)
--
--   where i is a pixel index and P_i, B_i, and f_i are the Power, Background
--   noise power and average frequency of the ith pixel. 
      freq_average      REAL NOT NULL WITH DEFAULT 0,


-- Gaussian bandwidth estimate of the signal, i.e.
--
--       fSig = Sum(i; (P_i - B_i)*(f_i-fAvg)2) / Sum(i; P_i - B_i)
--
--  Note that trigger algorithms that use a single frequency bin (i.e. 
--  time domain algorithms) would by construction have fSig=0.
      freq_sigma        REAL NOT NULL WITH DEFAULT 0,


-- Total expected noise power summed over all pixels used in the Gaussian 
--   estimate, e.g.
--
--        Pnoise = Sum(i; B_i)
--
      noise_power       REAL NOT NULL WITH DEFAULT 0,


-- Total signal power minus the background power used in the Gaussian 
--   estimate, i.e.
--
--        Psignal = Sum(i; P_i - B_i)
--
      signal_power      REAL NOT NULL WITH DEFAULT 0,


-- Number of Pixels used in the Gaussian estimate, i.e.
--
--        Npix = Sum(i; 1)
--
      pixel_count       REAL NOT NULL WITH DEFAULT 0,


-- Negative Log of the probability that the measured signal would be 
-- caused by expected gaussian noise. A zero confidence indicates that
--  configdence is unknown.
      confidence        REAL NOT NULL WITH DEFAULT 0,

-- Full result, e.g. parameters from a fit.  User-defined binary format.
-- This will generally be very small (or NULL), so we prefer to store this
-- in a VARCHAR rather than in a BLOB.
      binarydata         VARCHAR(1024) FOR BIT DATA,
-- Length of full result, in bytes (0 if no full_result is recorded)
      binarydata_length  INTEGER NOT NULL,
-- A unique identifier for this trigger
      event_id           CHAR(13) FOR BIT DATA NOT NULL,
      
-- Insertion time (automatically assigned by the database)
      insertion_time     TIMESTAMP WITH DEFAULT CURRENT TIMESTAMP,

      CONSTRAINT gdstrig_pk
      PRIMARY KEY (event_id, creator_db),

      CONSTRAINT gdstrig_fk_pid
      FOREIGN KEY (process_id, creator_db)
          REFERENCES process(process_id, creator_db),

-- Note that filter_id is allowed to be null, in which case no check is made.
      CONSTRAINT gdstrig_fk_filt
      FOREIGN KEY (filter_id, creator_db)
          REFERENCES filter(filter_id, creator_db),
-- **********************************************************************
--   Integrity Constraints
-- **********************************************************************
-- Make sure the time is valid
      CONSTRAINT time_integrity
      CHECK ( start_time > 0),
-- Make sure the duration makes sense
      CONSTRAINT duration_integrity
      CHECK (duration >= 0)
)
-- The following line is needed for this table to be replicated to other sites
DATA CAPTURE CHANGES
;
-- Create a clustering index based on name
CREATE INDEX gdstrig_cind ON gds_trigger(name) CLUSTER
;
-- Create an index based on name and subtype
CREATE INDEX gdstrig_ind_subt ON gds_trigger(name, subtype, start_time)
;
-- Create an index based on time alone
CREATE INDEX gdstrig_ind_time ON gds_trigger(start_time)
;
-- Create an index based on process_id
CREATE INDEX gdstrig_ind_pid ON gds_trigger(process_id, start_time)
;
-- Create an SQL trigger so that if a gds_trigger entry is deleted, any
-- associated sngl_datasource, sngl_transdata, and/or sngl_mime entries
-- are deleted too.
-- Must be done this way because there is no foreign-key relationship.
-- Run script gds_trigger_tr_del.sql to create delete trigger for 
-- sngl_datasource, sngl_transdata, and sngl_mime records.
