"""Deploy infrastructure"""
import json
import logging
import os
import shutil
import subprocess
import zipfile
from pathlib import Path

import botocore

import deterministic_zip as dz

from ..cloud import aws
from ..config import Config
from . import interface as ui

LOG = logging.getLogger(__name__)


def to_teal_cloud(args, config):
    raise NotImplementedError

    python_zip = config.project.data_dir / "python.zip"
    teal_zip = config.project.data_dir / "teal.zip"
    config_json = config.project.data_dir / "config.json"

    with ui.spin(args, "Building packages...") as sp:
        deploy.package_python(config, python_zip)
        deploy.package_teal(config, teal_zip)
        deploy.package_config(Path(args["--config"]), config_json)
        sp.ok(ui.TICK)

    with ui.spin(args, "Uploading data"):
        # response = requests.post(TEAL_ENDPOINT + "/new_upload", project=project)
        # teal_uri = response["teal_uri"]
        # s3 upload everything
        deploy.upload_to_s3(teal_uri, teal_zip)
        deploy.upload_to_s3(python_uri, python_zip)
        sp.ok(ui.TICK)

    with ui.spin(args, f"Deploying {tag}"):
        # upload to bucket
        # requests.post(
        #     TEAL_ENDPOINT + "/deploy", project=project, tag=tag, s3_url=s3_url
        # )
        sp.ok(ui.TICK)

    print(good(f"\nDone. `teal invoke` to run main()."))


def to_own_cloud(args, config):
    layer_zip = config.project.data_dir / "python_layer.zip"

    _make_python_layer_zip(config, layer_zip)

    deploy_config = aws.DeployConfig(
        uuid=config.instance_uuid,
        instance=config.instance,
        source_layer_hash=aws.hash_file(layer_zip),
        source_layer_file=layer_zip,
    )

    with ui.spin(args, "Deploying infrastructure") as sp:
        # this is idempotent
        aws.deploy(deploy_config)
        sp.text += ui.dim(f" {config.project.data_dir}/")
        sp.ok(ui.TICK)

    with ui.spin(args, "Checking API") as sp:
        api = aws.get_api()
        response = call_cloud_api("version", {}, deploy_config)
        sp.text += " Teal " + ui.dim(response["version"])
        sp.ok(ui.TICK)

    with ui.spin(args, f"Deploying {config.project.teal_file}") as sp:
        payload = _package_teal(config)
        _call_cloud_api("set_exe", payload, deploy_config)
        sp.ok(ui.TICK)

    LOG.info(f"Uploaded {config.project.teal_file}")
    print(good(f"\nDone. `teal invoke` to run main()."))


def call_cloud_api(function: str, args: dict, config: aws.DeployConfig, as_json=True):
    """Call a teal API endpoint and handle errors"""
    LOG.debug("Calling Teal cloud: %s %s", function, args)

    try:
        api = aws.get_api()
        logs, response = getattr(api, function).invoke(config, args)
    except botocore.exceptions.ClientError as exc:
        if exc.response["Error"]["Code"] == "KMSAccessDeniedException":
            msg = "\nAWS is not ready (KMSAccessDeniedException). Please try again in a few minutes."
            print(ui.bad(msg))
            ui.let_us_know("Deployment Failed (KMSAccessDeniedException)")
            sys.exit(1)
        raise

    LOG.info(logs)

    # This is when there's an unhandled exception in the Lambda.
    if "errorMessage" in response:
        print("\n" + ui.bad("Unexpected exception!"))
        msg = response["errorMessage"]
        ui.let_us_know(msg)
        ui.exit_fail(msg, response.get("stackTrace", None))

    code = response.get("statusCode", None)
    if code == 400:
        print("\n")
        # This is when there's a (handled) error
        err = json.loads(response["body"])
        ui.exit_fail(err.get("message", "StatusCode 400"), err.get("traceback", None))

    if code != 200:
        print("\n")
        msg = f"Unexpected response code: {code}"
        ui.let_us_know(msg)
        ui.exit_fail(msg)

    body = json.loads(response["body"]) if as_json else response["body"]
    LOG.info(body)

    return body


## Helpers:


def _package_teal(config: Config) -> dict:
    with open(config.project.teal_file) as f:
        content = f.read()

    # See teal_lang/executors/awslambda.py
    return {"content": content}


def _zip_dir(dirname: Path, dest: Path, deterministic=True):
    """Zip a directory"""
    # https://github.com/bboe/deterministic_zip/blob/master/deterministic_zip/__init__.py
    with zipfile.ZipFile(dest, "w") as zip_file:
        dz.add_directory(zip_file, dirname, dirname.name)


def _make_teal_zip(config, dest: Path):
    raise NotImplementedError


def _make_python_layer_zip(config: Config, dest: Path):
    """Create the python code layer Zip, saving it in dest"""
    root = Path(__file__).parents[3]

    if not config.project.python_src.exists():
        raise DeploymentFailed(
            f"Python source directory ({config.project.python_src}) not found"
        )

    LOG.info(f"Building Source Layer package in {dest}...")
    workdir = config.project.data_dir / "source_build" / "python"
    os.makedirs(workdir, exist_ok=True)

    # User source
    shutil.copytree(
        config.project.python_src,
        workdir / config.project.python_src.name,
        dirs_exist_ok=True,
    )

    # Pip requirements if they exist
    reqs_file = config.project.python_requirements
    if reqs_file.exists():
        LOG.info(
            f"Installing pip packages from {config.project.python_requirements}..."
        )
        subprocess.check_output(
            ["pip", "install", "-q", "--target", workdir, "-r", reqs_file]
        )

    # Make sure the zip is not empty
    with open(workdir / ".packaged_by_teal.txt", "w") as f:
        f.write("Packed by Teal :)")

    _zip_dir(workdir, dest)
