# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['rgx']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'rgx',
    'version': '1.3.0',
    'description': 'Typed, simple and readable regexp generation',
    'long_description': 'Many people complain about unreadable and complex syntax of regular expressions.    \nMany others complain about how they can\'t remember all constructs and features.\n\n`rgx` solves those problems: it is a straightforward regexp builder. It also places non-capturing groups where needed to respect intended operator priority.   \nIt tries to produce a clean  \nIt can produce a regular expression string to use in `re.compile` or any other regex library of your choice.    \n\nIn other words, with `rgx` you can build a regular expression from parts, using straightforward and simple expressions.\n\n## Installation\n\n`pip install rgx`\n\nThat\'s it.\n\n## Basic usage\n\n### Hello, regex world\n\n```python\nimport rgx\nimport re\n\nword = rgx.meta.WORD_CHAR.many().capture() # (\\w+), a capturing group\ncomma = rgx.pattern(",").maybe()\n\nregex = rgx.pattern((\n    "hello",\n    comma,\n    rgx.meta.WHITESPACE,\n    (\n        word + rgx.meta.WHITESPACE\n    ).maybe(),\n    "world"\n)) # (?:hello,?\\s(?:(\\w+)\\s)?world)\n\nre.compile(\n    regex.render_str("i") # global flag (case-insensitive)\n)\n\n```\n\n### Match some integers\n\nthis regex will match valid Python integer literals:\n\n```python\nimport rgx\nimport re\n\nnonzero = rgx.pattern("1").to("9") # [1-9]\nzero = "0"\ndigit = zero | nonzero # [0-9]\ninteger = zero | (nonzero + digit.some()) # 0|[1-9][0-9]*\n\nint_regex = re.compile(str(integer))\n\n```\n\n...or this one:\n\n```python\nimport rgx\nimport re\n\nnonzero = rgx.pattern("1").to("9") # [1-9]\ndigit = rgx.meta.DIGIT # \\d\ninteger = digit | (nonzero + digit.some()) # \\d|[1-9]\\d*\n\nint_regex = re.compile(str(integer))\n\n```\n\n## Quickstart\n\n*in this readme, `x` means some pattern object. Occasionaly, `y` is introduced to mean some other pattern object (or literal)*\n\n### Literals and pattern objects\n\n`rgx` operates mostly on so-called "pattern objects" — `rgx.entities.RegexPattern` istances.    \nYour starting point would be `rgx.pattern` — it creates pattern objects from literals (and from pattern objects, which doesn\'t make a lot of sense).\n\n- `rgx.pattern(str, escape: bool = True)` creates a literal pattern — one that exactly matches given string. If you want to disable escaping, pass `escape=False`\n- `rgx.pattern(tuple[AnyRegexPattern])` creates a non-capturing group of patterns (nested literals will be converted too)\n- `rgx.pattern(list[str])` creates a character class (for example, `rgx.pattern(["a", "b", "c"])` creates pattern `[abc]`, that matches any character of those in brackets)\n    - Same can be achieved by `rgx.pattern("a").to("c")` or `rgx.pattern("a") | "b" | "c"`\n\nMost operations with pattern objects support using Python literals on one side, for example: `rgx.pattern("a") | b` would produce `[ab]` pattern object (specifically, `rgx.entities.Chars`)    \n\n### Rendering patterns\n\n```python\n\nimport rgx\n\nx = rgx.pattern("one")\ny = rgx.pattern("two")\npattern = x | y\n\nrendered_with_str = str(pattern) # "one|two"\nrendered_with_method = pattern.render_str() # "one|two"\nrendered_with_method_flags = pattern.render_str("im") # (?im)one|two\n```    \n\n### Capturing Groups\n\n```python\nimport rgx\n\nx = rgx.pattern("x")\n\nprint(x.capture()) # (x)\n\nprint(rgx.reference(1)) # \\1\n\n\nnamed_x = x.named("some_x") # x.named(name: str)\n\nprint(named_x) # (?P<some_x>x)\n\nnamed_x_reference = rgx.named("some_x")\n\nprint(named_x_reference) # (?P=x)\n\n```\n\nTo create a capturing group, use `x.capture()`, or `rgx.reference(group: int)` for a reference.    \nTo create a named capturing group, use `rgx.named(name: str, x)`, or `rgx.named(name: str)` for a named reference.    \n\n### Character classes\n\n```python\nimport rgx\n\n\naz = rgx.pattern("a").to("z") # rgx.Chars.to(other: str | Literal | Chars)\nprint(az) # [a-z]\n\ndigits_or_space = rgx.pattern(["1", "2", "3", rgx.meta.WHITESPACE]) \nprint(digits_or_space) # [123\\s]\n\nprint(az | digits_or_space) # [a-z123\\s]\n\n\nprint( # rgx.entities.Chars.reverse(self)\n    (az | digits_or_space).reverse() # [^a-z123\\s]\n)\n\n```  \n\n### Conditional pattern\n\n```python\nimport rgx\n\nx = rgx.pattern("x")\ny = rgx.pattern("y")\nz = rgx.pattern("z")\n\ncapture = x.capture()\n\n# (x)(?(1)y|z)\nprint(\n    capture + rgx.conditional(1, y, z)\n)\n``` \n\n### Repeating patterns\n\nIf you need to match a repeating pattern, you can use `pattern.repeat(count, lazy)`:\n\n```python\na = pattern("a")\n\na.repeat(5)                      # a{5}\n# or\na * 5                            # a{5}, multiplication is an alias for .repeat\n\na.repeat(5).or_more()            # a{5,}\na.repeat(5).or_less()            # a{,5}\n\na.repeat_from(4).to(5)           # a{4, 5}  [.repeat_from is just an alias]\n# or\na.repeat(4) >> 5                 # a{4, 5}\n\na.repeat(1).or_less()            # a?\n# or\n-a.repeat(1)                     # a?\n# or\na.maybe()                        # a?\n\na.repeat(1).or_more()            # a+\n# or\n+a.repeat(1)                     # a+\n# or\n+a                               # a+\n# or\na.many()                         # a+\n\na.repeat(0).or_more()            # a*\n# or\n+a.repeat(0)                     # a*\n# or\na.some()                         # a*\n# or (what)\n+-(a * 38)                       # a*\n```\nHere\'s what\'s going on:    \n`pattern.repeat(count, lazy)` returns a `{count, count}` `Range` object\n`pattern * count` is the same as `pattern.repeat(count, False)`\n\n`Range` implements `or_more`, `or_less` and `to` methods:\n\n- `Range.or_more()` [or `+Range`] moves (on a copy) upper bound of range to infinity (actually `None`)\n- `Range.or_less()` [or `-Range`] moves (on a copy) lower bound of range to 0\n- `Range.to(count)` [or `Range >> count` (right shift)] replaces upper bound with given number\n\nAlso, RegexPattern implements unary plus (`+pattern`) as an alias for `pattern.many()`\n\n## Docs\n\n### Pattern methods\n\n#### `pattern.render_str(flags: str = \'\') -> str`\n\nRenders given pattern into a string with specified global flags.\n\n---\n\n#### `pattern.set_flags(flags: str) -> LocalFlags`\n\nThis method adds local flags to given pattern\n\n```python\nx.flags("y") # "(?y:x)"\n```\n\n---\n\n#### `pattern.concat(other: AnyRegexPattern) -> Concat`\n\nUse to match one pattern and then another.\n\n`A.concat(B)` is equivalent to `A + B` (works if either A or B is a RegexPart object, not a Python literal)\n\n```python\nx.concat(y) # "xy"\nx + y # "xy"\n```\n\n---\n\n#### `pattern.option(other: AnyRegexPattern) -> Chars | ReversedChars | Option`\n\nUse to match either one pattern or another.\n\n`A.option(B)` is equivalent to `A | B` (if either A or B is a RegexPart object, not a Python literal)\n\n```python\nx.option(y) # "x|y"\nx | y # "x|y"\n```\n\n---\n\n#### `pattern.many(lazy: bool = False) -> Range`\n\nUse this for repeating patterns (one or more times)\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.many() # "x+"\nx.many(True) # "x+?"\n```\n\n---\n\n#### `pattern.some(lazy: bool = False) -> Range`\n\nUse this for repeating optional patterns (zero or more times)\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.some() # "x*"\nx.some(True) # "x*?"\n```\n\n---\n\n#### `pattern.maybe(lazy: bool = False) -> Range`\n\nUse this for optional patterns (zero or one times)\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.maybe() # "x?"\nx.maybe(True) # "x??"\n```\n\n---\n\n#### `pattern.x_or_less_times(count: int, lazy: bool = False) -> Range`\n\nUse this to match pattern x or less times (hence the name).\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.x_or_less_times(5) # "x{,5}"\nx.x_or_less_times(5, True) # "x{,5}?"\n```\n\n---\n\n#### `pattern.x_or_more_times(count: int, lazy: bool = False) -> Range`\n\nUse this to match pattern x or more times (hence the name).\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.x_or_more_times(5) # "x{5,}"\nx.x_or_more_times(5, True) # "x{5,}?"\n```\n\n---\n\n#### `pattern.x_times(count: int, lazy: bool = False) -> Range`\n\nUse this to match pattern exactly x times (hence the name).\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.x_times(5) # "x{5}"\nx.x_times(5, True) # "x{5}?"\nx.repeat(5) # x{5}\n```\n\n---\n\n#### `pattern.between_x_y_times(min_count: int, max_count: int, lazy: bool = False) -> Range`\n\nUse this to match pattern between x and y times, inclusive (hence the name).\n\nWhen not lazy, matches as many times as possible, otherwise matches as few times as possible.\n\n```python\nx.between_x_y_times(5, 6) # "x{5,6}"\nx.between_x_y_times(5, 6, True) # "x{5,6}?"\n```\n\n---\n\n#### `pattern.lookahead(other: RegexPattern) -> Concat`\n\nUse this to indicate that given pattern occurs before some another pattern (lookahead).\n\nIn other words, `x.lookahead(y)` matches a pattern `x` only if there is `y` after it\n\nLookahead pattern won\'t be captured.\n\n```python\nx.lookahead(y) # x(?=y)\nx.before(y) # x(?=y)\n```\n\n---\n\n#### `pattern.negative_lookahead(other) -> Concat`\n\nUse this to indicate that given pattern doesn\'t occur before some another pattern (negative lookahead).\n\nIn other words, `x.negative_lookahead(y)` matches a pattern `x` only if there is no `y` after it\n\nLookahead pattern won\'t be captured.\n\n```python\nx.negative_lookahead(y) # x(?!y)\nx.not_before(y) # x(?!y)\n```\n\n---\n\n#### `pattern.lookbehind(other: RegexPattern) -> Concat`\n\nUse this to indicate that given pattern occurs after some another pattern (lookbehind).\n\nIn other words, `x.lookbehind(y)` matches a pattern `x` only if there is `y` before it\n\nLookbehind pattern won\'t be captured.\n\n```python\nx.lookbehind(y) # (?<=y)x\nx.after(y) # (?<=y)x\n```\n\n---\n\n#### `pattern.negative_lookbehind(other) -> Concat`\n\nUse this to indicate that given pattern goes before some another pattern (negative lookbehind).\n\nIn other words, `x.negative_lookbehind(y)` matches a pattern `x` only if there is NO `y` before it\n\nLookbehind pattern won\'t be captured.\n\n```python\nx.negative_lookbehind(y) # (?<!y)x\nx.not_after(y) # (?<!y)x\n```\n\n---\n\n#### `pattern.capture() -> Group`\n\nUse this to make a capturing group out of pattern.\n\n```python\nx.capture() # (x)\n```\n\n### Meta\n\n`rgx.meta` is a collection of different meta-sequences and anchors:\n\n```python\nWORD_CHAR = UnescapedLiteral(r"\\w")\nNON_WORD_CHAR = UnescapedLiteral(r"\\W")\nDIGIT = UnescapedLiteral(r"\\d")\nNON_DIGIT = UnescapedLiteral(r"\\D")\nWHITESPACE = UnescapedLiteral(r"\\s")\nNON_WHITESPACE = UnescapedLiteral(r"\\S")\nWORD_BOUNDARY = UnescapedLiteral(r"\\b")\nNON_WORD_BOUNDARY = UnescapedLiteral(r"\\B")\nANY = UnescapedLiteral(".")\nNEWLINE = UnescapedLiteral(r"\\n")\nCARRIAGE_RETURN = UnescapedLiteral(r"\\r")\nTAB = UnescapedLiteral(r"\\t")\nNULL_CHAR = UnescapedLiteral(r"\\0")\nSTRING_START = UnescapedLiteral("^")\nSTRING_END = UnescapedLiteral("$")\n```\n\nAlso `rgx.meta.CHAR_ESCAPE(char_number: int)` is available:\n\n```python\nfrom rgx import meta\n\nprint(meta.CHAR_ESCAPE(32)) # \\x20\nprint(meta.CHAR_ESCAPE(320)) # \\u0140\nprint(meta.CHAR_ESCAPE(320000)) # \\U0004e200\n\n```\n\n### Unicode meta\n\n`rgx.unicode_meta` is a collection of functions and constants, mostly for `\\p` and `\\P` usage:\n\nFunctions:    \n- `unicode_meta.PROPERTY(value: str)` renders into `\\p{value}` (any character with property specified by value, e.g. `PROPERTY("Ll") -> \\p{Ll}`)\n- `unicode_meta.PROPERTY_INVERSE(value: str)` - matches all characters *not* matched by corresponding `PROPERTY` (`\\P{value}`)\n\n- `unicode_meta.NAMED_PROPERTY(name: str, value: str)` - renders into `\\p{name=value}` and matches any character which property `name` equals `value`\n- `unicode_meta.NAMED_PROPERTY_INVERSE(name: str, value: str)` - same, but inverted (`\\P{name=value}`)\n\nConstants:    \n- `LETTER = PROPERTY("L")`\n- `NON_LETTER = PROPERTY_INVERSE("L")`\n\n- `WHITESPACE = PROPERTY("Z")`\n- `NON_WHITESPACE = PROPERTY_INVERSE("Z")`\n\n- `DIGIT = PROPERTY("Nd")`\n- `NON_DIGIT = PROPERTY("Nd")`\n\n## Extending\n\nYou can extend generation by subclassing one of the classes of `rgx.entities` module.    \nThe one neccessary method to provide is `.render(self)`. It should return an iterable of strings (e.g. `["something"]`).    \nBuilt-in components (and this section) are using generators for that purpose, but you\'re free to choose whatever works for you.\nFor example, if you want to render a PCRE accept control verb - `(*ACCEPT)`, you can do it like this:\n\n\n```python\nfrom rgx.entities import RegexPattern, Concat\nfrom rgx import pattern\nfrom typing import Iterable\n\n\nclass Accept(RegexPattern):\n    def render(self) -> Iterable[str]:\n        yield "(*ACCEPT)"\n\n\ndef accept(self) -> Concat:\n    return self + Accept()\n\n\nRegexPattern.accept = accept\n\nx = pattern("something").accept() \nprint(x) # something(*ACCEPT)\n```\n\nOr like this:\n\n```python\nfrom rgx.entities import RegexPattern, Concat\nfrom rgx import pattern\nfrom typing import Iterable\n\n\nclass Accept(RegexPattern):\n    def __init__(self, accepted_pattern: RegexPattern):\n        self.accepted_pattern = accepted_pattern\n\n    def render(self) -> Iterable[str]:\n        yield from accepted_pattern.render()\n        yield "(*ACCEPT)"\n\n\ndef accept(self) -> Concat:\n    return Accept(self)\n\nRegexPattern.accept = accept\n\nx = pattern("something").accept() # something(*ACCEPT)\n```\n\n### Priority\n\nIf your extension has to rely on some priority, you can use `respect_priority` function.    \nLet\'s say you want to add a `x/y` operation, which does something (wow) and has prority between `a|b` and `ab` — so `a|b/cd` is `a|(b/(cd))`.    \n\n```python\nfrom rgx.entities import RegexPattern, Concat, Option, AnyRegexPattern, respect_priority, pattern\nfrom typing import Iterable\n\nclass MagicSlash(RegexPattern):\n    priority = (Concat.priority + Option.priority) // 2 # let\'s take something in the middle\n\n    def __init__(self, left: RegexPattern, right: RegexPattern):\n        self.left = respect_priority(left, self.priority) # you need to wrap all parts of your expression in respect_priority() \n        self.right = respect_priority(right, self.priority) # ...and pass your expression priority as a second argument\n\n    def render(self) -> Iterable[str]:\n        yield from self.left.render()\n        yield "/"\n        yield from self.right.render()\n\n\ndef slash(self, other: AnyRegexPattern) -> MagicSlash: # AnyRegexPattern is either a RegexPattern instance or a Python literal\n    return MagicSlash(self, other) # respect_priority already takes literals in consideration, so no extra actions needed\n\ndef rslash(self, other: AnyRegexPattern) -> MagicSlash: # other/self\n    other = pattern(other)\n    return other / self\n\n\nRegexPattern.slash = slash\nRegexPattern.__truediv__ = slash # / operator\nRegexPattern.__rtruediv__ = rslash\n\n\na = pattern("a")\nb = pattern("b")\nc = pattern("c")\nd = pattern("d")\n\nprint(\n    (a | b) / (c + d) # [ab]/cd\n)\n\nprint(\n    ((a | b) / c) + d # (?:[ab]/c)d\n)\n\nprint(\n    a | (b / c) + d   # a|(?:b/c)d\n)\n\n```\n\n\n## Common questions\n\n### Difference between `(x, y)` and `x + y`\n\nPrevious examples used `()` and `+`, and the difference might not be so obvious.    \n\n- `x + y` creates a concatenation of patterns (`rgx.entities.Concat`), with no extra characters apart from those of patterns\n- `x + y` can be used only if at least one of the operands is a pattern object (that is, created with one of `rgx` functions or is one of `rgx` constants)\n- `x + y` produces a pattern object itself, so you won\'t need to call `pattern` on it to call pattern methods\n\n- `pattern((x, y))` creates a non-capturing group (`rgx.entities.NonCapturingGroup`): `pattern((x, y)).render_str()` -> `(?:xy)`\n- `(x, y)` can be used with any pattern-like literals or pattern objects\n- `(x, y)` is a tuple literal, so you can\'t use pattern methods on it directly or convert it into a complete expression (you need to use `rgx.pattern` on it first)',
    'author': 'Dmitry Gritsenko',
    'author_email': 'rgx@evtn.ru',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/courage-tci/rgx',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
