#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Note: To use the 'upload' functionality of this file, you must:
#   $ pipenv install twine --dev

import io
import os
import sys
from shutil import rmtree

from setuptools import find_packages, setup, Command

from mskit import __info__

NAME = __info__.__package_name__
DESCRIPTION = __info__.__short_description__
URL = __info__.__repository__
EMAIL = __info__.__email__
AUTHOR = __info__.__author__
VERSION = __info__.__version__
LICENSE = __info__.__license__
REQUIRES_PYTHON = '>=3.10.0'

REQUIRED = [
    'numpy', 'pandas', 'scipy',
    'matplotlib', 'seaborn',
    'scikit-learn',
    'requests',
    'prettytable', 'tqdm',
    'joblib', 'dask', 'distributed',
]

EXTRAS = {
    'MSDataAccess': ['pymzml', 'pyteomics', 'pymsfilereader'],
    'SpeedUp': ['numba'],
    'Image': ['opencv-python'],
    'torch': ['pytorch'],
    'tensorflow': ['tensorflow-gpu'],
    'GUI': ['wxpython'],
}
EXTRAS['ALL'] = sum(list(EXTRAS.values()), [])

PACKAGEDATA = {
    'unimod': ['package_data/unimod.xml'],
}

classifiers = [
    # Full list: https://pypi.python.org/pypi?%3Aaction=list_classifiers
    'License :: OSI Approved :: MIT License',

    'Development Status :: 2 - Pre-Alpha',

    'Programming Language :: Python :: 3 :: Only',
    'Programming Language :: Python :: 3.10',

    'Topic :: Scientific/Engineering :: Bio-Informatics',
    'Topic :: Scientific/Engineering :: Visualization',

    'Topic :: Software Development :: Version Control :: Git'
]

here = os.path.abspath(os.path.dirname(__file__))

# Import the README and use it as the long-description.
# Note: this will only work if 'README.md' is present in your MANIFEST.in file!
try:
    with io.open(os.path.join(here, 'README.md'), encoding='utf-8') as f:
        long_description = '\n' + f.read()
except FileNotFoundError:
    long_description = DESCRIPTION

# Load the package's __info__.py module as a dictionary.
about = {}
if not VERSION:
    project_slug = NAME.lower().replace("-", "_").replace(" ", "_")
    with open(os.path.join(here, project_slug, '__info__.py')) as f:
        exec(f.read(), about)
else:
    about['__version__'] = VERSION


class UploadCommand(Command):
    """Support setup.py upload."""

    description = 'Build and publish the package.'
    user_options = []

    @staticmethod
    def status(s):
        """Prints things in bold."""
        print('\033[1m{0}\033[0m'.format(s))

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass

    def run(self):
        try:
            self.status('Removing previous builds…')
            rmtree(os.path.join(here, 'dist'))
        except OSError:
            pass

        self.status('Building Source and Wheel (universal) distribution…')
        os.system(
            '{0} setup.py sdist bdist_wheel --universal'.format(sys.executable))

        self.status('Uploading the package to PyPI via Twine…')
        os.system('twine upload dist/*')

        self.status('Pushing git tags…')
        os.system('git tag v{0}'.format(about['__version__']))
        os.system('git push --tags')

        sys.exit()


setup(
    name=NAME,
    version=about['__version__'],
    description=DESCRIPTION,
    long_description=long_description,
    long_description_content_type='text/markdown',
    author=AUTHOR,
    author_email=EMAIL,
    python_requires=REQUIRES_PYTHON,
    url=URL,
    packages=find_packages(
        exclude=[
            "tests",
            "*.tests",
            "*.tests.*",
            "tests.*"]),

    # entry_points={
    #     'console_scripts': ['mycli=mymodule:cli'],
    # },
    install_requires=REQUIRED,
    extras_require=EXTRAS,
    include_package_data=True,
    package_data=PACKAGEDATA,
    license=LICENSE,
    classifiers=classifiers,
    cmdclass={
        'upload': UploadCommand,
    },
)
