/*
    (C) Copyright 2012 CEA LIST. All Rights Reserved.
    Contributor(s): Olivier BICHLER (olivier.bichler@cea.fr)

    This software is governed by the CeCILL-C license under French law and
    abiding by the rules of distribution of free software.  You can  use,
    modify and/ or redistribute the software under the terms of the CeCILL-C
    license as circulated by CEA, CNRS and INRIA at the following URL
    "http://www.cecill.info".

    As a counterpart to the access to the source code and  rights to copy,
    modify and redistribute granted by the license, users are provided only
    with a limited warranty  and the software's author,  the holder of the
    economic rights,  and the successive licensors  have only  limited
    liability.

    The fact that you are presently reading this means that you have had
    knowledge of the CeCILL-C license and that you accept its terms.
*/

#ifndef N2D2_PROGRAMOPTIONS_H
#define N2D2_PROGRAMOPTIONS_H

#include <algorithm>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <iterator>
#include <map>
#include <sstream>
#include <stdexcept>
#include <string>
#include <tuple>
#include <utility>
#include <vector>

#include "utils/Utils.hpp"

namespace N2D2 {
/**
 * This class is a lightweight and easy to use handler to add command line
 * options to a program.
 * The philosophy is that any type of option can be added to the program by
 * calling only one function, taking directly all the
 * necessary arguments.
*/
class ProgramOptions {
public:
    /**
     * Constructor
     *
     * @param argc              First argument of main()
     * @param argv              Second argument of main()
     * @param abortOnError      Indicate if the program should exit when an
     *invalid value is used for an option or if the
     *                          remaining argument list is not empty when
     *calling done()
     * @param saveArgs          If true, save the argument list to a file
     *[binary].args
    */
    ProgramOptions(int argc,
                   char* argv[],
                   bool abortOnError = true,
                   bool saveArgs = true);

    /**
     * Search for an option identifier in the remaining argument list and remove
     *it from the list.
     *
     * @param option            Option identifier
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return true if the option is found, false if not found
    */
    bool parse(const std::string& option, const std::string& desc = "");

    /**
     * Parse an option of type "option value" in the remaining argument list and
     *remove both the option and value from the list.
     * If the keyword "option" is not found, or if it is not followed by a value
     *(i.e. it is the last item in the argument
     * list), uses @p defaultValue instead.
     *
     * @param option            Option identifier
     * @param defaultValue      Default value returned if the option is not
     *found.
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return Option value if found or @p defaultValue
    */
    template <class T>
    T parse(const std::string& option,
            const T& defaultValue,
            const std::string& desc = "");

    /**
     * Parse an option of type "option value" in the remaining argument list and
     *remove both the option and value from the list.
     * If the keyword "option" is not found, or if it is not followed by a value
     *(i.e. it is the last item in the argument
     * list), uses @p defaultValue instead.
     *
     * @param option            Option identifier
     * @param defaultValue      Default value returned if the option is not
     *found.
     * @param vmin              Minimum value allowed
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return Option value if found or @p defaultValue
    */
    template <class T>
    typename std::enable_if<std::is_arithmetic<T>::value, T>::type
    parse(const std::string& option,
          const T& defaultValue,
          const T& vmin,
          const std::string& desc = "");

    /**
     * Parse an option of type "option value" in the remaining argument list and
     *remove both the option and value from the list.
     * If the keyword "option" is not found, or if it is not followed by a value
     *(i.e. it is the last item in the argument
     * list), uses @p defaultValue instead.
     *
     * @param option            Option identifier
     * @param defaultValue      Default value returned if the option is not
     *found.
     * @param vmin              Minimum value allowed
     * @param vmax              Maximum value allowed
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return Option value if found or @p defaultValue
    */
    template <class T>
    T parse(const std::string& option,
            const T& defaultValue,
            const T& vmin,
            const T& vmax,
            const std::string& desc = "");

    /**
     * Parse an option of type "option value" in the remaining argument list and
     *remove both the option and value from the list.
     * If the keyword "option" is not found, or if it is not followed by a value
     *(i.e. it is the last item in the argument
     * list), uses @p defaultValue instead.
     *
     * @param option            Option identifier
     * @param defaultValue      Default value returned if the option is not
     *found.
     * @param allowed           Discrete list of allowed values for the option
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return Option value if found or @p defaultValue
    */
    template <class T>
    T parse(const std::string& option,
            const T& defaultValue,
            const std::vector<T>& allowed,
            const std::string& desc = "");

    /**
     * Grab the mandatory first element in the remaining argument list and
     *remove it from the list. If the list is empty,
     * displays an error; shows the help and exit the program when done() is
     *called.
     *
     * @param name              Option name, used for the help (autogenerated
     *"-h" command)
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return First element in the remaining argument list or T()
    */
    template <class T>
    T grab(const std::string& name, const std::string& desc = "");

    /**
     * Grab the first element in the remaining argument list and remove it from
     *the list. If the list is empty, uses
     * @p defaultValue instead.
     *
     * @param defaultValue      Default value returned if the argument list is
     *empty.
     * @param name              Option name, used for the help (autogenerated
     *"-h" command)
     * @param desc              Option description, used for the help
     *(autogenerated "-h" command)
     * @return First element in the remaining argument list or @p defaultValue
    */
    template <class T>
    T grab(const T& defaultValue,
           const std::string& name,
           const std::string& desc = "");

    std::map<std::string, std::string> grab(const std::string& prefix,
                                            const std::string& desc = "");

    /**
     * Check for unknown options, display help if asked and clean-up memory.
     * If the program is called with the "-h" option, this function display the
     *list of the available options and terminate the
     * program.
     *
     * @note The parse() and grab() functions must always be called before
     *calling this function.
    */
    void done();
    /// Destructor
    virtual ~ProgramOptions() {};

private:
    void displayHelp(const std::string& name,
                     const std::string& desc,
                     const std::string& defaultValue);

    bool mAbortOnError;
    bool mHelp;
    std::string mCommand;
    std::vector<std::string> mArgs;
    std::map<std::string, std::pair<std::string, std::string> > mOptions;
    std::vector<std::tuple<std::string, std::string, std::string> > mPosOptions;
};
}

template <class T>
T N2D2::ProgramOptions::parse(const std::string& option,
                              const T& defaultValue,
                              const std::string& desc)
{
    T value = defaultValue;
    std::vector<std::string>::iterator it
        = std::find(mArgs.begin(), mArgs.end(), option);

    if (it != mArgs.end() && it + 1 != mArgs.end()) {
        std::stringstream argVal(*(it + 1));
        Utils::signChecked<T>(argVal) >> value;
        mArgs.erase(it, it + 2);

        std::cout << "Option " << option << ": " << desc << " [" << value << "]"
                  << std::endl;
    }

    std::ostringstream valueStr;
    valueStr << defaultValue;

    bool newInsert;
    std::tie(std::ignore, newInsert) = mOptions.insert(
        std::make_pair(option, std::make_pair(desc, valueStr.str())));

    if (!newInsert)
        throw std::runtime_error("Option already parsed: " + option);

    return value;
}

template <class T>
typename std::enable_if<std::is_arithmetic<T>::value, T>::type
N2D2::ProgramOptions::parse(const std::string& option,
                            const T& defaultValue,
                            const T& vmin,
                            const std::string& desc)
{
    const T value = parse(option, defaultValue, desc);

    if (value < vmin) {
        std::cout << "Option " << option
                  << " value must be equal or greater than " << vmin
                  << std::endl;

        if (mAbortOnError)
            std::exit(0);

        return defaultValue;
    } else
        return value;
}

template <class T>
T N2D2::ProgramOptions::parse(const std::string& option,
                              const T& defaultValue,
                              const T& vmin,
                              const T& vmax,
                              const std::string& desc)
{
    const T value = parse(option, defaultValue, desc);

    if (value < vmin || value > vmax) {
        std::cout << "Option " << option
                  << " value not within permissible range [" << vmin << ":"
                  << vmax << "]" << std::endl;

        if (mAbortOnError)
            std::exit(0);

        return defaultValue;
    } else
        return value;
}

template <class T>
T N2D2::ProgramOptions::parse(const std::string& option,
                              const T& defaultValue,
                              const std::vector<T>& allowed,
                              const std::string& desc)
{
    const T value = parse(option, defaultValue, desc);

    if (std::find(allowed.begin(), allowed.end(), value) == allowed.end()) {
        std::cout << "Option " << option << " value not allowed [";
        std::copy(allowed.begin(),
                  allowed.end(),
                  std::ostream_iterator<T>(std::cout, ", "));
        std::cout << "]" << std::endl;

        if (mAbortOnError)
            std::exit(0);

        return defaultValue;
    } else
        return value;
}

template <class T>
T N2D2::ProgramOptions::grab(const std::string& name, const std::string& desc)
{
    if (mArgs.empty()) {
        std::cout << "Option " << name << " is required" << std::endl;
        mHelp = true;
    }

    return grab(T(), name, desc);
}

template <class T>
T N2D2::ProgramOptions::grab(const T& defaultValue,
                             const std::string& name,
                             const std::string& desc)
{
    T value = defaultValue;

    if (!mArgs.empty()) {
        std::stringstream argVal(mArgs[0]);

        if (!(argVal >> value) || !argVal.eof()) {
            std::cout << "Unreadable value for option " << name << std::endl;

            if (mAbortOnError)
                std::exit(0);
        }

        mArgs.erase(mArgs.begin());
    }

    std::ostringstream valueStr;
    valueStr << defaultValue;

    mPosOptions.push_back(std::make_tuple(name, desc, valueStr.str()));

    return value;
}

namespace N2D2 {
template <>
std::string ProgramOptions::grab<std::string>(const std::string& defaultValue,
                                              const std::string& name,
                                              const std::string& desc);
}

#endif // N2D2_PROGRAMOPTIONS_H
