/*
    (C) Copyright 2010 CEA LIST. All Rights Reserved.
    Contributor(s): Olivier BICHLER (olivier.bichler@cea.fr)
                    Damien QUERLIOZ (damien.querlioz@cea.fr)

    This software is governed by the CeCILL-C license under French law and
    abiding by the rules of distribution of free software.  You can  use,
    modify and/ or redistribute the software under the terms of the CeCILL-C
    license as circulated by CEA, CNRS and INRIA at the following URL
    "http://www.cecill.info".

    As a counterpart to the access to the source code and  rights to copy,
    modify and redistribute granted by the license, users are provided only
    with a limited warranty  and the software's author,  the holder of the
    economic rights,  and the successive licensors  have only  limited
    liability.

    The fact that you are presently reading this means that you have had
    knowledge of the CeCILL-C license and that you accept its terms.
*/

#ifndef N2D2_NODE_H
#define N2D2_NODE_H

#include <algorithm>
#include <utility>
#include <vector>

#include "Network.hpp"

namespace N2D2 {
/**
 * Description of a square area with an absolute position of (@p x,@p y) and a
 * size of (@p width,@p height).
 * Used to store the connectivity area of the nodes.
*/
struct Area {
    Area(unsigned short x_,
         unsigned short y_,
         unsigned short width_,
         unsigned short height_)
        : x(x_), y(y_), width(width_), height(height_) {};

    unsigned short x;
    unsigned short y;
    unsigned short width;
    unsigned short height;
};

/**
 * This is an abstract class for the one and only basic component type of the
 * network, a node.
 * There is however several type of nodes:
 * - NodeEnv: input node, receives external stimuli. It is not a real neuron as
 * it has no input or integration circuit.
 * - NodeNeuron: as the name suggests it, it's a neuron.
 * - NodeSync: this is a very special node type, which is used to emulate
 * pipelining and synchronization between the layers of
 * the network, in the currently frame-based proposed architecture.
*/
class Node : public NetworkObserver {
public:
    Node(Network& net);
    void addBranch(Node* branch);
    void removeBranch(Node* branch);

    /**
     * Method called from the input nodes to propagate incoming event(s) through
     *the synaptic connection. Input nodes never
     * call incomingSpike() directly, instead, they must call propagateSpike(),
     *which will generate an event that will be passed
     * to incomingSpike().
     * This method is not intended to be called directly.
     *
     * @param origin        Input node address
     * @param timestamp     Timestamp of the event
     * @param type          Event type
    */
    inline virtual void
    propagateSpike(Node* origin, Time_T timestamp, EventType_T type = 0);

    /**
     * Method receiving the input events of the node, after having passed
     *through the synaptic connection. These events were
     * generated by propagateSpike().
     * This method is not intended to be called directly.
     *
     * @param link          Input node address
     * @param timestamp     Timestamp of the event
     * @param type          Event type
    */
    virtual void
    incomingSpike(Node* link, Time_T timestamp, EventType_T type = 0) = 0;

    /**
     * Method handling internal events of the node and generating output events.
     * This method is not intended to be called directly.
     *
     * @param timestamp     Timestamp of the event
     * @param type          Event type
    */
    virtual void emitSpike(Time_T timestamp, EventType_T type = 0);

    inline virtual void notify(Time_T timestamp, NotifyType notify);

    /// Enable or disable activity recording for this node (used in Monitor).
    void setActivityRecording(bool activityRecording)
    {
        mActivityRecording = activityRecording;
    };
    /// Returns last activation time of the node
    Time_T getLastActivationTime() const
    {
        return mLastActivationTime;
    };

    /**
     * Get the activity of the node between time @p start and time @p stop, for
     *event type @p type.
     *
     * @param start         Start time
     * @param stop          Stop time
     * @param type          Event type
     * @return Number of corresponding activations of the node
    */
    unsigned int
    getActivity(Time_T start = 0, Time_T stop = 0, EventType_T type = 0) const;

    /**
     * Get the first activation time of the node between time @p start and time
     *@p stop, for event type @p type.
     *
     * @param start         Start time
     * @param stop          Stop time
     * @param type          Event type
     * @return std::pair, first element is the activation time, second element
     *indicates if it is valid (if the node activated)
    */
    std::pair<Time_T, bool> getFirstActivationTime(Time_T start = 0,
                                                   Time_T stop = 0,
                                                   EventType_T type = 0) const;
    /// Returns the scale from the map associated to the node
    float getScale() const
    {
        return mScale;
    };
    /// Returns the orientation of the filter associated with the node
    float getOrientation() const
    {
        return mOrientation;
    };
    /// Returns the connectivity area of the node
    const Area& getArea() const
    {
        return mArea;
    };
    /// Returns node unique ID
    NodeId_T getId() const
    {
        return mId;
    };
    /// Returns node layer number
    unsigned short getLayer() const
    {
        return mLayer;
    };
    /// Returns node branches
    const std::vector<Node*>& getBranches()
    {
        return mBranches;
    };
    /// Destructor
    virtual ~Node() {};

protected:
    // Parameters
    /// Indicates if activity recording for this node is enabled or not
    bool mActivityRecording;

    // Internal variables
    /// Node unique ID
    const NodeId_T mId;
    /// Branches of the node
    std::vector<Node*> mBranches;
    Time_T mLastActivationTime;
    float mScale;
    float mOrientation;
    unsigned short mLayer;
    Area mArea;

    static unsigned int mIdCnt;

private:
    // A Node has an unique ID and is therefore non-copyable.
    Node(const Node&); // non construction-copyable
    const Node& operator=(const Node&); // non-copyable
};
}

void
N2D2::Node::propagateSpike(Node* origin, Time_T timestamp, EventType_T type)
{
    incomingSpike(origin, timestamp, type);
}

void N2D2::Node::notify(Time_T /*timestamp*/, NotifyType notify)
{
    if (notify == Reset)
        mLastActivationTime = 0; // 0 means it was never activated, which also
    // allows activation checking
}

#endif // N2D2_NODE_H
