# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['wtforms_bootstrap5']

package_data = \
{'': ['*']}

install_requires = \
['WTForms>=3.0.1,<4.0.0']

setup_kwargs = {
    'name': 'wtforms-bootstrap5',
    'version': '0.1.3',
    'description': 'Simple library for rendering WTForms in HTML as Bootstrap 5 form controls',
    'long_description': '# wtforms-bootstrap5\nSimple library for rendering WTForms in HTML as Bootstrap 5 form controls\n\n**Notice: this project is still in very early stage, the API may change a lots rapidly**\n\n## Features\n\n- **MIT licensed** - it doesn\'t infect your code\n- **Light weight** - not much code and little dependencies\n- **Latest Bootstrap 5** - generates forms in latest Bootstrap 5 style \n- **High customizable** - you can generate different kind of Bootstrap 5 form controls and layouts\n- **Template engine friendly** - chained method calls making it easy to integrate with template engine\n- **Covered with automatic tests** - yep, we have test cases\n\n## Why?\n\nEverytime I build a website with [WTForms](https://wtforms.readthedocs.io), I spend way too much time in writing HTML and [Jinja template](https://jinja.palletsprojects.com/) for rendering a form as [Bootstrap 5 form controls](https://getbootstrap.com/docs/5.2/forms/overview/).\nWork smart is an important value we have here at [Launch Platform](https://launchplatform.com), so I wonder why not make a library for making rendering Bootstrap 5 style WTForms controls easily?\nSo here you go, wtforms-bootstrap5 is created, open sourced under MIT license.\nIt\'s a simple Python library for rendering WTForms in Bootstrap 5 favor.\n\n## Install\n\nTo install the formatter, simply run\n\n```bash\npip install wtforms-bootstrap5\n```\n\n## Example\n\nFirst, you define your form as you would usually do with WTForms:\n\n```python\nfrom wtforms import Form\nfrom wtforms import EmailField\nfrom wtforms import PasswordField\nfrom wtforms import SelectField\nfrom wtforms import BooleanField\nfrom wtforms import SubmitField\n\n\nclass MyForm(Form):\n    email = EmailField("Email", render_kw=dict(placeholder="Foobar"))\n    password = PasswordField("Password", description="Your super secret password")\n    city = SelectField("City", choices=["Los Angle", "San Francisco", "New York"])\n    agree_terms = BooleanField("I agrees to terms and service")\n    submit = SubmitField()\n\n```\n\nThen you can use `RenderContext` for rendering your form like this\n\n```python\nfrom wtforms_bootstrap5 import RendererContext\n\nform = MyForm()\ncontext = RendererContext()\nhtml = context.render(form)\n```\n\nThe form will be rendered as HTML like\n\n```html\n<form method="POST"><div class="mb-3"><label class="form-label" for="email">Email</label><input class="form-control" id="email" name="email" type="email" value=""></div>\n<div class="mb-3"><label class="form-label" for="password">Password</label><input class="form-control" id="password" name="password" type="password" value=""><div class="form-text">Your super secret password</div></div>\n<div class="mb-3"><label class="form-label" for="city">City</label><select class="form-select" id="city" name="city"><option value="Los Angle">Los Angle</option><option value="San Francisco">San Francisco</option><option value="New York">New York</option></select></div>\n<div class="mb-3"><div class="form-check"><label class="form-check-label" for="agree_terms">I agrees to terms and service</label><input class="form-check-input" id="agree_terms" name="agree_terms" type="checkbox" value="y"></div></div>\n<div class="mb-3"><input class="btn btn-primary" id="submit" name="submit" type="submit" value="Submit"></div></form>\n```\n\nAnd it will look like this\n\n<p align="center">\n  <img src="assets/default-style-example.png?raw=true" alt="Form rendered in Bootstrap 5 favor" />\n</p>\n\nBy default, a sensible simple layout style will be used.\n\n## Customize the form\n\nThere are many similar open source libraries out there, but most of them are very hard to customize.\nOnce you adopt it, then you can only render your form in a specific style.\nAs a result, I found myself writing HTML manually without using the library to save time.\n\nTo avoid the same mistake, we want to make wtforms-bootstrap5 very easy to customize without compromising too much of its reusability.\nHere\'s an example how you can turn the example above into a column based form.\n\n```python\nhtml = (\n    renderer_context\n    .form(action="/sign-up")\n    .default_field(\n        row_class="row mb-3",\n        label_class="form-label col-2",\n        field_wrapper_class="col-10",\n        field_wrapper_enabled=True,\n    )\n    .field(\n        "agree_terms",\n        wrapper_class="offset-2",\n        wrapper_enabled=True,\n        field_wrapper_enabled=False,\n    )\n    .field(\n        "submit",\n        field_wrapper_class="offset-2",\n        field_wrapper_enabled=True,\n    )\n).render(form)\n```\n\nAnd this is how it looks like\n\n<p align="center">\n  <img src="assets/column-style-example.png?raw=true" alt="Form rendered in Bootstrap 5 favor" />\n</p>\n\nAs you can see in the example, we use `default_field` method for overwriting the options of all fields by default.\nWe also use `field` method for overwriting the options for a specific field.\nThe `field` method takes multiple input name arguments, so that you can overwrite options for multiple fields at once like this\n\n```python\nhtml = (context\n    .field("email", "password", label_class="my-custom-class", ...)\n)\n```\n\nPlease notice that, **the order of `default_field` and `field` method calls matter**.\nWhen `field` is called, the current default field options will be used as the default values.\nSo if you make the calls in order like this\n\n```python\nhtml = (context\n    .field("email", row_class="row")\n    .default_field(label_class="my-custom-class")\n)\n```\n\nThe `label_class` for `email` field here will be `form-label` instead of `my-custom-class` since when it\'s called, the original default value was still `form-label`.\n\nTo customize the form element, you can use the `form` method like this\n\n```python\nhtml = (context\n    .form(\n        method="POST",\n        action="/sign-up",\n        enctype="application/x-www-form-urlencoded",\n        form_class="my-form",\n        form_attrs=dict(custom="value")\n    )\n)\n```\n\n### Field HTML structure\n\nIn general, the field HTML structure can be controlled by the option values and it looks like this\n\n```html\n<!-- enabled by .row_enabled, default: true -->\n<div class=".row_class" {.row_attrs}>\n  <!-- enabled by .wrapper_enabled, default: false -->\n  <div class=".wrapper_class" {.wrapper_attrs}>\n    <!-- enabled by .label_enabled, default: true -->\n    <label class=".label_class" for="email" {.label_attrs}>Email</label>\n    <!-- enabled by .field_wrapper_enabled, default: false -->\n    <div class=".field_wrapper" {.field_wrapper_attrs}>\n      <input class=".field_class" id="email" name="email" type="email" value="" {.field_attrs}>\n      <!-- enabled by .help_enabled, default: true -->\n      <div class=".help_class" {.helper_attrs}>Your super secret password</div>\n      <!-- enabled by .error_enabled, default: true -->\n      <div class=".error_class" {.error_attrs}>Bad password</div>\n    </div>\n  </div>\n</div>\n```\n\n## Integrate with template engine\n\nWe want to make it as easy as possible to integrate with template engine such as [Jinja](https://jinja.palletsprojects.com/).\nThat\'s why we use chained method calls for customizing the form.\nYou should be able to pass the `form` and `RenderContext` objects and write all your form customization from the template.\nThis way, you don\'t get your view relative code pollute your controller code.\nFor example, after passing `form` and `render_context` object, you can write this in Jinja:\n\n```html\n<h1>New user</h1>\n\n{{\n    renderer_context\n        .default_field(\n            row_class="row mb-3",\n            label_class="form-label col-2",\n            field_wrapper_class="col-10",\n            field_wrapper_enabled=True,\n        )\n        .field(\n            "agree_terms",\n            wrapper_class="offset-2",\n            wrapper_enabled=True,\n            field_wrapper_enabled=False,\n        )\n        .field(\n            "submit",\n            field_wrapper_class="offset-2",\n            field_wrapper_enabled=True,\n        )\n    ).render(form)\n}}\n```\n\n## Feedbacks\n\nFeedbacks, bugs reporting or feature requests are welcome 🙌, just please open an issue.\nNo guarantee we have time to deal with them, but will see what we can do.\n',
    'author': 'Fang-Pen Lin',
    'author_email': 'fangpen@launchplatform.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/LaunchPlatform/wtforms-bootstrap5',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
