"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrowApi = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const lambda = require("aws-cdk-lib/aws-lambda");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const logs = require("aws-cdk-lib/aws-logs");
const fse = require("fs-extra");
/**
 * @experimental
 */
class CrowApi extends constructs_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Pulling out props
        const { sourceDirectory = 'src', sharedDirectory = 'shared', useAuthorizerLambda = false, authorizerDirectory = 'authorizer', authorizerLambdaConfiguration = {}, tokenAuthorizerConfiguration = {}, createApiKey = false, logRetention = logs.RetentionDays.ONE_WEEK, apiGatewayConfiguration = {}, apiGatewayName = 'crow-api', lambdaConfigurations = {}, lambdaIntegrationOptions = {}, models = [], requestValidators = [], methodConfigurations = {}, } = props;
        // Initializing constants
        const LAMBDA_RUNTIME = lambda.Runtime.NODEJS_14_X;
        const SPECIAL_DIRECTORIES = [
            sharedDirectory,
            authorizerDirectory,
        ];
        // Helpers functions for constructor
        // Prepares default Lambda props and overrides them with user input
        function bundleLambdaProps(codePath, userConfiguration, sharedLayer) {
            let layers;
            if (sharedLayer) {
                const { layers: userLayers = [], } = userConfiguration;
                layers = [sharedLayer, ...userLayers];
            }
            const defaultProps = {
                runtime: LAMBDA_RUNTIME,
                code: lambda.Code.fromAsset(codePath),
                handler: 'index.handler',
                logRetention,
            };
            const lambdaProps = {
                ...defaultProps,
                ...userConfiguration,
                layers,
            };
            return lambdaProps;
        }
        // Returns child directories given the path of a parent
        function getDirectoryChildren(parentDirectory) {
            try {
                const directories = fse.readdirSync(parentDirectory, { withFileTypes: true })
                    .filter((dirent) => dirent.isDirectory())
                    .map((dirent) => dirent.name);
                return directories;
            }
            catch {
                /**
                 * The only time I have run into this was when the src/ directory
                 * was empty.
                 * If it is empty, let CDK tree validation tell user that the
                 * REST API does not have any methods.
                 */
            }
            return [];
        }
        // API Gateway log group
        const gatewayLogGroup = new logs.LogGroup(this, 'api-access-logs', {
            retention: logs.RetentionDays.ONE_WEEK,
        });
        // The API Gateway itself
        const gateway = new apigateway.RestApi(this, apiGatewayName, {
            deploy: true,
            deployOptions: {
                loggingLevel: apigateway.MethodLoggingLevel.ERROR,
                accessLogDestination: new apigateway.LogGroupLogDestination(gatewayLogGroup),
            },
            apiKeySourceType: createApiKey ? apigateway.ApiKeySourceType.HEADER : undefined,
            ...apiGatewayConfiguration,
        });
        const createdModels = {};
        models.forEach((model) => {
            // modelName is used as ID and can now be used for referencing model in method options
            createdModels[model.modelName] = gateway.addModel(model.modelName, model);
        });
        const createdRequestValidators = {};
        requestValidators.forEach((requestValidator) => {
            // requestValidatorName is used as ID and can now be used for referencing model in method options
            createdRequestValidators[requestValidator.requestValidatorName] = gateway.addRequestValidator(requestValidator.requestValidatorName, requestValidator);
        });
        // Create API key if desired
        if (createApiKey) {
            const apiKey = gateway.addApiKey('api-key');
            const usagePlan = new apigateway.UsagePlan(this, 'usage-plan', {
                throttle: {
                    burstLimit: 5000,
                    rateLimit: 10000,
                },
                apiStages: [
                    {
                        api: gateway,
                        stage: gateway.deploymentStage,
                    },
                ],
            });
            usagePlan.addApiKey(apiKey);
            this.usagePlan = usagePlan;
        }
        // Create Lambda layer out of shared directory if it exists
        const sourceSharedDirectory = `${sourceDirectory}/${sharedDirectory}`;
        let sharedLayer;
        if (fse.existsSync(sourceSharedDirectory)) {
            sharedLayer = new lambda.LayerVersion(this, 'shared-layer', {
                code: lambda.Code.fromAsset(sourceSharedDirectory),
                compatibleRuntimes: [LAMBDA_RUNTIME],
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            this.lambdaLayer = sharedLayer;
        }
        // Create Lambda authorizer to be used in subsequent Methods
        let tokenAuthorizer;
        if (useAuthorizerLambda) {
            const fullAuthorizerDirectory = `${sourceDirectory}/${authorizerDirectory}`;
            const authorizerLambdaProps = bundleLambdaProps(fullAuthorizerDirectory, authorizerLambdaConfiguration, sharedLayer);
            const authorizerLambda = new lambda.Function(this, 'authorizer-lambda', authorizerLambdaProps);
            this.authorizerLambda = authorizerLambda;
            const bundledTokenAuthConfig = {
                handler: authorizerLambda,
                resultsCacheTtl: cdk.Duration.seconds(3600),
                ...tokenAuthorizerConfiguration,
            };
            tokenAuthorizer = new apigateway.TokenAuthorizer(this, 'token-authorizer', bundledTokenAuthConfig);
            this.authorizer = tokenAuthorizer;
        }
        // Time to start walking the directories
        const root = sourceDirectory;
        const verbs = ['get', 'post', 'put', 'delete'];
        const graph = {};
        const lambdasByPath = {};
        // Initialize with root
        graph['/'] = {
            resource: gateway.root,
            path: root,
            paths: [],
            verbs: [],
        };
        // First element in tuple is directory path, second is API path
        const nodes = [[root, '/']];
        // BFS that creates API Gateway structure using addMethod
        while (nodes.length) {
            // The `|| ['type', 'script']` piece is needed or TS throws a fit
            const [directoryPath, apiPath] = nodes.shift() || ['type', 'script'];
            const children = getDirectoryChildren(directoryPath);
            // For debugging purposes
            // console.log(`${apiPath}'s children are: ${children}`);
            // Don't have to worry about previously visited nodes
            // since this is a file structure
            // ...unless there are symlinks? Haven't run into that
            children.forEach((child) => {
                const newDirectoryPath = `${directoryPath}/${child}`;
                // If we're on the root path, don't separate with a slash (/)
                //   because it ends up looking like //child-path
                const newApiPath = apiPath === '/' ? `/${child}` : `${apiPath}/${child}`;
                if (verbs.includes(child)) {
                    // If directory is a verb, we don't traverse it anymore
                    //   and need to create an API Gateway method and Lambda
                    const userLambdaConfiguration = lambdaConfigurations[newApiPath]
                        || {};
                    const lambdaProps = bundleLambdaProps(newDirectoryPath, userLambdaConfiguration, sharedLayer);
                    const newLambda = new lambda.Function(this, newDirectoryPath, lambdaProps);
                    // Pull out useAuthorizerLambda value and the tweaked model values
                    const { useAuthorizerLambda: authorizerLambdaConfigured = false, requestModels: crowRequestModels, methodResponses: crowMethodResponses, requestValidator: requestValidatorString, ...userMethodConfiguration } = methodConfigurations[newApiPath] || {};
                    let bundledMethodConfiguration = {
                        ...userMethodConfiguration,
                    };
                    // Map models
                    const requestModels = {};
                    if (crowRequestModels) {
                        Object.entries(crowRequestModels).forEach(([contentType, modelName]) => {
                            requestModels[contentType] = createdModels[modelName];
                        });
                    }
                    const methodResponses = [];
                    if (crowMethodResponses && crowMethodResponses.length > 0) {
                        crowMethodResponses.forEach((crowMethodResponse) => {
                            const responseModels = {};
                            if (crowMethodResponse.responseModels) {
                                const crowResponseModels = crowMethodResponse.responseModels;
                                Object.entries(crowResponseModels).forEach(([contentType, modelName]) => {
                                    responseModels[contentType] = createdModels[modelName];
                                });
                            }
                            const { statusCode, responseParameters, } = crowMethodResponse;
                            methodResponses.push({
                                statusCode,
                                responseParameters,
                                responseModels,
                            });
                        });
                    }
                    // Find request validator
                    if (requestValidatorString
                        && createdRequestValidators[requestValidatorString]) {
                        bundledMethodConfiguration.requestValidator = createdRequestValidators[requestValidatorString];
                    }
                    bundledMethodConfiguration.requestModels = requestModels;
                    bundledMethodConfiguration.methodResponses = methodResponses;
                    // If this method should be behind an authorizer Lambda
                    //   construct the methodConfiguration object as such
                    if (authorizerLambdaConfigured && useAuthorizerLambda) {
                        bundledMethodConfiguration.authorizationType = apigateway.AuthorizationType.CUSTOM;
                        bundledMethodConfiguration.authorizer = tokenAuthorizer;
                    }
                    const integrationOptions = lambdaIntegrationOptions[newApiPath] || {};
                    graph[apiPath].resource.addMethod(child.toUpperCase(), new apigateway.LambdaIntegration(newLambda, integrationOptions), bundledMethodConfiguration);
                    graph[apiPath].verbs.push(child);
                    lambdasByPath[newApiPath] = newLambda;
                }
                else if (SPECIAL_DIRECTORIES.includes(child)) {
                    // The special directories should not result in an API path
                    // This means the API also cannot have a resource with the
                    //   same name
                }
                else {
                    // If directory is not a verb, create new API Gateway resource
                    //   for use by verb directory later
                    const newResource = graph[apiPath].resource
                        .resourceForPath(child);
                    nodes.push([newDirectoryPath, newApiPath]);
                    // Add child to parent's paths
                    graph[apiPath].paths.push(child);
                    // Initialize graph node to include child
                    graph[newApiPath] = {
                        resource: newResource,
                        path: newDirectoryPath,
                        paths: [],
                        verbs: [],
                    };
                }
            });
        }
        // For debugging purposes
        // console.log(graph);
        // Expose API Gateway
        this.gateway = gateway;
        this.lambdaFunctions = lambdasByPath;
        this.models = createdModels;
        this.requestValidators = createdRequestValidators;
    }
}
exports.CrowApi = CrowApi;
_a = JSII_RTTI_SYMBOL_1;
CrowApi[_a] = { fqn: "crow-api.CrowApi", version: "2.4.0" };
//# sourceMappingURL=data:application/json;base64,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