"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Mode = exports.StaticSiteAuthorization = exports.SpaAuthorization = exports.Authorization = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const core_1 = require("@aws-cdk/core");
const cdk_lambda_at_edge_pattern_1 = require("@cloudcomponents/cdk-lambda-at-edge-pattern");
const auth_flow_1 = require("./auth-flow");
const retrieve_user_pool_client_secret_1 = require("./retrieve-user-pool-client-secret");
const secret_generator_1 = require("./secret-generator");
const user_pool_client_redirects_1 = require("./user-pool-client-redirects");
const user_pool_domain_1 = require("./user-pool-domain");
class Authorization extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.userPool = props.userPool;
        this.redirectPaths = (_a = props.redirectPaths) !== null && _a !== void 0 ? _a : {
            signIn: '/parseauth',
            authRefresh: '/refreshauth',
            signOut: '/',
        };
        this.signOutUrlPath = (_b = props.signOutUrl) !== null && _b !== void 0 ? _b : '/signout';
        this.httpHeaders = (_c = props.httpHeaders) !== null && _c !== void 0 ? _c : {
            'Content-Security-Policy': "default-src 'none'; img-src 'self'; script-src 'self'; style-src 'self' 'unsafe-inline'; object-src 'none'; connect-src 'self'",
            'Strict-Transport-Security': 'max-age=31536000; includeSubdomains; preload',
            'Referrer-Policy': 'same-origin',
            'X-XSS-Protection': '1; mode=block',
            'X-Frame-Options': 'DENY',
            'X-Content-Type-Options': 'nosniff',
            'Cache-Control': 'no-cache',
        };
        this.oauthScopes = (_d = props.oauthScopes) !== null && _d !== void 0 ? _d : [aws_cognito_1.OAuthScope.PHONE, aws_cognito_1.OAuthScope.EMAIL, aws_cognito_1.OAuthScope.PROFILE, aws_cognito_1.OAuthScope.OPENID, aws_cognito_1.OAuthScope.COGNITO_ADMIN];
        this.cookieSettings = props.cookieSettings;
        this.userPoolClient = this.createUserPoolClient();
        this.nonceSigningSecret = this.generateNonceSigningSecret();
        this.cognitoAuthDomain = this.retrieveCognitoAuthDomain();
        this.authFlow = this.createAuthFlow((_e = props.logLevel) !== null && _e !== void 0 ? _e : cdk_lambda_at_edge_pattern_1.LogLevel.WARN);
    }
    updateUserPoolClientCallbacks(redirects) {
        const { callbackUrls, logoutUrls } = redirects;
        new user_pool_client_redirects_1.UserPoolClientRedirects(this, 'UserPoolClientRedirects', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            callbackUrls,
            logoutUrls,
        });
    }
    createDefaultBehavior(origin, options) {
        return {
            origin,
            forwardQueryString: true,
            compress: true,
            viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
            edgeLambdas: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createLegacyDefaultBehavior(options) {
        return {
            isDefaultBehavior: true,
            forwardedValues: {
                queryString: true,
            },
            compress: true,
            lambdaFunctionAssociations: [this.authFlow.checkAuth, this.authFlow.httpHeaders],
            ...options,
        };
    }
    createAdditionalBehaviors(origin, options) {
        return {
            [this.redirectPaths.signIn]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.parseAuth],
                ...options,
            },
            [this.redirectPaths.authRefresh]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.refreshAuth],
                ...options,
            },
            [this.signOutUrlPath]: {
                origin,
                forwardQueryString: true,
                compress: true,
                viewerProtocolPolicy: aws_cloudfront_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
                edgeLambdas: [this.authFlow.signOut],
                ...options,
            },
        };
    }
    createLegacyAdditionalBehaviors(options) {
        return [
            {
                pathPattern: this.redirectPaths.signIn,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.parseAuth],
                ...options,
            },
            {
                pathPattern: this.redirectPaths.authRefresh,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.refreshAuth],
                ...options,
            },
            {
                pathPattern: this.signOutUrlPath,
                forwardedValues: {
                    queryString: true,
                },
                lambdaFunctionAssociations: [this.authFlow.signOut],
                ...options,
            },
        ];
    }
    generateNonceSigningSecret() {
        const { secret } = new secret_generator_1.SecretGenerator(this, 'SecretGenerator');
        return secret;
    }
    retrieveCognitoAuthDomain() {
        const userPoolDomain = new user_pool_domain_1.UserPoolDomain(this, 'UserPoolDomain', {
            userPool: this.userPool,
        });
        return userPoolDomain.cognitoAuthDomain;
    }
}
exports.Authorization = Authorization;
class SpaAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.SPA;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: false,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; SameSite=Lax',
                accessToken: 'Path=/; Secure; SameSite=Lax',
                refreshToken: 'Path=/; Secure; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
}
exports.SpaAuthorization = SpaAuthorization;
class StaticSiteAuthorization extends Authorization {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.mode = Mode.STATIC_SITE;
    }
    createUserPoolClient() {
        return this.userPool.addClient('UserPoolClient', {
            generateSecret: true,
            oAuth: {
                flows: {
                    authorizationCodeGrant: true,
                },
                scopes: this.oauthScopes,
            },
            supportedIdentityProviders: [aws_cognito_1.UserPoolClientIdentityProvider.COGNITO],
            preventUserExistenceErrors: true,
        });
    }
    createAuthFlow(logLevel) {
        var _a;
        const clientSecret = this.retrieveUserPoolClientSecret();
        return new auth_flow_1.AuthFlow(this, 'AuthFlow', {
            logLevel,
            httpHeaders: this.httpHeaders,
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
            oauthScopes: this.oauthScopes,
            redirectPaths: this.redirectPaths,
            nonceSigningSecret: this.nonceSigningSecret,
            cognitoAuthDomain: this.cognitoAuthDomain,
            clientSecret,
            cookieSettings: (_a = this.cookieSettings) !== null && _a !== void 0 ? _a : {
                idToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                accessToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                refreshToken: 'Path=/; Secure; HttpOnly; SameSite=Lax',
                nonce: 'Path=/; Secure; HttpOnly; SameSite=Lax',
            },
        });
    }
    retrieveUserPoolClientSecret() {
        const { clientSecret } = new retrieve_user_pool_client_secret_1.RetrieveUserPoolClientSecret(this, 'RetrieveUserPoolClientSecret', {
            userPool: this.userPool,
            userPoolClient: this.userPoolClient,
        });
        return clientSecret;
    }
}
exports.StaticSiteAuthorization = StaticSiteAuthorization;
var Mode;
(function (Mode) {
    Mode["SPA"] = "SPA";
    Mode["STATIC_SITE"] = "STATIC_SITE";
})(Mode = exports.Mode || (exports.Mode = {}));
//# sourceMappingURL=data:application/json;base64,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