"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LustreFileSystem = exports.LustreDeploymentType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const file_system_1 = require("./file-system");
const fsx_generated_1 = require("./fsx.generated");
/**
 * The different kinds of file system deployments used by Lustre.
 *
 * @stability stable
 */
var LustreDeploymentType;
(function (LustreDeploymentType) {
    LustreDeploymentType["SCRATCH_1"] = "SCRATCH_1";
    LustreDeploymentType["SCRATCH_2"] = "SCRATCH_2";
    LustreDeploymentType["PERSISTENT_1"] = "PERSISTENT_1";
})(LustreDeploymentType = exports.LustreDeploymentType || (exports.LustreDeploymentType = {}));
/**
 * The FSx for Lustre File System implementation of IFileSystem.
 *
 * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-fsx-filesystem.html
 * @stability stable
 * @resource AWS::FSx::FileSystem
 */
class LustreFileSystem extends file_system_1.FileSystemBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.validateProps(props);
        const updatedLustureProps = {
            importedFileChunkSize: props.lustreConfiguration.importedFileChunkSizeMiB,
            weeklyMaintenanceStartTime: (_b = props.lustreConfiguration.weeklyMaintenanceStartTime) === null || _b === void 0 ? void 0 : _b.toTimestamp(),
        };
        const lustreConfiguration = Object.assign({}, props.lustreConfiguration, updatedLustureProps);
        const securityGroup = (props.securityGroup || new aws_ec2_1.SecurityGroup(this, 'FsxLustreSecurityGroup', {
            vpc: props.vpc,
        }));
        securityGroup.addIngressRule(securityGroup, aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort));
        this.connections = LustreFileSystem.configureConnections(securityGroup);
        this.fileSystem = new fsx_generated_1.CfnFileSystem(this, 'Resource', {
            fileSystemType: LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE,
            subnetIds: [props.vpcSubnet.subnetId],
            backupId: props.backupId,
            kmsKeyId: (props.kmsKey ? props.kmsKey.keyId : undefined),
            lustreConfiguration,
            securityGroupIds: [securityGroup.securityGroupId],
            storageCapacity: props.storageCapacityGiB,
        });
        this.fileSystem.applyRemovalPolicy(props.removalPolicy);
        this.fileSystemId = this.fileSystem.ref;
        this.dnsName = `${this.fileSystemId}.fsx.${this.stack.region}.${core_1.Aws.URL_SUFFIX}`;
        this.mountName = this.fileSystem.attrLustreMountName;
    }
    /**
     * Import an existing FSx for Lustre file system from the given properties.
     *
     * @stability stable
     */
    static fromLustreFileSystemAttributes(scope, id, attrs) {
        class Import extends file_system_1.FileSystemBase {
            constructor() {
                super(...arguments);
                this.dnsName = attrs.dnsName;
                this.fileSystemId = attrs.fileSystemId;
                this.connections = LustreFileSystem.configureConnections(attrs.securityGroup);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Configures a Connections object with all the ports required by FSx for Lustre
     */
    static configureConnections(securityGroup) {
        const connections = new aws_ec2_1.Connections({
            securityGroups: [securityGroup],
            defaultPort: aws_ec2_1.Port.tcpRange(LustreFileSystem.DEFAULT_PORT_RANGE.startPort, LustreFileSystem.DEFAULT_PORT_RANGE.endPort),
        });
        return connections;
    }
    /**
     * Validates the props provided for a new FSx for Lustre file system.
     */
    validateProps(props) {
        const lustreConfiguration = props.lustreConfiguration;
        const deploymentType = lustreConfiguration.deploymentType;
        // Make sure the import path is valid before validating the export path
        this.validateImportPath(lustreConfiguration.importPath);
        this.validateExportPath(lustreConfiguration.exportPath, lustreConfiguration.importPath);
        this.validateImportedFileChunkSize(lustreConfiguration.importedFileChunkSizeMiB);
        this.validatePerUnitStorageThroughput(deploymentType, lustreConfiguration.perUnitStorageThroughput);
        this.validateStorageCapacity(deploymentType, props.storageCapacityGiB);
    }
    /**
     * Validates the export path is in the correct format and matches the import path.
     */
    validateExportPath(exportPath, importPath) {
        if (exportPath === undefined) {
            return;
        }
        if (importPath === undefined) {
            throw new Error('Cannot define an export path without also defining an import path');
        }
        if (core_1.Token.isUnresolved(exportPath) && core_1.Token.isUnresolved(importPath)) {
            return;
        }
        if (core_1.Token.isUnresolved(importPath) !== core_1.Token.isUnresolved(exportPath)) {
            throw new Error('The importPath and exportPath must each be Tokens or not Tokens, you cannot use a mix');
        }
        if (!exportPath.startsWith(importPath)) {
            throw new Error(`The export path "${exportPath}" is invalid. Expecting the format: s3://{IMPORT_PATH}/optional-prefix`);
        }
        if (exportPath.length > 900) {
            throw new Error(`The export path "${exportPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the importedFileChunkSize is in the correct range.
     */
    validateImportedFileChunkSize(importedFileChunkSize) {
        if (importedFileChunkSize === undefined) {
            return;
        }
        if (importedFileChunkSize < 1 || importedFileChunkSize > 512000) {
            throw new Error(`importedFileChunkSize cannot be ${importedFileChunkSize} MiB. It must be a value from 1 to 512,000 MiB`);
        }
    }
    /**
     * Validates the import path is the correct format.
     */
    validateImportPath(importPath) {
        if (importPath === undefined || core_1.Token.isUnresolved(importPath)) {
            return;
        }
        const regexp = /^s3:\/\//;
        if (importPath.search(regexp) === -1) {
            throw new Error(`The import path "${importPath}" is invalid. Expecting the format: s3://{BUCKET_NAME}/optional-prefix`);
        }
        if (importPath.length > 900) {
            throw new Error(`The import path "${importPath}" exceeds the maximum length of 900 characters`);
        }
    }
    /**
     * Validates the perUnitStorageThroughput is defined correctly for the given deploymentType.
     */
    validatePerUnitStorageThroughput(deploymentType, perUnitStorageThroughput) {
        if (perUnitStorageThroughput === undefined) {
            return;
        }
        if (deploymentType !== LustreDeploymentType.PERSISTENT_1) {
            throw new Error('perUnitStorageThroughput can only be set for the PERSISTENT_1 deployment type');
        }
        if (![50, 100, 200].includes(perUnitStorageThroughput)) {
            throw new Error('perUnitStorageThroughput must be 50, 100, or 200 MB/s/TiB');
        }
    }
    /**
     * Validates the storage capacity is an acceptable value for the deployment type.
     */
    validateStorageCapacity(deploymentType, storageCapacity) {
        if (deploymentType === LustreDeploymentType.SCRATCH_1) {
            if (![1200, 2400, 3600].includes(storageCapacity) && storageCapacity % 3600 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, 3,600, or a multiple of 3,600');
            }
        }
        else {
            if (![1200, 2400].includes(storageCapacity) && storageCapacity % 2400 !== 0) {
                throw new Error('storageCapacity must be 1,200, 2,400, or a multiple of 2,400');
            }
        }
    }
}
exports.LustreFileSystem = LustreFileSystem;
_a = JSII_RTTI_SYMBOL_1;
LustreFileSystem[_a] = { fqn: "@aws-cdk/aws-fsx.LustreFileSystem", version: "1.128.0" };
/**
 * The default FSx file system type used by FSx for Lustre.
 */
LustreFileSystem.DEFAULT_FILE_SYSTEM_TYPE = 'LUSTRE';
/**
 * The default ports the file system listens on. Actual port list is: [988, 1021, 1022, 1023]
 */
LustreFileSystem.DEFAULT_PORT_RANGE = { startPort: 988, endPort: 1023 };
//# sourceMappingURL=data:application/json;base64,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