import os
import xml.etree.ElementTree as ET
from datetime import datetime

from rich.console import Console

from .vars import ignore_file_content, logger
from .vars import PROJECT_FILENAME, WATCH_FILENAME, IGNORE_FILENAME
from .vars import project_file_path, watch_file_path
from .vars import DEFAULT_DATETIME_FORMAT, DEFAULT_DATE_FORMAT
from .exceptions import ProjectNotSetupCorrectlyError


def build_project(data: dict, path):
    files = [
        os.path.join(path, PROJECT_FILENAME),
        os.path.join(path, WATCH_FILENAME),
        IGNORE_FILENAME
    ]
    os.mkdir(path)

    for file in files:
        with open(file, 'w') as _:
            pass

    if not build_project_file(data, files[0]):
        logger.error(f'Could not build <{files[0]}>')
        return False

    if not build_watch_file(files[1]):
        logger.error(f'Could not build <{files[1]}>')
        return False

    if not build_ignore_file(files[2]):
        logger.error(f'Could not build <{files[2]}>')
        return False
    return True


# Insert the XML document with newlines and indentation space after elements
# 'tree' is the ElementTree that's being modified
# 'space' adds whitespace for each indentation level, two characters by default
# 'level' controls where the indentation starts. For a subtree nested further down, set value >0
def indent(tree, space="  ", level=0):
    if isinstance(tree, ET.ElementTree):
        tree = tree.getroot()

    # Reduce memory consumption by resuing indentation strings
    indentations = ["\n" + level * space]

    def indent_child(element, level):
        # New indent level
        child_level = level + 1
        try:
            child_indentation = indentations[child_level]
        except IndexError:
            child_indentation = indentations[level] + space
            indentations.append(child_indentation)

        if not element.text or not element.text.strip():
            element.text = child_indentation

        for child in element:
            if len(child):
                indent_child(child, child_level)
            if not child.tail or not child.tail.strip():
                child.tail = child_indentation

        # Indent and then overwrite previous indent
        if not child.tail.strip():
            child.tail = indentations[level]

    indent_child(tree, 0)


def build_project_file(data: dict, file):
    squirrel = ET.Element(
        'squirrel', name=f"{data.get('name') if data.get('name') is not None else ''}")

    _ = ET.SubElement(squirrel, 'path', src=f'{os.path.dirname(file)}')

    description = ET.SubElement(squirrel, 'description')
    description.text = data.get('description')

    due_date = ET.SubElement(squirrel, 'due-date')
    arg_due = data.get('due')
    due_date.text = str(arg_due) if arg_due is not None else None

    goal = ET.SubElement(squirrel, 'goal')
    arg_goal = data.get('goal')
    goal.text = str(arg_goal) if arg_goal is not None else None

    project_type = ET.SubElement(squirrel, 'project-type')
    p_type = data.get('project_type', 'text')
    project_type.text = p_type if p_type is not None else 'text'

    tree = ET.ElementTree(squirrel)
    indent(tree)
    tree.write(file, encoding='utf-8', xml_declaration=True)
    return True


def build_watch_file(file):
    squirrel = ET.Element('squirrel')
    comment = ET.Comment(
        'This is a file generated by squirrel. Modify it at your own risk.')
    squirrel.insert(1, comment)
    tree = ET.ElementTree(squirrel)
    indent(tree)
    tree.write(file, encoding='utf-8', xml_declaration=True)
    return True


def build_ignore_file(file):
    with open(file, 'w') as f:
        f.write(ignore_file_content)
    return True


def update_project_file(data: dict):
    path = project_file_path
    try:
        tree = parse(path)
    except FileNotFoundError:
        return False

    squirrel = tree.getroot()

    valid = True
    if (name := data.get('name')) is not None:
        squirrel.set('name', name)

    if (desc := data.get('description')) is not None:
        try:
            squirrel.find('description').text = desc
        except AttributeError:
            valid = False
            logger.error('[bold red blink]description[/] element was not found in the xml file'
                         ' try initializing the project again', extra={'markup': True})

    if (goal := data.get('goal')) is not None:
        try:
            squirrel.find('goal').text = str(goal)
        except AttributeError:
            valid = False
            logger.error('goal element was not found in the xml file'
                         ' try initializing the project again')

    if (due := data.get('due')) is not None:
        try:
            squirrel.find('due-date').text = str(due)
        except AttributeError:
            valid = False
            logger.error('due-date element was not found in the xml file'
                         'try init project again')

    if (project_type := data.get('project_type')) is not None:
        try:
            squirrel.find('project-type').text = project_type
        except AttributeError:
            valid = False
            logger.error('[bold red blink]project-type[/] element was not found in the xml file'
                         ' try initializing the project again', extra={'markup': True})

    tree.write(path, encoding='utf-8', xml_declaration=True)
    return valid


def get_data_from_project_file(basedir=''):
    """Returns all the metadata available in the project
    Raises FileNotFoundError, if files were not found"""
    data = {}
    path = os.path.join(basedir, project_file_path)
    try:
        tree = parse(path)
    except FileNotFoundError:
        raise

    squirrel = tree.getroot()

    try:
        name = squirrel.attrib['name']
        name = name if name != '' else None
    except (AttributeError, KeyError):
        logger.error('Could not find name attribute')
        name = None

    try:
        path = squirrel.find('path').attrib['src']
    except (AttributeError, KeyError):
        logger.error('Could not find path field')
        path = None

    try:
        description = squirrel.find('description').text
    except (AttributeError, KeyError):
        logger.error('Could not find description field')
        description = None

    try:
        goal = squirrel.find('goal').text
    except (AttributeError, KeyError):
        logger.error('Could not find goal field')
        goal = None

    try:
        due_date = datetime.strptime(squirrel.find(
            'due-date').text, DEFAULT_DATE_FORMAT).date()
    except (AttributeError, KeyError, TypeError):
        logger.error('Could not find due_date field')
        due_date = None

    try:
        project_type = squirrel.find('project-type').text
    except (AttributeError, KeyError):
        logger.error('Could not find project-type field')
        project_type = None

    data = {
        'name': name,
        'path': path,
        'description': description,
        'goal': goal,
        'due-date': due_date,
        'project-type': project_type
    }
    return data


def get_watches_data():
    """returns all <watches> tag data

    Raises FileNotFoundError when project files are not found.
    Raises ProjectNotSetupCorrectlyError if xml files are not valid.
    """
    path = watch_file_path
    try:
        tree = parse(path)
    except FileNotFoundError:
        raise
    squirrel = tree.getroot()

    data = []
    try:
        for watches in squirrel.findall('watches'):
            date = datetime.strptime(
                watches.attrib['date'], DEFAULT_DATE_FORMAT).date()
            data.append((date,
                         int(watches.attrib['prev_count']),
                         get_watches_last_count(watches)))
    except (AttributeError, KeyError):
        logger.error(f'Could not parse {path!r} correctly')
        raise ProjectNotSetupCorrectlyError(
            'Could not parse <watches> from xml file'
        )

    return data


def get_watches_last_count(watches):
    """Return the last count of a <watches> tag.
    (the last <watch> in a <watches>)"""
    if len(watches) == 0:
        return 0

    try:
        return int(watches[-1].text)
    except AttributeError:
        return 0


def get_day_watches(watches_tag):
    """Returns all the <watch> tags in a <watches>.
    Raises ProjectNotSetupCorrectlyError if project not initialized"""
    for watch in watches_tag:
        try:
            watch_datetime = datetime.strptime(
                watch.attrib['datetime'], DEFAULT_DATETIME_FORMAT)
            watch_count = int(watch.text)
            yield watch_datetime, watch_count
        except (AttributeError, KeyError):
            raise ProjectNotSetupCorrectlyError(
                'Watch file corrupted or not setup correctly'
            )


# TODO: Change name to get_watches_entry_by_date
def get_watches_entry(date):
    """returns the <watches> tag of the passed date and root element.
    defaults to (None, root)
    Raises FileNotFoundError if files are not present"""
    path = watch_file_path
    try:
        tree = parse(path)
    except FileNotFoundError:
        raise
    squirrel = tree.getroot()

    for watches in squirrel.findall('watches'):
        try:
            if watches.attrib['date'] == date.strftime(DEFAULT_DATE_FORMAT):
                return watches, squirrel
        except (KeyError, AttributeError):
            logger.warning(
                f'Some issues are present in {watch_file_path!r}'
                f'make sure that project was initialized correclty'
            )
    return None, squirrel


def make_watch_entry(parent, dt: str, value: str):
    """Create a <watch> tag associated with the parent"""
    watch = ET.SubElement(parent, 'watch', datetime=dt)
    watch.text = value
    return watch


def add_watch_entry(total, dt: datetime):
    """Add a <watch> tag to the <watches> tag of that date.
    Raises ProjectNotSetupCorrectlyError when project is not initialized."""
    path = watch_file_path

    # We try to get the watch of the datetime passed
    try:
        watches_date, root = get_watches_entry(dt.date())
    except FileNotFoundError:
        raise ProjectNotSetupCorrectlyError()

    if watches_date is not None:
        # If there are already <watch> inside <watches>, we have to verify that
        # they are different counts otherwise it's useless to record it
        # if in the other hand, we don't find any <watch> in <watches>
        # we should simply add a new <watch> tag
        if (len(watches_date) > 0 and
                watches_date[-1].text == str(total)):
            return False
        make_watch_entry(watches_date, str(dt), str(total))

    elif root is not None:
        # If haven't found <watches> tag, we need to create
        # a new one with a <watch> tag inside of it.
        try:
            prev_count = root[-1][-1].text
        except (IndexError, AttributeError):
            prev_count = str(0)

        watches = ET.SubElement(root,
                                'watches',
                                prev_count=prev_count,
                                date=dt.date().strftime(DEFAULT_DATE_FORMAT))
        make_watch_entry(watches, dt.strftime(
            DEFAULT_DATETIME_FORMAT), str(total))
    else:
        return False

    tree = ET.ElementTree(root)
    indent(tree)
    tree.write(path, encoding='utf-8', xml_declaration=True)
    return True


def parse(path):
    """Returns the ElementTree of an xml file.
    Raises FileNotFoundError when project is not initialized."""
    try:
        parser_save_comments = ET.XMLParser(
            target=ET.TreeBuilder(insert_comments=True))
        tree = ET.parse(path, parser_save_comments)
        return tree
    except FileNotFoundError:
        console = Console(stderr=True)
        console.print(
            f'Could not find {path!r};'
            ' Verify that the project was initialized correctly.'
        )
        raise
