"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssertionsProvider = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
let SDK_METADATA = undefined;
/**
 * integ-tests can only depend on '@aws-cdk/core' so
 * this construct creates a lambda function provider using
 * only CfnResource
 */
class LambdaFunctionProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.policies = [];
        const staging = new aws_cdk_lib_1.AssetStaging(this, 'Staging', {
            sourcePath: path.join(__dirname, 'lambda-handler.bundle'),
        });
        const stack = aws_cdk_lib_1.Stack.of(this);
        const asset = stack.synthesizer.addFileAsset({
            fileName: staging.relativeStagedPath(stack),
            sourceHash: staging.assetHash,
            packaging: aws_cdk_lib_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const role = new aws_cdk_lib_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const policies = this.policies.length > 0 ? [
                            {
                                PolicyName: 'Inline',
                                PolicyDocument: {
                                    Version: '2012-10-17',
                                    Statement: this.policies,
                                },
                            },
                        ] : undefined;
                        return policies;
                    },
                }),
            },
        });
        const handler = new aws_cdk_lib_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Runtime: 'nodejs14.x',
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: aws_cdk_lib_1.Duration.minutes(2).toSeconds(),
                Handler: props?.handler ?? 'index.handler',
                Role: role.getAtt('Arn'),
            },
        });
        this.serviceToken = aws_cdk_lib_1.Token.asString(handler.getAtt('Arn'));
        this.roleArn = role.getAtt('Arn');
    }
    addPolicies(policies) {
        this.policies.push(...policies);
    }
}
/**
 * Mimic the singletonfunction construct in '@aws-cdk/aws-lambda'
 */
class SingletonFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureFunction(props);
        this.serviceToken = this.lambdaFunction.serviceToken;
    }
    ensureFunction(props) {
        const constructName = 'SingletonFunction' + slugify(props.uuid);
        const existing = aws_cdk_lib_1.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        return new LambdaFunctionProvider(aws_cdk_lib_1.Stack.of(this), constructName, {
            handler: props.handler,
        });
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * singleton.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: 's3:GetObject',
     *   Resources: '*',
     * });
     */
    addToRolePolicy(statement) {
        this.lambdaFunction.addPolicies([statement]);
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        if (SDK_METADATA === undefined) {
            // eslint-disable-next-line
            SDK_METADATA = require('./sdk-api-metadata.json');
        }
        const srv = service.toLowerCase();
        const iamService = (SDK_METADATA[srv] && SDK_METADATA[srv].prefix) || srv;
        const iamAction = api.charAt(0).toUpperCase() + api.slice(1);
        this.lambdaFunction.addPolicies([{
                Action: [`${iamService}:${iamAction}`],
                Effect: 'Allow',
                Resource: resources || ['*'],
            }]);
    }
}
/**
 * Represents an assertions provider. The creates a singletone
 * Lambda Function that will create a single function per stack
 * that serves as the custom resource provider for the various
 * assertion providers
 */
class AssertionsProvider extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_AssertionsProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AssertionsProvider);
            }
            throw error;
        }
        this.handler = new SingletonFunction(this, 'AssertionsProvider', {
            handler: props?.handler,
            uuid: props?.uuid ?? '1488541a-7b23-4664-81b6-9b4408076b81',
        });
        this.handlerRoleArn = this.handler.lambdaFunction.roleArn;
        this.serviceToken = this.handler.serviceToken;
    }
    /**
     * Encode an object so it can be passed
     * as custom resource parameters. Custom resources will convert
     * all input parameters to strings so we encode non-strings here
     * so we can then decode them correctly in the provider function
     */
    encode(obj) {
        if (!obj) {
            return obj;
        }
        return JSON.parse(JSON.stringify(obj), (_k, v) => {
            switch (v) {
                case true:
                    return 'TRUE:BOOLEAN';
                case false:
                    return 'FALSE:BOOLEAN';
                default:
                    return v;
            }
        });
    }
    /**
     * Create a policy statement from a specific api call
     */
    addPolicyStatementFromSdkCall(service, api, resources) {
        this.handler.addPolicyStatementFromSdkCall(service, api, resources);
    }
    /**
     * Add an IAM policy statement to the inline policy of the
     * lambdas function's role
     *
     * **Please note**: this is a direct IAM JSON policy blob, *not* a `iam.PolicyStatement`
     * object like you will see in the rest of the CDK.
     *
     *
     * @example
     * declare const provider: AssertionsProvider;
     * provider.addToRolePolicy({
     *   Effect: 'Allow',
     *   Action: ['s3:GetObject'],
     *   Resource: ['*'],
     * });
     */
    addToRolePolicy(statement) {
        this.handler.addToRolePolicy(statement);
    }
    /**
     * Grant a principal access to invoke the assertion provider
     * lambda function
     *
     * @param principalArn the ARN of the principal that should be given
     *  permission to invoke the assertion provider
     */
    grantInvoke(principalArn) {
        new aws_cdk_lib_1.CfnResource(this, 'Invoke', {
            type: 'AWS::Lambda::Permission',
            properties: {
                Action: 'lambda:InvokeFunction',
                FunctionName: this.serviceToken,
                Principal: principalArn,
            },
        });
    }
}
exports.AssertionsProvider = AssertionsProvider;
_a = JSII_RTTI_SYMBOL_1;
AssertionsProvider[_a] = { fqn: "@aws-cdk/integ-tests-alpha.AssertionsProvider", version: "2.58.0-alpha.0" };
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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