"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat]
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1])
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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