"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const widget_1 = require("./widget");
// This file contains widgets that exist for layout purposes
/**
 * A widget that contains other widgets in a horizontal row
 *
 * Widgets will be laid out next to each other
 */
class Row {
    constructor(...widgets) {
        /**
         * Relative position of each widget inside this row
         */
        this.offsets = [];
        this.widgets = widgets;
        this.width = 0;
        this.height = 0;
        let x = 0;
        let y = 0;
        for (const widget of widgets) {
            // See if we need to horizontally wrap to add this widget
            if (x + widget.width > widget_1.GRID_WIDTH) {
                y = this.height;
                x = 0;
            }
            this.offsets.push({ x, y });
            this.width = Math.max(this.width, x + widget.width);
            this.height = Math.max(this.height, y + widget.height);
            x += widget.width;
        }
    }
    position(x, y) {
        for (let i = 0; i < this.widgets.length; i++) {
            this.widgets[i].position(x + this.offsets[i].x, y + this.offsets[i].y);
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Row = Row;
/**
 * A widget that contains other widgets in a vertical column
 *
 * Widgets will be laid out next to each other
 */
class Column {
    constructor(...widgets) {
        this.widgets = widgets;
        // There's no vertical wrapping so this one's a lot easier
        this.width = Math.max(...this.widgets.map(w => w.width));
        this.height = sum(...this.widgets.map(w => w.height));
    }
    position(x, y) {
        let widgetY = y;
        for (const widget of this.widgets) {
            widget.position(x, widgetY);
            widgetY += widget.height;
        }
    }
    toJson() {
        const ret = [];
        for (const widget of this.widgets) {
            ret.push(...widget.toJson());
        }
        return ret;
    }
}
exports.Column = Column;
/**
 * A widget that doesn't display anything but takes up space
 */
class Spacer {
    constructor(props) {
        this.width = props.width || 1;
        this.height = props.height || 1;
    }
    position(_x, _y) {
        // Don't need to do anything, not a physical widget
    }
    toJson() {
        return [];
    }
}
exports.Spacer = Spacer;
/**
 * Return the sum of a list of numbers
 */
function sum(...xs) {
    let ret = 0;
    for (const x of xs) {
        ret += x;
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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