"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const rendering_1 = require("./private/rendering");
const widget_1 = require("./widget");
/**
 * Display the metric associated with an alarm, including the alarm line
 */
class AlarmWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    annotations: {
                        alarms: [this.props.alarm.alarmArn]
                    },
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined
                    }
                }
            }];
    }
}
exports.AlarmWidget = AlarmWidget;
/**
 * A dashboard widget that displays metrics
 */
class GraphWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 6);
        this.props = props;
    }
    toJson() {
        const horizontalAnnotations = [
            ...(this.props.leftAnnotations || []).map(mapAnnotation('left')),
            ...(this.props.rightAnnotations || []).map(mapAnnotation('right')),
        ];
        const metrics = rendering_1.allMetricsGraphJson(this.props.left || [], this.props.right || []);
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'timeSeries',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    stacked: this.props.stacked,
                    metrics: metrics.length > 0 ? metrics : undefined,
                    annotations: horizontalAnnotations.length > 0 ? { horizontal: horizontalAnnotations } : undefined,
                    yAxis: {
                        left: this.props.leftYAxis !== undefined ? this.props.leftYAxis : undefined,
                        right: this.props.rightYAxis !== undefined ? this.props.rightYAxis : undefined,
                    }
                }
            }];
    }
}
exports.GraphWidget = GraphWidget;
/**
 * A dashboard widget that displays the most recent value for every metric
 */
class SingleValueWidget extends widget_1.ConcreteWidget {
    constructor(props) {
        super(props.width || 6, props.height || 3);
        this.props = props;
    }
    toJson() {
        return [{
                type: 'metric',
                width: this.width,
                height: this.height,
                x: this.x,
                y: this.y,
                properties: {
                    view: 'singleValue',
                    title: this.props.title,
                    region: this.props.region || cdk.Aws.REGION,
                    metrics: rendering_1.allMetricsGraphJson(this.props.metrics, []),
                    setPeriodToTimeRange: this.props.setPeriodToTimeRange
                }
            }];
    }
}
exports.SingleValueWidget = SingleValueWidget;
var Shading;
(function (Shading) {
    /**
     * Don't add shading
     */
    Shading["NONE"] = "none";
    /**
     * Add shading above the annotation
     */
    Shading["ABOVE"] = "above";
    /**
     * Add shading below the annotation
     */
    Shading["BELOW"] = "below";
})(Shading = exports.Shading || (exports.Shading = {}));
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 */
class Color {
}
exports.Color = Color;
/** blue - hex #1f77b4 */
Color.BLUE = '#1f77b4';
/** brown - hex #8c564b */
Color.BROWN = '#8c564b';
/** green - hex #2ca02c */
Color.GREEN = '#2ca02c';
/** grey - hex #7f7f7f */
Color.GREY = '#7f7f7f';
/** orange - hex #ff7f0e */
Color.ORANGE = '#ff7f0e';
/** pink - hex #e377c2 */
Color.PINK = '#e377c2';
/** purple - hex #9467bd */
Color.PURPLE = '#9467bd';
/** red - hex #d62728 */
Color.RED = '#d62728';
function mapAnnotation(yAxis) {
    return (a) => {
        return { ...a, yAxis };
    };
}
//# sourceMappingURL=data:application/json;base64,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