"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionProvider = exports.Product = exports.Platform = void 0;
const fs = require("fs");
const https = require("https");
const url = require("url");
const version_1 = require("./version");
var Platform;
(function (Platform) {
    Platform["linux"] = "linux";
    Platform["mac"] = "mac";
    Platform["windows"] = "windows";
})(Platform = exports.Platform || (exports.Platform = {}));
var Product;
(function (Product) {
    Product["deadline"] = "Deadline";
    Product["deadlineDocker"] = "DeadlineDocker";
})(Product = exports.Product || (exports.Product = {}));
/**
 * The version provider parses a JSON file containing version information for the Deadline and DockerDeadline products.
 * It  can be downloaded or loaded from local file and returns URIs for the specific products.
 * By default returns the last version of URIs or specified full or partial version.
 * If platform is not defined returns URIs for each platform.
 */
class VersionProvider {
    constructor(indexFilePath) {
        this.indexFilePath = indexFilePath;
    }
    /**
     * Returns URIs for specified product
     */
    async getVersionUris(resourceProperties) {
        const indexJson = this.indexFilePath ? this.readInstallersIndex() : await this.downloadInstallerIndex();
        const productSection = indexJson[resourceProperties.product];
        if (!productSection) {
            throw new Error(`Information about product ${resourceProperties.product} can't be found`);
        }
        let installers = new Map();
        if (resourceProperties.platform) {
            const versionedUris = this.getUrisForPlatform(resourceProperties.product, productSection, resourceProperties.platform, resourceProperties.versionString);
            if (versionedUris) {
                installers.set(resourceProperties.platform, versionedUris);
            }
        }
        else {
            Object.values(Platform).forEach(async (p) => {
                const versionedUris = this.getUrisForPlatform(resourceProperties.product, productSection, p, resourceProperties.versionString);
                if (versionedUris) {
                    installers.set(p, versionedUris);
                }
            });
        }
        return installers;
    }
    async downloadInstallerIndex() {
        const parsedUrl = url.parse(VersionProvider.VERSION_INDEX_URL);
        const options = {
            host: parsedUrl.hostname,
            path: parsedUrl.path,
        };
        return new Promise((resolve, reject) => {
            https.get(options, (res) => {
                let json = '';
                res.on('data', (chunk) => {
                    // keep appending the response chunks until we get 'end' event.
                    json += chunk;
                });
                res.on('end', () => {
                    // complete response is available here:
                    if (res.statusCode === 200) {
                        try {
                            // convert the response to a json object and return.
                            const data = JSON.parse(json);
                            resolve(data);
                        }
                        catch (e) {
                            reject(e);
                        }
                    }
                    else {
                        reject(new Error(`Expected status code 200, but got ${res.statusCode}`));
                    }
                });
            }).on('error', (err) => {
                reject(err);
            });
        });
    }
    /**
     * This method reads index file and return parsed JSON object from this file content.
     */
    readInstallersIndex() {
        if (!this.indexFilePath) {
            throw new Error('File path should be defined.');
        }
        if (!fs.existsSync(this.indexFilePath)) {
            throw new Error(`File ${this.indexFilePath} was not found`);
        }
        const data = fs.readFileSync(this.indexFilePath, 'utf8');
        // convert the response to a json object and return.
        const json = JSON.parse(data);
        return json;
    }
    /**
     * This method returns IVersionedUris (the patch version plus installer URI's) for a specific platform.
     */
    getUrisForPlatform(product, productSection, platform, version) {
        const versionString = version ? version : this.getLatestVersion(platform, productSection);
        const requestedVersion = version_1.Version.parseFromVersionString(versionString);
        if (!requestedVersion) {
            throw new Error(`Couldn't parse version from ${versionString}`);
        }
        return this.getRequestedUriVersion(requestedVersion, productSection.versions, platform, product);
    }
    /**
     * This method returns the latest version for specified platform.
     */
    getLatestVersion(platform, indexedVersionInfo) {
        const latestSection = indexedVersionInfo.latest;
        if (!latestSection) {
            throw new Error('Information about latest version can not be found');
        }
        const latestVersion = latestSection[platform];
        if (!latestVersion) {
            throw new Error(`Information about latest version for platform ${platform} can not be found`);
        }
        return latestVersion;
    }
    /**
     * This method looks for the requested version (partial or complete) in the
     * indexed version information. Based on the input, it iterates through all
     * four numbers in the version string and compares the requested version
     * with the indexed info.
     * If any of the requested version number is missing, it fetches the latest
     * (highest) available version for it.
     */
    getRequestedUriVersion(requestedVersion, indexedVersionInfo, platform, product) {
        let versionMap = indexedVersionInfo;
        const versionArray = [];
        // iterate through all 4 major, minor, release and patch numbers,
        // and get the matching version from the indexed version map.
        for (let versionIndex = 0; versionIndex < 4; versionIndex++) {
            let version;
            if (requestedVersion[versionIndex + 1] == null) {
                // version is not provided, get the max version.
                const numberValues = (Object.keys(versionMap)).map((val) => {
                    return parseInt(val, 10);
                });
                version = (Math.max(...numberValues)).toString();
            }
            else {
                version = requestedVersion[versionIndex + 1];
            }
            versionArray[versionIndex] = version;
            versionMap = versionMap[version];
        }
        let uriIndex;
        if ((platform in versionMap)) {
            const platformVersionMap = versionMap[platform];
            if (product == Product.deadline) {
                uriIndex = {
                    bundle: platformVersionMap.bundle,
                    clientInstaller: platformVersionMap.clientInstaller,
                    repositoryInstaller: platformVersionMap.repositoryInstaller,
                    certificateInstaller: platformVersionMap.certificateInstaller,
                };
            }
            else { // Product.deadlineDocker
                uriIndex = {
                    bundle: platformVersionMap,
                };
            }
        }
        if (uriIndex) {
            return {
                MajorVersion: versionArray[0],
                MinorVersion: versionArray[1],
                ReleaseVersion: versionArray[2],
                PatchVersion: versionArray[3],
                Uris: uriIndex,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.VersionProvider = VersionProvider;
VersionProvider.VERSION_INDEX_URL = 'https://downloads.thinkboxsoftware.com/version_info.json';
//# sourceMappingURL=data:application/json;base64,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