"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const path = require("path");
const version_1 = require("../version");
const version_provider_1 = require("../version-provider");
const versionProvider = new version_provider_1.VersionProvider(path.join(__dirname, 'index-test.json'));
const indexTest = versionProvider['readInstallersIndex']();
const productSection = indexTest[version_provider_1.Product.deadline];
test.each([[version_provider_1.Platform.linux, '10.1.9.2'],
    [version_provider_1.Platform.mac, '10.1.9.2'],
    [version_provider_1.Platform.windows, '10.1.8.5'],
])('latest version ', (platform, versionString) => {
    const result = versionProvider['getLatestVersion'](platform, productSection);
    expect(result).toEqual(versionString);
});
test.each([
    [version_provider_1.Platform.linux, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
        }],
    [version_provider_1.Platform.windows, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/Deadline-10.1.9.2-windows-installers.zip',
            clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineClient-10.1.9.2-windows-installer.exe',
            repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Windows/DeadlineRepository-10.1.9.2-windows-installer.exe',
        }],
    [version_provider_1.Platform.mac, {
            bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
        }],
])('get Uri for platform', async (platform, versionedUris) => {
    const result = versionProvider['getUrisForPlatform'](version_provider_1.Product.deadline, productSection, platform, '10.1.9.2');
    expect(result).not.toBeNull();
    expect(result === null || result === void 0 ? void 0 : result.Uris).toEqual(versionedUris);
});
test('get Uri for platform - bad version', async () => {
    const badVersion = 'badVersionString';
    expect(() => versionProvider['getUrisForPlatform'](version_provider_1.Product.deadline, productSection, version_provider_1.Platform.linux, badVersion)).toThrowError(`Couldn't parse version from ${badVersion}`);
});
test('get deadline version', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
        platform: version_provider_1.Platform.linux,
        versionString: '10.1',
    });
    expect(result).not.toBeNull();
    const installerVersion = result.get(version_provider_1.Platform.linux);
    expect(installerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.MajorVersion).toEqual('10');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.MinorVersion).toEqual('1');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.ReleaseVersion).toEqual('9');
    expect(installerVersion === null || installerVersion === void 0 ? void 0 : installerVersion.PatchVersion).toEqual('2');
});
test('product is not in file', async () => {
    await expect(versionProvider.getVersionUris({
        product: version_provider_1.Product.deadlineDocker,
    })).rejects.toThrowError(/Information about product DeadlineDocker can't be found/);
});
test('get deadline version for all platforms', async () => {
    const result = await versionProvider.getVersionUris({
        product: version_provider_1.Product.deadline,
    });
    expect(result).not.toBeNull();
    const linuxInstallerVersion = result.get(version_provider_1.Platform.linux);
    expect(linuxInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/Deadline-10.1.9.2-linux-installers.tar',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineClient-10.1.9.2-linux-x64-installer.run',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.9.2/Linux/DeadlineRepository-10.1.9.2-linux-x64-installer.run',
    });
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.MajorVersion).toEqual('10');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.MinorVersion).toEqual('1');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.ReleaseVersion).toEqual('9');
    expect(linuxInstallerVersion === null || linuxInstallerVersion === void 0 ? void 0 : linuxInstallerVersion.PatchVersion).toEqual('2');
    const macInstallerVersion = result.get(version_provider_1.Platform.mac);
    expect(macInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.9.2/Mac/Deadline-10.1.9.2-osx-installers.dmg',
    });
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.MajorVersion).toEqual('10');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.MinorVersion).toEqual('1');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.ReleaseVersion).toEqual('9');
    expect(macInstallerVersion === null || macInstallerVersion === void 0 ? void 0 : macInstallerVersion.PatchVersion).toEqual('2');
    const windowsInstallerVersion = result.get(version_provider_1.Platform.windows);
    expect(windowsInstallerVersion).not.toBeNull();
    if (result === null) {
        return;
    }
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.Uris).toEqual({
        bundle: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/Deadline-10.1.8.5-windows-installers.zip',
        clientInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineClient-10.1.8.5-windows-installer.exe',
        repositoryInstaller: 's3://thinkbox-installers/Deadline/10.1.8.5/Windows/DeadlineRepository-10.1.8.5-windows-installer.exe',
    });
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.MajorVersion).toEqual('10');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.MinorVersion).toEqual('1');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.ReleaseVersion).toEqual('8');
    expect(windowsInstallerVersion === null || windowsInstallerVersion === void 0 ? void 0 : windowsInstallerVersion.PatchVersion).toEqual('5');
});
test('not defined file path', () => {
    expect(() => (new version_provider_1.VersionProvider())['readInstallersIndex']()).toThrowError(/File path should be defined./);
});
test('invalid file path', () => {
    expect(() => (new version_provider_1.VersionProvider('test.txt'))['readInstallersIndex']()).toThrowError(/File test.txt was not found/);
});
test('get latest version without latest section', () => {
    expect(() => versionProvider['getLatestVersion']('linux', {})).toThrowError(/Information about latest version can not be found/);
});
test('get latest version without informtion for platform', () => {
    expect(() => versionProvider['getLatestVersion']('linux', { latest: {} })).toThrowError(/Information about latest version for platform linux can not be found/);
});
test('get requested Uri version for existing product.', () => {
    const requestedVersion = version_1.Version.parseFromVersionString('10.1.9.2');
    expect(requestedVersion).not.toBeNull();
    if (requestedVersion === null) {
        return;
    }
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        },
    }, version_provider_1.Platform.linux, version_provider_1.Product.deadlineDocker)).toEqual({
        MajorVersion: '10',
        MinorVersion: '1',
        ReleaseVersion: '9',
        PatchVersion: '2',
        Uris: { bundle: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz' },
    });
});
test('get requested Uri version for not existing product.', () => {
    const requestedVersion = version_1.Version.parseFromVersionString('10.1.9.2');
    expect(requestedVersion).not.toBeNull();
    if (requestedVersion === null) {
        return;
    }
    expect(versionProvider['getRequestedUriVersion'](requestedVersion, {
        10: {
            1: {
                9: {
                    2: {
                        linux: 's3://thinkbox-installers/DeadlineDocker/10.1.9.2/DeadlineDocker-10.1.9.2.tar.gz',
                    },
                },
            },
        },
    }, version_provider_1.Platform.windows, version_provider_1.Product.deadlineDocker)).toEqual(undefined);
});
//# sourceMappingURL=data:application/json;base64,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