"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../../core/lib");
const tag_helpers_1 = require("../../core/test/tag-helpers");
const token_regex_helpers_1 = require("../../core/test/token-regex-helpers");
const lib_2 = require("../lib");
const asset_constants_1 = require("./asset-constants");
let app;
let stack;
let wfstack;
let vpc;
let renderQueue;
let rcsImage;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(stack, 'VPC');
    rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
    const version = new lib_2.VersionQuery(stack, 'VersionQuery');
    renderQueue = new lib_2.RenderQueue(stack, 'RQ', {
        vpc,
        images: { remoteConnectionServer: rcsImage },
        repository: new lib_2.Repository(stack, 'Repository', {
            vpc,
            version,
        }),
        version,
    });
    wfstack = new core_1.Stack(app, 'workerFleetStack', {
        env: {
            region: 'us-east-1',
        },
    });
});
test('default worker fleet is created correctly', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.large',
        spotPrice: assert_1.ABSENT,
    }));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        ToPort: parseInt(renderQueue.endpoint.portAsString(), 10),
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                'GroupId',
            ],
        },
        GroupId: {
            'Fn::ImportValue': 'infraStack:ExportsOutputFnGetAttRQLBSecurityGroupAC643AEDGroupId8F9F7830',
        },
    }));
    assert_1.expect(wfstack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: '/renderfarm/workerFleet',
    }));
    expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    expect(fleet.node.metadata[1].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
    expect(fleet.node.metadata[1].data).toContain('being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy');
});
test('security group is added to fleet after its creation', () => {
    // WHEN
    const fleet = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
    });
    fleet.addSecurityGroup(aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
        allowAllOutbound: false,
    }));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            {
                'Fn::GetAtt': [
                    stack.getLogicalId(fleet.fleet.connections.securityGroups[0].node.defaultChild),
                    'GroupId',
                ],
            },
            'sg-123456789',
        ],
    }));
});
test('WorkerFleet uses given security group', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericWindowsImage({
            'us-east-1': 'ami-any',
        }),
        renderQueue,
        securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'SG', 'sg-123456789', {
            allowAllOutbound: false,
        }),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SecurityGroups: [
            'sg-123456789',
        ],
    }));
});
test('default worker fleet is created correctly with linux image', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration'));
});
test('default worker fleet is created correctly with spot config', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        spotPrice: 2.5,
    });
    // THEN
    assert_1.expect(wfstack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
    assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        SpotPrice: '2.5',
    }));
});
test('default worker fleet is not created with incorrect spot config', () => {
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT + 1,
        });
    }).toThrowError(/Invalid value: 256 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
    // WHEN
    expect(() => {
        new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            spotPrice: lib_2.WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT / 2,
        });
    }).toThrowError(/Invalid value: 0.0005 for property 'spotPrice'. Valid values can be any decimal between 0.001 and 255./);
});
test('default worker fleet is created correctly custom Instance type', () => {
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
        InstanceType: 't2.medium',
    }));
});
test.each([
    'test-prefix/',
    '',
])('default worker fleet is created correctly with custom LogGroup prefix %s', (testPrefix) => {
    // GIVEN
    const id = 'workerFleet';
    // WHEN
    new lib_2.WorkerInstanceFleet(stack, id, {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        logGroupProps: {
            logGroupPrefix: testPrefix,
        },
    });
    assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
        RetentionInDays: 3,
        LogGroupName: testPrefix + id,
    }));
});
test('default worker fleet is created correctly custom subnet values', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        healthCheckConfig: {
            port: 6161,
        },
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
        VPCZoneIdentifier: [{
                Ref: 'VPC1AzPublicSubnet1Subnet9649CC17',
            }],
    }));
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': [
            '',
            [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'set -e\n' +
                    'chmod +x \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                {
                                    Ref: asset_constants_1.CWA_ASSET_LINUX.Key,
                                },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\'\n\'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                '\' ',
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' '' '' '' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}'`,
            ],
        ],
    });
});
test('default worker fleet is created correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // WHEN
    const workers = new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
        vpc,
        workerMachineImage: new aws_ec2_1.GenericLinuxImage({
            'us-east-1': '123',
        }),
        renderQueue,
        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.MEDIUM),
        vpcSubnets: {
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        },
        groups: ['A', 'B'],
        pools: ['C', 'D'],
        region: 'E',
    });
    // THEN
    const userData = stack.resolve(workers.fleet.userData.render());
    expect(userData).toStrictEqual({
        'Fn::Join': ['', [
                '#!/bin/bash\nfunction exitTrap(){\nexitCode=$?\n/opt/aws/bin/cfn-signal --stack infraStack --resource workerFleetASG25520D69 --region us-east-1 -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n}\ntrap exitTrap EXIT\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' ",
                { Ref: 'workerFleetStringParameterDB3717DA' },
                '\nmkdir -p $(dirname \'/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\')\naws s3 cp \'s3://',
                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\'\n' +
                    'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                    '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                    'fi\n' +
                    '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '" --render-queue "http://',
                {
                    'Fn::GetAtt': [
                        'RQLB3B7B1CBC',
                        'DNSName',
                    ],
                },
                ':8080" \n' +
                    'rm -f "/tmp/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                            ],
                        },
                    ],
                },
                '\"\n' +
                    'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                    '  service deadline10launcher restart\n' +
                    'fi\n' +
                    "mkdir -p $(dirname '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "')\naws s3 cp 's3://",
                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Bucket },
                '/',
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "' '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\nset -e\nchmod +x '/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                "'\n'/tmp/",
                {
                    'Fn::Select': [
                        0,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                {
                    'Fn::Select': [
                        1,
                        {
                            'Fn::Split': [
                                '||',
                                { Ref: asset_constants_1.CONFIG_WORKER_ASSET_LINUX.Key },
                            ],
                        },
                    ],
                },
                `' 'a,b' 'c,d' 'E' '${lib_2.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION}'`,
            ]],
    });
});
test('worker fleet does validation correctly with groups, pools and region', () => {
    vpc = new aws_ec2_1.Vpc(stack, 'VPC1Az', {
        maxAzs: 1,
    });
    // group name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'none'],
        });
    }).toThrowError();
    // group name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet1', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            groups: ['A', 'no ne'],
        });
    }).toThrowError(/Invalid value: no ne for property 'groups'/);
    // pool name with whitespace
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet2', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // pool name as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet3', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            pools: ['A', 'none'],
        });
    }).toThrowError(/Invalid value: none for property 'pools'/);
    // region as 'none'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet4', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none',
        });
    }).toThrowError(/Invalid value: none for property 'region'/);
    // region as 'all'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet5', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'all',
        });
    }).toThrowError(/Invalid value: all for property 'region'/);
    // region as 'unrecognized'
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet6', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'unrecognized',
        });
    }).toThrowError(/Invalid value: unrecognized for property 'region'/);
    // region with invalid characters
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet7', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none@123',
        });
    }).toThrowError(/Invalid value: none@123 for property 'region'/);
    // region with reserved name as substring
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet8', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'none123',
        });
    }).not.toThrowError();
    // region with case-insensitive name
    expect(() => {
        new lib_2.WorkerInstanceFleet(stack, 'workerFleet9', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': '123',
            }),
            renderQueue,
            region: 'None',
        });
    }).toThrowError(/Invalid value: None for property 'region'/);
});
describe('Block Device Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Warning if no BlockDevices provided', () => {
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch('being created without being provided any block devices so the Source AMI\'s devices will be used. Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
    });
    test('No Warnings if Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: true }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: true,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('Warnings if non-Encrypted BlockDevices Provided', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE, { encrypted: false }),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        Encrypted: false,
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('Warnings for BlockDevices without encryption specified', () => {
        const VOLUME_SIZE = 50;
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(VOLUME_SIZE),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    Ebs: {
                        VolumeSize: VOLUME_SIZE,
                    },
                },
            ],
        }));
        expect(fleet.node.metadata[0].type).toMatch(cloud_assembly_schema_1.ArtifactMetadataEntryType.WARN);
        expect(fleet.node.metadata[0].data).toMatch(`The BlockDevice \"${DEVICE_NAME}\" on the worker-fleet workerFleet is not encrypted. Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.`);
    });
    test('No warnings for Ephemeral blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.ephemeral(0),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                    VirtualName: 'ephemeral0',
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
    test('No warnings for Suppressed blockDeviceVolumes', () => {
        const DEVICE_NAME = '/dev/xvda';
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            blockDevices: [{
                    deviceName: DEVICE_NAME,
                    volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                }],
        });
        //THEN
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            BlockDeviceMappings: [
                {
                    DeviceName: DEVICE_NAME,
                },
            ],
        }));
        expect(fleet.node.metadata).toHaveLength(0);
    });
});
describe('HealthMonitor Tests', () => {
    let healthMonitor;
    beforeEach(() => {
        // create a health monitor so it does not trigger warnings
        healthMonitor = new lib_1.HealthMonitor(wfstack, 'healthMonitor', {
            vpc,
        });
    });
    test('Monitor is configured for Windows', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('&\'C:/temp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('Monitor is configured for Linux', () => {
        // WHEN
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        // Ensure the configuration script is executed with the expected arguments.
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex('\'/tmp/${Token[TOKEN.\\d+]}${Token[TOKEN.\\d+]}\' \'63415\' \'10.1.9.2\'')));
        // Ensure that the health monitor target group has been set up.
        //  Note: It's sufficient to just check for any resource created by the HealthMonitor registration.
        //   The HealthMonitor tests cover ensuring that all of the resources are set up.
        assert_1.expect(wfstack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '63415',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
    });
    test('UserData is added', () => {
        // WHEN
        class UserDataProvider extends lib_2.InstanceUserDataProvider {
            preCloudWatchAgent(host) {
                host.userData.addCommands('echo preCloudWatchAgent');
            }
            preRenderQueueConfiguration(host) {
                host.userData.addCommands('echo preRenderQueueConfiguration');
            }
            preWorkerConfiguration(host) {
                host.userData.addCommands('echo preWorkerConfiguration');
            }
            postWorkerLaunch(host) {
                host.userData.addCommands('echo postWorkerLaunch');
            }
        }
        const fleet = new lib_2.WorkerInstanceFleet(wfstack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            healthMonitor,
            userDataProvider: new UserDataProvider(wfstack, 'UserDataProvider'),
        });
        const userData = fleet.fleet.userData.render();
        // THEN
        expect(userData).toContain('echo preCloudWatchAgent');
        expect(userData).toContain('echo preRenderQueueConfiguration');
        expect(userData).toContain('echo preWorkerConfiguration');
        expect(userData).toContain('echo postWorkerLaunch');
    });
});
describe('tagging', () => {
    tag_helpers_1.testConstructTags({
        constructName: 'WorkerInstanceFleet',
        createConstruct: () => {
            // GIVEN
            const healthMonitorStack = new core_1.Stack(app, 'HealthMonitorStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            const healthMonitor = new lib_1.HealthMonitor(healthMonitorStack, 'healthMonitor', {
                vpc,
            });
            const deviceName = '/dev/xvda';
            // WHEN
            new lib_2.WorkerInstanceFleet(wfstack, 'WorkerFleet', {
                vpc,
                workerMachineImage: new aws_ec2_1.GenericLinuxImage({
                    'us-east-1': 'ami-any',
                }),
                renderQueue,
                healthMonitor,
                blockDevices: [{
                        deviceName,
                        volume: aws_autoscaling_1.BlockDeviceVolume.noDevice(),
                    }],
            });
            return wfstack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::ElasticLoadBalancingV2::TargetGroup': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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