"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
describe('Version', () => {
    describe('.isGreaterThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '10.0.9.2',
                    secondVersion: '10.1.9.2',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ], [
                'greater than',
                {
                    firstVersion: '2.0.0.1',
                    secondVersion: '2.0.0.0',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isGreaterThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isGreaterThan constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 1, 9, 2]);
        const result = lhs.isGreaterThan(lib_1.Version.parse('10.0.9.2'));
        expect(result).toEqual(true);
    });
    describe('.isLessThan', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'greater minor version',
                {
                    firstVersion: '10.1.9.2',
                    secondVersion: '10.0.9.2',
                    expectedValue: false,
                },
            ], [
                'greater patch version',
                {
                    firstVersion: '1.1.1.2',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '2.0.0.0',
                    secondVersion: '2.0.0.1',
                    expectedValue: true,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isLessThan(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isEqual', () => {
        test.each([
            [
                'equal version',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: true,
                },
            ], [
                'unequal',
                {
                    firstVersion: '2.1.1.1',
                    secondVersion: '1.1.1.1',
                    expectedValue: false,
                },
            ], [
                'less than',
                {
                    firstVersion: '1.1.1.1',
                    secondVersion: '1.1.1.2',
                    expectedValue: false,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { firstVersion, secondVersion, expectedValue } = testcase;
            // WHEN
            const lhs = lib_1.Version.parse(firstVersion);
            const result = lhs.isEqual(lib_1.Version.parse(secondVersion));
            expect(result).toEqual(expectedValue);
        });
    });
    describe('.isLessThan using constructor', () => {
        // WHEN
        const lhs = new lib_1.Version([10, 0, 9, 2]);
        const result = lhs.isLessThan(lib_1.Version.parse('10.1.9.2'));
        expect(result).toEqual(true);
    });
    describe('constructor validation', () => {
        test.each([
            [
                'incorrect component count',
                {
                    version: [10, 1, 9],
                    expectedException: /Invalid version format. Version should contain exactly 4 components./,
                },
            ], [
                'negative value',
                {
                    version: [10, -1, 9, 2],
                    expectedException: /Invalid version format. None of the version components can be negative./,
                },
            ], [
                'decimal value',
                {
                    version: [10, 1, 9.2, 2],
                    expectedException: /Invalid version format. None of the version components can contain decimal values./,
                },
            ], [
                'correct value',
                {
                    version: [10, 1, 9, 2],
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => new lib_1.Version(version)).toThrow(expectedException);
            }
            else {
                const versionObj = new lib_1.Version(version);
                expect(versionObj.majorVersion).toEqual(version[0]);
                expect(versionObj.minorVersion).toEqual(version[1]);
                expect(versionObj.releaseVersion).toEqual(version[2]);
                expect(versionObj.patchVersion).toEqual(version[3]);
            }
        });
    });
    describe('.parse throws exception', () => {
        test.each([
            [
                'ending with .',
                {
                    version: '10.1.9.',
                    expectedException: /Invalid version format/,
                },
            ], [
                'empty string',
                {
                    version: '',
                    expectedException: /Invalid version format/,
                },
            ], [
                'negative value',
                {
                    version: '10.-1.9.2',
                    expectedException: /Invalid version format/,
                },
            ], [
                'correct version',
                {
                    version: '10.1.9.2',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { version, expectedException } = testcase;
            // WHEN
            if (expectedException) {
                expect(() => lib_1.Version.parse(version)).toThrow(expectedException);
            }
            else {
                expect(() => lib_1.Version.parse(version)).not.toThrow();
            }
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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