"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Stage', () => {
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    /**
     * A sub-class of Stage that makes the constructor public for testing purposes.
     *
     * This is to be able to specify a manifest rather than load it from the file-system via {@link Stage.fromDirectory}.
     */
    class StageWithPulicConstructor extends lib_1.Stage {
        constructor(props) {
            super(props);
        }
    }
    beforeEach(() => {
        jest.resetModules();
    });
    describe('.fromDirectory', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {},
        };
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        let stage;
        beforeAll(() => {
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            stage = require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        });
        afterAll(() => {
            readFileSync.mockRestore();
            jest.resetModules();
        });
        test('has dirPath', () => {
            // THEN
            expect(stage.dirPath).toBe(STAGE_PATH);
        });
        test('loads manifest.json from directory', () => {
            // THEN
            const expectedPath = path.join(STAGE_PATH, 'manifest.json');
            expect(readFileSync).toHaveBeenCalledWith(expectedPath, expect.objectContaining({
                encoding: 'utf8',
            }));
            expect(stage.manifest).toEqual(manifest);
        });
    });
    describe('.fromDirectory', () => {
        test('not supported version failure', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.8.0',
                recipes: {},
            };
            // WHEN
            const readFileSync = jest.spyOn(fs, 'readFileSync');
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            // THEN
            expect(() => {
                require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
            }).toThrow('Staged Deadline Version (10.1.8.0) is less than the minimum supported version (10.1.9.2)');
            readFileSync.mockRestore();
            jest.resetModules();
        });
    });
    test('has manifest', () => {
        const manifest = {
            schema: 1,
            version: '10.1.9.2',
            recipes: {
                a: {
                    title: 'a-title',
                    description: 'a-description',
                    buildArgs: {
                        argone: 'a-argone-value',
                        argtwo: 'a-argtwo-value',
                    },
                },
                b: {
                    title: 'b-title',
                    description: 'b-description',
                    buildArgs: {
                        argone: 'b-argone-value',
                        argtwo: 'b-argtwo-value',
                    },
                },
            },
        };
        const stage = new StageWithPulicConstructor({
            manifest,
            path: STAGE_PATH,
        });
        // THEN
        expect(stage.manifest).toEqual(manifest);
    });
    describe('manifest validation', () => {
        test.each([
            [
                'mainfest wrong type',
                {
                    manifest: 1,
                    expectedError: /Expected object but got/,
                },
            ],
            [
                'manifest version not string',
                {
                    manifest: {
                        version: 1,
                        recipes: {},
                    },
                },
            ],
            [
                'missing schema',
                {
                    manifest: {
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "schema" key/,
                },
            ],
            [
                'wrong schema type', {
                    manifest: {
                        schema: 'a',
                        version: '10.1.9.2',
                        recipes: {},
                    },
                    expectedError: /Expected a numeric "schema" but got:/,
                },
            ],
            [
                'missing version', {
                    manifest: {
                        schema: 1,
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "version" key/,
                },
            ],
            [
                'incorrect version format', {
                    manifest: {
                        schema: 1,
                        version: '10.1.',
                        recipes: {},
                    },
                    expectedError: /Invalid version format/,
                },
            ],
            [
                'version not supported', {
                    manifest: {
                        schema: 1,
                        version: '10.1.0.0',
                        recipes: {},
                    },
                    expectedError: 'Staged Deadline Version (10.1.0.0) is less than the minimum supported version (10.1.9.2)',
                },
            ],
        ])('%s', (_name, testcase) => {
            const { manifest, expectedError } = testcase;
            // WHEN
            function when() {
                new StageWithPulicConstructor({
                    path: STAGE_PATH,
                    manifest,
                });
            }
            expect(when).toThrow(expectedError);
        });
    });
    describe('valid recipes', () => {
        test.each([
            [
                'build args and no target',
                {
                    title: 'Eugene\'s Favorite Cookies',
                    description: 'C is for cookie. That\'s good enough for me',
                    buildArgs: {
                        flour: '3 cups',
                        butter: '1 cup',
                        sugar: '1/2 cup',
                        egg: '2',
                        bakingpowder: '1 tsp',
                        bakingsoda: '1 tsb',
                        vanilla: '1 tsp',
                        salt: '1 pinch',
                    },
                },
            ],
            [
                'buildArgs and a target',
                {
                    title: 'Josh\'s S\'mores',
                    description: 'a secret family recipe',
                    buildArgs: {
                        grahamcracker: '2',
                        marshmellow: '1',
                        chocolate: '2',
                        campfire: '1',
                    },
                    target: 'simplicity',
                },
            ],
            [
                'target and no buildArgs',
                {
                    title: 'Jericho\s special brownie batch with a strawberry milkshake',
                    description: 'a slight improvement to the original recipe',
                    target: 'target',
                },
            ],
            [
                'no target or buildArgs',
                {
                    title: 'Yash\'s Tequila Lime Shot (TLS)',
                    description: 'A sure-fire way to brighten your day',
                },
            ],
        ])('%s', (_scenario, recipe) => {
            // GIVEN
            const manifest = {
                recipes: {
                    recipeName: recipe,
                },
                schema: 1,
                version: '10.1.9.2',
            };
            // WHEN
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // THEN
            expect(stage.manifest.recipes.recipeName).toEqual(recipe);
        });
    });
    describe('.imageFromRecipe()', () => {
        let app;
        let stack;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
        });
        test('creates DockerImageAssets from existing recipes', () => {
            jest.resetModules();
            const mockDockerImageAssetConstructor = jest.fn();
            jest.mock('@aws-cdk/aws-ecr-assets', () => {
                class DockerImageAsset {
                    constructor(...args) {
                        mockDockerImageAssetConstructor(...args);
                    }
                }
                return {
                    DockerImageAsset,
                };
            });
            // Import the class under test now that the ec2 mock is in place
            const ReloadedStage = jest.requireActual('../lib').Stage; // eslint-disable-line
            class ReloadedStageWithPublicConstructor extends ReloadedStage {
                constructor(props) {
                    super(props);
                }
            }
            // GIVEN
            const recipeName = 'myname';
            const recipe = {
                description: 'some description',
                title: 'some title',
                buildArgs: {
                    a: 'a',
                    b: 'b',
                },
            };
            const stage = new ReloadedStageWithPublicConstructor({
                path: STAGE_PATH,
                manifest: {
                    version: '10.1.9.2',
                    schema: 1,
                    recipes: {
                        [recipeName]: recipe,
                    },
                },
            });
            // WHEN
            stage.imageFromRecipe(stack, 'Image', recipeName);
            // THEN
            expect(mockDockerImageAssetConstructor).toHaveBeenCalledWith(stack, 'Image', {
                ...recipe,
                directory: STAGE_PATH,
            });
            expect(mockDockerImageAssetConstructor.mock.calls[0][2]).not.toMatchObject({
                target: expect.anything(),
            });
        });
        test('throws exception when recipe does not exist', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const invalidRecipeName = 'this-recipe-does-not-exist';
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            function when() {
                stage.imageFromRecipe(stack, 'Image', invalidRecipeName);
            }
            // THEN
            expect(when).toThrowError(`No such recipe: ${invalidRecipeName}`);
        });
    });
    describe('.getVersion()', () => {
        test('returns a version in the success case', () => {
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            const manifest = {
                schema: 1,
                version: '10.1.9.2',
                recipes: {},
            };
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            stage.getVersion(stack, 'Version');
            assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_DEADLINE_INSTALLERS', {
                forceRun: assert_1.stringLike('*'),
                versionString: '10.1.9',
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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