"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = exports.InstanceUserDataProvider = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const version_1 = require("./version");
/**
 * Implementation of {@link IInstanceUserDataProvider}.
 *
 * Can be used as sub-class with override the desired methods
 * to add custom user data commands for WorkerInstanceFleet or WorkerInstanceConfiguration.
 */
class InstanceUserDataProvider extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Method that is invoked before configuring the Cloud Watch Agent.
     *
     * @inheritdoc true
     */
    preCloudWatchAgent(_host) {
    }
    /**
     * Method that is invoked before the render queue configuration.
     *
     * @inheritdoc true
     */
    preRenderQueueConfiguration(_host) {
    }
    /**
     * Method that is invoked after configuring the connection to the render queue and before configuring the Deadline Worker.
     *
     * @inheritdoc true
     */
    preWorkerConfiguration(_host) {
    }
    /**
     * Method that is invoked after all configuration is done and worker started.
     *
     * @inheritdoc true
     */
    postWorkerLaunch(_host) {
    }
}
exports.InstanceUserDataProvider = InstanceUserDataProvider;
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * This configuration performs the following steps in order:
 * 1) Configure Cloud Watch Agent
 * 2) Configure Deadline Worker RenderQueue connection
 * 3) Configure Deadline Worker settings
 *
 * A `userDataProvider` can be specified that defines callback functions.
 * These callbacks can be used to inject user data commands at different points during the Worker instance configuration.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 */
class WorkerInstanceConfiguration extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        (_a = props.userDataProvider) === null || _a === void 0 ? void 0 : _a.preCloudWatchAgent(props.worker);
        if (props.cloudwatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudwatchLogSettings);
        }
        (_b = props.userDataProvider) === null || _b === void 0 ? void 0 : _b.preRenderQueueConfiguration(props.worker);
        (_c = props.renderQueue) === null || _c === void 0 ? void 0 : _c.configureClientInstance({ host: props.worker });
        (_d = props.userDataProvider) === null || _d === void 0 ? void 0 : _d.preWorkerConfiguration(props.worker);
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
        (_e = props.userDataProvider) === null || _e === void 0 ? void 0 : _e.postWorkerLaunch(props.worker);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch service.
     *
     * The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *    for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param logGroupProps Configuration for the log group in CloudWatch.
     */
    configureCloudWatchLogStream(worker, id, logGroupProps) {
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_2.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance.
     *
     * From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     */
    configureWorkerSettings(worker, id, settings) {
        var _a, _b, _c, _d, _e;
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = (_b = (_a = settings === null || settings === void 0 ? void 0 : settings.groups) === null || _a === void 0 ? void 0 : _a.map(val => val.toLowerCase()).join(',')) !== null && _b !== void 0 ? _b : '';
        const pools = (_d = (_c = settings === null || settings === void 0 ? void 0 : settings.pools) === null || _c === void 0 ? void 0 : _c.map(val => val.toLowerCase()).join(',')) !== null && _d !== void 0 ? _d : '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${(_e = settings === null || settings === void 0 ? void 0 : settings.region) !== null && _e !== void 0 ? _e : ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
//# sourceMappingURL=data:application/json;base64,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