"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const crypto_1 = require("crypto");
const path_1 = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 *
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaCode = aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs'));
        const lambdaFunc = new aws_lambda_1.SingletonFunction(this, 'VersionProviderFunction', {
            uuid: '2e19e243-16ee-4d1a-a3c9-18d35eddd446',
            description: 'Used by the Version construct to get installer locations for a specific Deadline version.',
            code: lambdaCode,
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'version-provider.handler',
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        const deadlineProperties = {
            versionString: props === null || props === void 0 ? void 0 : props.version,
            // If we don't have a full static version string, create a random string that will force the Lambda to always
            // run on redeploys, effectively checking for version updates.
            forceRun: this.forceRun(props === null || props === void 0 ? void 0 : props.version),
        };
        const deadlineResource = new core_1.CustomResource(this, 'DeadlineResource', {
            serviceToken: lambdaFunc.functionArn,
            properties: deadlineProperties,
            resourceType: 'Custom::RFDK_DEADLINE_INSTALLERS',
        });
        this.majorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MajorVersion'));
        this.minorVersion = core_1.Token.asNumber(deadlineResource.getAtt('MinorVersion'));
        this.releaseVersion = core_1.Token.asNumber(deadlineResource.getAtt('ReleaseVersion'));
        this.linuxInstallers = {
            patchVersion: core_1.Token.asNumber(deadlineResource.getAtt('LinuxPatchVersion')),
            repository: {
                objectKey: core_1.Token.asString(deadlineResource.getAtt('LinuxRepositoryInstaller')),
                s3Bucket: aws_s3_1.Bucket.fromBucketName(scope, 'InstallerBucket', core_1.Token.asString(deadlineResource.getAtt('S3Bucket'))),
            },
        };
    }
    /**
     * Construct the full version string for the linux patch release referenced in this version object.
     *
     * This is constructed by joining the major, minor,
     * release, and patch versions by dots.
     */
    linuxFullVersionString() {
        const major = core_1.Token.isUnresolved(this.majorVersion) ? core_1.Token.asString(this.majorVersion) : this.majorVersion.toString();
        const minor = core_1.Token.isUnresolved(this.minorVersion) ? core_1.Token.asString(this.minorVersion) : this.minorVersion.toString();
        const release = core_1.Token.isUnresolved(this.releaseVersion) ? core_1.Token.asString(this.releaseVersion) : this.releaseVersion.toString();
        const patch = core_1.Token.isUnresolved(this.linuxInstallers.patchVersion)
            ? core_1.Token.asString(this.linuxInstallers.patchVersion)
            : this.linuxInstallers.patchVersion.toString();
        return `${major}.${minor}.${release}.${patch}`;
    }
    /**
     * Check if we have a full version in the supplied version string. If we don't, we want to make sure the Lambda
     * that fetches the full version number and the installers for it is always run. This allows for Deadline updates
     * to be discovered.
     */
    forceRun(version) {
        return !this.isFullVersion(version) ? crypto_1.randomBytes(32).toString('base64').slice(0, 32) : undefined;
    }
    /**
     * Checks if the supplied version contains the major, minor, release, and patch version numbers,
     * and returns true only if all 4 are supplied.
     */
    isFullVersion(version) {
        const components = version === null || version === void 0 ? void 0 : version.split('.').map(x => parseInt(x));
        if (!components || (components === null || components === void 0 ? void 0 : components.length) != 4) {
            return false;
        }
        return true;
    }
}
exports.VersionQuery = VersionQuery;
//# sourceMappingURL=data:application/json;base64,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