"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 */
class Repository extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q;
        super(scope, id);
        if (props.database && ((_a = props.backupOptions) === null || _a === void 0 ? void 0 : _a.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_b = props.removalPolicy) === null || _b === void 0 ? void 0 : _b.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem and Database components of the repository
        this.fileSystem = (_d = props.fileSystem) !== null && _d !== void 0 ? _d : new core_2.MountableEfs(this, {
            filesystem: new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
            }),
        });
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                this.node.addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_h = props.databaseAuditLogging) !== null && _h !== void 0 ? _h : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_j = props.documentDbInstanceCount) !== null && _j !== void 0 ? _j : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_k = props.vpcSubnets) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                },
                instances,
                backup: {
                    retention: (_m = (_l = props.backupOptions) === null || _l === void 0 ? void 0 : _l.databaseRetention) !== null && _m !== void 0 ? _m : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_p = (_o = props.removalPolicy) === null || _o === void 0 ? void 0 : _o.database) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_q = props.vpcSubnets) !== null && _q !== void 0 ? _q : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _a, _b;
        const hostMountPoint = (_a = props.containerInstances.filesystemMountPoint) !== null && _a !== void 0 ? _a : '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = (_b = props.containers.filesystemMountPoint) !== null && _b !== void 0 ? _b : `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _a;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _a;
        const prefix = (_a = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _a !== void 0 ? _a : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tags.of(this.installerGroup).add(tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                version.linuxFullVersionString(),
            ],
        });
    }
}
exports.Repository = Repository;
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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