"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Generate cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Cannot have a CertificateValidFor property if not given one. Adding one
    // would cause existing certificates to be re-generated on re-deploy, and thus
    // risk breaking customer's setups.
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        CertificateValidFor: assert_1.anything(),
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a policy that can interact with DynamoDB and SecretsManager
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-F53F5427': 'f53f5427b2e9eb4739661fcc0b249b6e',
                        },
                    },
                },
            ],
        },
    }));
    // Expect no KMS key for encryption
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect Lambda for doing the cert generation to use the generate() handler and openssl layer
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        var _a, _b;
        if (!props.Handler || props.Handler !== 'x509-certificate.generate') {
            error.failureReason = 'x509-certificate.generate handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = (_b = (_a = props.Environment) === null || _a === void 0 ? void 0 : _a.Variables) === null || _b === void 0 ? void 0 : _b.DATABASE;
            return false;
        }
        return true;
    }));
    // Should not be any errors.
    expect(cert.node.metadata.length).toBe(0);
});
test('Generate cert, all options set', () => {
    const stack = new core_1.Stack();
    const subject = {
        cn: 'testCN',
        o: 'testO',
        ou: 'testOu',
    };
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    const signingCertificate = new x509_certificate_1.X509CertificatePem(stack, 'SigningCert', { subject });
    const signingCertPassphraseID = stack.getLogicalId(signingCertificate.passphrase.node.defaultChild);
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        encryptionKey,
        signingCertificate,
        validFor: 3000,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    // Expect the custom resource for cert generation
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'testO',
            OU: 'testOu',
        },
        SigningCertificate: {
            Cert: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'SigningCert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: signingCertPassphraseID,
            },
            CertChain: '',
        },
        CertificateValidFor: '3000',
    }));
    // Expect the resource for converting to PKCS #12 not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
    // Expect the DynamoDB table used for custom resource tracking
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table'));
    // Expect a Secret used to store the cert
    assert_1.expect(stack).to(assert_1.haveResource('AWS::SecretsManager::Secret'));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the signing cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'kms:Encrypt',
                        'kms:ReEncrypt*',
                        'kms:GenerateDataKey*',
                    ],
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-B2B09A60': 'b2b09a6086e87fe14005f4e0b800e4f0',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'SigningCert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: 'SigningCertPassphrase42F0BC4F',
                    },
                },
            ],
        },
    }));
    // Expect a policy that can interact with DynamoDB and SecretsManager for the cert's custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: signingCertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-BA0FA489': 'ba0fa4898b2088c5b25f15075f605300',
                        },
                    },
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert generation to use the generate() handler
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.generate',
    }));
});
test('Grant cert read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantCertRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantCertRead() to add this policy
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
            ],
        },
    }));
    // Expect the grantCertRead() not to add this full read policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Grant full read', () => {
    const stack = new core_1.Stack();
    const grantable = new aws_iam_1.Role(stack, 'TestRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
    });
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
    });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    cert.grantFullRead(grantable);
    // Expect the custom resource to be created
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
        DistinguishedName: {
            CN: 'testCN',
            O: 'AWS',
            OU: 'Thinkbox',
        },
    }));
    // Expect the grantFullRead() to add this policy
    assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: 'secretsmanager:GetSecretValue',
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect the PKCS #12 generator not to be created
    assert_1.expect(stack).notTo(assert_1.haveResource('Custom::RFDK_X509_PKCS12'));
});
test('Validating expiry', () => {
    // GIVEN
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: 0,
    });
    // THEN
    expect(cert.node.metadata.length).toBe(1);
});
test('Validating expiry with token', () => {
    // GIVEN
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const subject = { cn: 'testCN' };
    // A numeric CDK token (see: https://docs.aws.amazon.com/cdk/latest/guide/tokens.html#tokens_number)
    const CDK_NUMERIC_TOKEN = -1.8881545897087626e+289;
    // WHEN
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', {
        subject,
        validFor: CDK_NUMERIC_TOKEN,
    });
    // THEN
    expect(cert.node.metadata.length).toBe(0);
});
test('Convert to PKCS #12', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const cert = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(cert.passphrase.node.defaultChild);
    const pkcs12Cert = new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', { sourceCertificate: cert });
    const pkcs12CertPassphraseID = stack.getLogicalId(pkcs12Cert.passphrase.node.defaultChild);
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect the PKCS #12 resource to have a policy with access to the X.509 resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: pkcs12CertPassphraseID,
                    },
                },
                {
                    Action: [
                        'secretsmanager:CreateSecret',
                        'secretsmanager:DeleteSecret',
                        'secretsmanager:TagResource',
                        'secretsmanager:PutSecretValue',
                    ],
                    Condition: {
                        StringEquals: {
                            'secretsmanager:ResourceTag/X509SecretGrant-71090F78': '71090f7809ce64f7c970cb645d4d473c',
                        },
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'Cert',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
            ],
        },
    }));
    // Expect no KMS key
    assert_1.expect(stack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS 12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
test('Convert to PKCS #12, use KMS', () => {
    const stack = new core_1.Stack();
    const subject = { cn: 'testCN' };
    const sourceCertificate = new x509_certificate_1.X509CertificatePem(stack, 'Cert', { subject });
    const certPassphraseID = stack.getLogicalId(sourceCertificate.passphrase.node.defaultChild);
    const encryptionKey = new aws_kms_1.Key(stack, 'Key');
    new x509_certificate_1.X509CertificatePkcs12(stack, 'CertPkcs12', {
        sourceCertificate,
        encryptionKey,
    });
    // Expect the PKCS #12 custom resource
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_X509_PKCS12', {
        Passphrase: {
            Ref: 'CertPkcs12Passphrase1E3DF360',
        },
        Secret: {
            NamePrefix: 'Default/CertPkcs12',
            Description: 'Default/CertPkcs12',
            Tags: [
                {
                    Key: 'X509SecretGrant-71090F78',
                    Value: '71090f7809ce64f7c970cb645d4d473c',
                },
            ],
        },
        Certificate: {
            Cert: {
                'Fn::GetAtt': [
                    'Cert',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Cert',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
        },
    }));
    // Expect the source certificate (custom resource)
    assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_X509Generator'));
    // Expect the PKCS #12 to have a password secret
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
        Description: 'Passphrase for the private key of the X509Certificate CertPkcs12',
        GenerateSecretString: {
            ExcludeCharacters: '"()$\'',
            ExcludePunctuation: true,
            IncludeSpace: false,
            PasswordLength: 24,
            RequireEachIncludedType: true,
        },
    }));
    // Expect a KMS key for encryption
    assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key'));
    // Expect the Lambda for converting the PEM to PKCS #12
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
        Handler: 'x509-certificate.convert',
    }));
});
//# sourceMappingURL=data:application/json;base64,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