"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const tag_helpers_1 = require("./tag-helpers");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
describe('HealthMonitor', () => {
    beforeEach(() => {
        app = new core_1.App();
        infraStack = new core_1.Stack(app, 'infraStack');
        hmStack = new core_1.Stack(app, 'hmStack');
        wfStack = new core_1.Stack(app, 'wfStack');
        vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
    });
    test('validating default health monitor properties', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: '*',
                    },
                ],
            },
            Description: `This key is used to encrypt SNS messages for ${healthMonitor.node.uniqueId}.`,
            EnableKeyRotation: true,
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::TopicPolicy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'cloudwatch.amazonaws.com',
                        },
                        Resource: {
                            Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                        },
                        Sid: '0',
                    },
                ],
            },
            Topics: [
                {
                    Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                },
            ],
        }));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: {
                'Fn::GetAtt': [
                    `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
                    'Arn',
                ],
            },
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating health monitor properties while passing a key', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
            encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
        });
        // THEN
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
        assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::KMS::Key'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            KmsMasterKeyId: 'arn:aws:kms:us-west-2:123456789012:key/testarn',
        }));
        assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                    'Arn',
                ],
            },
        }));
    });
    test('validating the target with default health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(hmStack).notTo((assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
            SecurityGroupIngress: assert_1.arrayWith(assert_1.deepObjectLike({
                CidrIp: '0.0.0.0/0',
                FromPort: 8081,
                IpProtocol: 'tcp',
                ToPort: 8081,
            })),
        })));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '8081',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('validating the target with custom health config', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {
            port: 7171,
        });
        // THEN
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
    });
    test('2 ASG gets registered to same LB', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, { port: 7171 });
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, { port: 7171 });
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckIntervalSeconds: 300,
            HealthCheckPort: '7171',
            HealthCheckProtocol: 'HTTP',
            Port: 8081,
            Protocol: 'HTTP',
            TargetType: 'instance',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 8,
            ActionsEnabled: true,
            DatapointsToAlarm: 8,
            Threshold: 0,
            TreatMissingData: 'notBreaching',
        }));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
            ComparisonOperator: 'GreaterThanThreshold',
            EvaluationPeriods: 1,
            ActionsEnabled: true,
            DatapointsToAlarm: 1,
            Threshold: 35,
            TreatMissingData: 'notBreaching',
        }));
    });
    test('validating LB target limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 50,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
            minCapacity: 50,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating LB listener limit', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'listeners-per-application-load-balancer',
                    max: 1,
                }, {
                    name: 'target-groups-per-action-on-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target group limit per lb', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'target-groups-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
            vpc,
        });
        healthMonitor.registerFleet(fleet2, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
            Scheme: 'internal',
            Type: 'application',
        }));
        assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
        assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
            Port: 8081,
            Protocol: 'HTTP',
        }));
    });
    test('validating target limit exhaustion', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            elbAccountLimits: [{
                    name: 'targets-per-application-load-balancer',
                    max: 1,
                }],
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
            minCapacity: 2,
        });
        expect(() => {
            healthMonitor.registerFleet(fleet, {});
        }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
    });
    test('validating deletion protection', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
            deletionProtection: false,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // WHEN
        healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
            vpc,
        });
        const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
            vpc,
        });
        healthMonitor.registerFleet(fleet, {});
        // THEN
        assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('tagging', () => {
        tag_helpers_1.testConstructTags({
            constructName: 'HealthMonitor',
            createConstruct: () => {
                // GIVEN
                const fleetStack = new core_1.Stack(app, 'FleetStack');
                const fleet = new TestMonitorableFleet(fleetStack, 'workerFleet', {
                    vpc,
                });
                // WHEN
                healthMonitor = new lib_1.HealthMonitor(hmStack, 'HealthMonitor', {
                    vpc,
                });
                healthMonitor.registerFleet(fleet, {});
                return hmStack;
            },
            resourceTypeCounts: {
                'AWS::KMS::Key': 1,
                'AWS::SNS::Topic': 1,
                'AWS::ElasticLoadBalancingV2::LoadBalancer': 1,
                'AWS::EC2::SecurityGroup': 1,
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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