"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 *
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Elastic Network Interface (ENI).
 * - Security Group for the ASG.
 * - Instance Role and corresponding IAM Policy.
 * - SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack.
 * - Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is deployed through this construct has broad IAM permissions to attach any Elastic
 *    Network Interface (ENI) to any instance. You should not grant any additional actors/principals the ability
 *    to modify or execute this Lambda.
 * - The SNS Topic that is deployed through this construct controls the execution of the Lambda discussed above.
 *    Principals that can publish messages to this SNS Topic will be able to trigger the Lambda to run. You should
 *    not allow any additional principals to publish messages to this SNS Topic.
 */
class StaticPrivateIpServer extends core_1.Construct {
    /**
     *
     */
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance in this construct's AutoScalingGroup when a new instance is launched.
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                description: `Created by RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = eventHandler.node.uniqueId;
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tags.of(this.autoscalingGroup).add(tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events.
     *
     * Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     *
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns : Topic, role: Role }
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For RFDK instance-launch notifications for stack '${stack.stackName}'`,
            });
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
//# sourceMappingURL=data:application/json;base64,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