"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk-mock");
const sinon = require("sinon");
const replica_handler_1 = require("../lib/replica-handler");
AWS.setSDK(require.resolve('aws-sdk'));
const createEvent = {
    RequestType: 'Create',
    ResourceProperties: {
        TableName: 'my-table',
        Region: 'eu-west-2',
        ServiceToken: 'token',
    },
    ServiceToken: 'token',
    ResponseURL: 'url',
    LogicalResourceId: 'logical-id',
    RequestId: 'request-id',
    StackId: 'stack-id',
    ResourceType: 'resource-type',
};
afterEach(() => {
    AWS.restore();
});
test('on event', async () => {
    const updateTableMock = sinon.fake.resolves({});
    AWS.mock('DynamoDB', 'updateTable', updateTableMock);
    const data = await replica_handler_1.onEventHandler(createEvent);
    sinon.assert.calledWith(updateTableMock, {
        TableName: 'my-table',
        ReplicaUpdates: [
            {
                Create: {
                    RegionName: 'eu-west-2',
                },
            },
        ],
    });
    expect(data).toEqual({
        PhysicalResourceId: 'eu-west-2',
    });
});
test('on event does not call updateTable for Update requests', async () => {
    const updateTableMock = sinon.fake.resolves({});
    AWS.mock('DynamoDB', 'updateTable', updateTableMock);
    const data = await replica_handler_1.onEventHandler({
        ...createEvent,
        RequestType: 'Update',
    });
    sinon.assert.notCalled(updateTableMock);
    expect(data).toEqual({
        PhysicalResourceId: 'eu-west-2',
    });
});
test('is complete for create returns false without replicas', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {},
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns false when replica is not active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'CREATING',
                },
            ],
        },
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns false when table is not active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'ACTIVE',
                },
            ],
            TableStatus: 'UPDATING',
        },
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: false });
});
test('is complete for create returns true when replica is active', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-2',
                    ReplicaStatus: 'ACTIVE',
                },
            ],
            TableStatus: 'ACTIVE',
        },
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler(createEvent);
    expect(data).toEqual({ IsComplete: true });
});
test('is complete for delete returns true when replica is gone', async () => {
    const describeTableMock = sinon.fake.resolves({
        Table: {
            Replicas: [
                {
                    RegionName: 'eu-west-1',
                    ReplicaStatus: 'ACTIVE',
                },
            ],
            TableStatus: 'ACTIVE',
        },
    });
    AWS.mock('DynamoDB', 'describeTable', describeTableMock);
    const data = await replica_handler_1.isCompleteHandler({
        ...createEvent,
        RequestType: 'Delete',
    });
    expect(data).toEqual({ IsComplete: true });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicmVwbGljYS1wcm92aWRlci50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicmVwbGljYS1wcm92aWRlci50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQ0Esb0NBQW9DO0FBQ3BDLCtCQUErQjtBQUMvQiw0REFBMkU7QUFFM0UsR0FBRyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7QUFFdkMsTUFBTSxXQUFXLEdBQW1CO0lBQ2xDLFdBQVcsRUFBRSxRQUFRO0lBQ3JCLGtCQUFrQixFQUFFO1FBQ2xCLFNBQVMsRUFBRSxVQUFVO1FBQ3JCLE1BQU0sRUFBRSxXQUFXO1FBQ25CLFlBQVksRUFBRSxPQUFPO0tBQ3RCO0lBQ0QsWUFBWSxFQUFFLE9BQU87SUFDckIsV0FBVyxFQUFFLEtBQUs7SUFDbEIsaUJBQWlCLEVBQUUsWUFBWTtJQUMvQixTQUFTLEVBQUUsWUFBWTtJQUN2QixPQUFPLEVBQUUsVUFBVTtJQUNuQixZQUFZLEVBQUUsZUFBZTtDQUM5QixDQUFDO0FBRUYsU0FBUyxDQUFDLEdBQUcsRUFBRTtJQUNiLEdBQUcsQ0FBQyxPQUFPLEVBQUUsQ0FBQztBQUNoQixDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxVQUFVLEVBQUUsS0FBSyxJQUFJLEVBQUU7SUFDMUIsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFaEQsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsYUFBYSxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sSUFBSSxHQUFHLE1BQU0sZ0NBQWMsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUUvQyxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxlQUFlLEVBQUU7UUFDdkMsU0FBUyxFQUFFLFVBQVU7UUFDckIsY0FBYyxFQUFFO1lBQ2Q7Z0JBQ0UsTUFBTSxFQUFFO29CQUNOLFVBQVUsRUFBRSxXQUFXO2lCQUN4QjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ25CLGtCQUFrQixFQUFFLFdBQVc7S0FDaEMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0RBQXdELEVBQUUsS0FBSyxJQUFJLEVBQUU7SUFDeEUsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7SUFFaEQsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsYUFBYSxFQUFFLGVBQWUsQ0FBQyxDQUFDO0lBRXJELE1BQU0sSUFBSSxHQUFHLE1BQU0sZ0NBQWMsQ0FBQztRQUNoQyxHQUFHLFdBQVc7UUFDZCxXQUFXLEVBQUUsUUFBUTtLQUN0QixDQUFDLENBQUM7SUFFSCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxlQUFlLENBQUMsQ0FBQztJQUV4QyxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDO1FBQ25CLGtCQUFrQixFQUFFLFdBQVc7S0FDaEMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsS0FBSyxJQUFJLEVBQUU7SUFDdkUsTUFBTSxpQkFBaUIsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUM1QyxLQUFLLEVBQUUsRUFBRTtLQUNWLENBQUMsQ0FBQztJQUVILEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBRXpELE1BQU0sSUFBSSxHQUFHLE1BQU0sbUNBQWlCLENBQUMsV0FBVyxDQUFDLENBQUM7SUFFbEQsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLFVBQVUsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO0FBQzlDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGlFQUFpRSxFQUFFLEtBQUssSUFBSSxFQUFFO0lBQ2pGLE1BQU0saUJBQWlCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDNUMsS0FBSyxFQUFFO1lBQ0wsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFVBQVUsRUFBRSxXQUFXO29CQUN2QixhQUFhLEVBQUUsVUFBVTtpQkFDMUI7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsZUFBZSxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFFekQsTUFBTSxJQUFJLEdBQUcsTUFBTSxtQ0FBaUIsQ0FBQyxXQUFXLENBQUMsQ0FBQztJQUVsRCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEVBQUUsVUFBVSxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7QUFDOUMsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsK0RBQStELEVBQUUsS0FBSyxJQUFJLEVBQUU7SUFDL0UsTUFBTSxpQkFBaUIsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztRQUM1QyxLQUFLLEVBQUU7WUFDTCxRQUFRLEVBQUU7Z0JBQ1I7b0JBQ0UsVUFBVSxFQUFFLFdBQVc7b0JBQ3ZCLGFBQWEsRUFBRSxRQUFRO2lCQUN4QjthQUNGO1lBQ0QsV0FBVyxFQUFFLFVBQVU7U0FDeEI7S0FDRixDQUFDLENBQUM7SUFFSCxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFlLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUV6RCxNQUFNLElBQUksR0FBRyxNQUFNLG1DQUFpQixDQUFDLFdBQVcsQ0FBQyxDQUFDO0lBRWxELE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsRUFBRSxVQUFVLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztBQUM5QyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw0REFBNEQsRUFBRSxLQUFLLElBQUksRUFBRTtJQUM1RSxNQUFNLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDO1FBQzVDLEtBQUssRUFBRTtZQUNMLFFBQVEsRUFBRTtnQkFDUjtvQkFDRSxVQUFVLEVBQUUsV0FBVztvQkFDdkIsYUFBYSxFQUFFLFFBQVE7aUJBQ3hCO2FBQ0Y7WUFDRCxXQUFXLEVBQUUsUUFBUTtTQUN0QjtLQUNGLENBQUMsQ0FBQztJQUVILEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLGVBQWUsRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO0lBRXpELE1BQU0sSUFBSSxHQUFHLE1BQU0sbUNBQWlCLENBQUMsV0FBVyxDQUFDLENBQUM7SUFFbEQsTUFBTSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU8sQ0FBQyxFQUFFLFVBQVUsRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO0FBQzdDLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBEQUEwRCxFQUFFLEtBQUssSUFBSSxFQUFFO0lBQzFFLE1BQU0saUJBQWlCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7UUFDNUMsS0FBSyxFQUFFO1lBQ0wsUUFBUSxFQUFFO2dCQUNSO29CQUNFLFVBQVUsRUFBRSxXQUFXO29CQUN2QixhQUFhLEVBQUUsUUFBUTtpQkFDeEI7YUFDRjtZQUNELFdBQVcsRUFBRSxRQUFRO1NBQ3RCO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsZUFBZSxFQUFFLGlCQUFpQixDQUFDLENBQUM7SUFFekQsTUFBTSxJQUFJLEdBQUcsTUFBTSxtQ0FBaUIsQ0FBQztRQUNuQyxHQUFHLFdBQVc7UUFDZCxXQUFXLEVBQUUsUUFBUTtLQUN0QixDQUFDLENBQUM7SUFFSCxNQUFNLENBQUMsSUFBSSxDQUFDLENBQUMsT0FBTyxDQUFDLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7QUFDN0MsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBPbkV2ZW50UmVxdWVzdCB9IGZyb20gJ0Bhd3MtY2RrL2N1c3RvbS1yZXNvdXJjZXMvbGliL3Byb3ZpZGVyLWZyYW1ld29yay90eXBlcyc7XG5pbXBvcnQgKiBhcyBBV1MgZnJvbSAnYXdzLXNkay1tb2NrJztcbmltcG9ydCAqIGFzIHNpbm9uIGZyb20gJ3Npbm9uJztcbmltcG9ydCB7IGlzQ29tcGxldGVIYW5kbGVyLCBvbkV2ZW50SGFuZGxlciB9IGZyb20gJy4uL2xpYi9yZXBsaWNhLWhhbmRsZXInO1xuXG5BV1Muc2V0U0RLKHJlcXVpcmUucmVzb2x2ZSgnYXdzLXNkaycpKTtcblxuY29uc3QgY3JlYXRlRXZlbnQ6IE9uRXZlbnRSZXF1ZXN0ID0ge1xuICBSZXF1ZXN0VHlwZTogJ0NyZWF0ZScsXG4gIFJlc291cmNlUHJvcGVydGllczoge1xuICAgIFRhYmxlTmFtZTogJ215LXRhYmxlJyxcbiAgICBSZWdpb246ICdldS13ZXN0LTInLFxuICAgIFNlcnZpY2VUb2tlbjogJ3Rva2VuJyxcbiAgfSxcbiAgU2VydmljZVRva2VuOiAndG9rZW4nLFxuICBSZXNwb25zZVVSTDogJ3VybCcsXG4gIExvZ2ljYWxSZXNvdXJjZUlkOiAnbG9naWNhbC1pZCcsXG4gIFJlcXVlc3RJZDogJ3JlcXVlc3QtaWQnLFxuICBTdGFja0lkOiAnc3RhY2staWQnLFxuICBSZXNvdXJjZVR5cGU6ICdyZXNvdXJjZS10eXBlJyxcbn07XG5cbmFmdGVyRWFjaCgoKSA9PiB7XG4gIEFXUy5yZXN0b3JlKCk7XG59KTtcblxudGVzdCgnb24gZXZlbnQnLCBhc3luYyAoKSA9PiB7XG4gIGNvbnN0IHVwZGF0ZVRhYmxlTW9jayA9IHNpbm9uLmZha2UucmVzb2x2ZXMoe30pO1xuXG4gIEFXUy5tb2NrKCdEeW5hbW9EQicsICd1cGRhdGVUYWJsZScsIHVwZGF0ZVRhYmxlTW9jayk7XG5cbiAgY29uc3QgZGF0YSA9IGF3YWl0IG9uRXZlbnRIYW5kbGVyKGNyZWF0ZUV2ZW50KTtcblxuICBzaW5vbi5hc3NlcnQuY2FsbGVkV2l0aCh1cGRhdGVUYWJsZU1vY2ssIHtcbiAgICBUYWJsZU5hbWU6ICdteS10YWJsZScsXG4gICAgUmVwbGljYVVwZGF0ZXM6IFtcbiAgICAgIHtcbiAgICAgICAgQ3JlYXRlOiB7XG4gICAgICAgICAgUmVnaW9uTmFtZTogJ2V1LXdlc3QtMicsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIF0sXG4gIH0pO1xuXG4gIGV4cGVjdChkYXRhKS50b0VxdWFsKHtcbiAgICBQaHlzaWNhbFJlc291cmNlSWQ6ICdldS13ZXN0LTInLFxuICB9KTtcbn0pO1xuXG50ZXN0KCdvbiBldmVudCBkb2VzIG5vdCBjYWxsIHVwZGF0ZVRhYmxlIGZvciBVcGRhdGUgcmVxdWVzdHMnLCBhc3luYyAoKSA9PiB7XG4gIGNvbnN0IHVwZGF0ZVRhYmxlTW9jayA9IHNpbm9uLmZha2UucmVzb2x2ZXMoe30pO1xuXG4gIEFXUy5tb2NrKCdEeW5hbW9EQicsICd1cGRhdGVUYWJsZScsIHVwZGF0ZVRhYmxlTW9jayk7XG5cbiAgY29uc3QgZGF0YSA9IGF3YWl0IG9uRXZlbnRIYW5kbGVyKHtcbiAgICAuLi5jcmVhdGVFdmVudCxcbiAgICBSZXF1ZXN0VHlwZTogJ1VwZGF0ZScsXG4gIH0pO1xuXG4gIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlVGFibGVNb2NrKTtcblxuICBleHBlY3QoZGF0YSkudG9FcXVhbCh7XG4gICAgUGh5c2ljYWxSZXNvdXJjZUlkOiAnZXUtd2VzdC0yJyxcbiAgfSk7XG59KTtcblxudGVzdCgnaXMgY29tcGxldGUgZm9yIGNyZWF0ZSByZXR1cm5zIGZhbHNlIHdpdGhvdXQgcmVwbGljYXMnLCBhc3luYyAoKSA9PiB7XG4gIGNvbnN0IGRlc2NyaWJlVGFibGVNb2NrID0gc2lub24uZmFrZS5yZXNvbHZlcyh7XG4gICAgVGFibGU6IHt9LFxuICB9KTtcblxuICBBV1MubW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGRlc2NyaWJlVGFibGVNb2NrKTtcblxuICBjb25zdCBkYXRhID0gYXdhaXQgaXNDb21wbGV0ZUhhbmRsZXIoY3JlYXRlRXZlbnQpO1xuXG4gIGV4cGVjdChkYXRhKS50b0VxdWFsKHsgSXNDb21wbGV0ZTogZmFsc2UgfSk7XG59KTtcblxudGVzdCgnaXMgY29tcGxldGUgZm9yIGNyZWF0ZSByZXR1cm5zIGZhbHNlIHdoZW4gcmVwbGljYSBpcyBub3QgYWN0aXZlJywgYXN5bmMgKCkgPT4ge1xuICBjb25zdCBkZXNjcmliZVRhYmxlTW9jayA9IHNpbm9uLmZha2UucmVzb2x2ZXMoe1xuICAgIFRhYmxlOiB7XG4gICAgICBSZXBsaWNhczogW1xuICAgICAgICB7XG4gICAgICAgICAgUmVnaW9uTmFtZTogJ2V1LXdlc3QtMicsXG4gICAgICAgICAgUmVwbGljYVN0YXR1czogJ0NSRUFUSU5HJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgfSxcbiAgfSk7XG5cbiAgQVdTLm1vY2soJ0R5bmFtb0RCJywgJ2Rlc2NyaWJlVGFibGUnLCBkZXNjcmliZVRhYmxlTW9jayk7XG5cbiAgY29uc3QgZGF0YSA9IGF3YWl0IGlzQ29tcGxldGVIYW5kbGVyKGNyZWF0ZUV2ZW50KTtcblxuICBleHBlY3QoZGF0YSkudG9FcXVhbCh7IElzQ29tcGxldGU6IGZhbHNlIH0pO1xufSk7XG5cbnRlc3QoJ2lzIGNvbXBsZXRlIGZvciBjcmVhdGUgcmV0dXJucyBmYWxzZSB3aGVuIHRhYmxlIGlzIG5vdCBhY3RpdmUnLCBhc3luYyAoKSA9PiB7XG4gIGNvbnN0IGRlc2NyaWJlVGFibGVNb2NrID0gc2lub24uZmFrZS5yZXNvbHZlcyh7XG4gICAgVGFibGU6IHtcbiAgICAgIFJlcGxpY2FzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBSZWdpb25OYW1lOiAnZXUtd2VzdC0yJyxcbiAgICAgICAgICBSZXBsaWNhU3RhdHVzOiAnQUNUSVZFJyxcbiAgICAgICAgfSxcbiAgICAgIF0sXG4gICAgICBUYWJsZVN0YXR1czogJ1VQREFUSU5HJyxcbiAgICB9LFxuICB9KTtcblxuICBBV1MubW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGRlc2NyaWJlVGFibGVNb2NrKTtcblxuICBjb25zdCBkYXRhID0gYXdhaXQgaXNDb21wbGV0ZUhhbmRsZXIoY3JlYXRlRXZlbnQpO1xuXG4gIGV4cGVjdChkYXRhKS50b0VxdWFsKHsgSXNDb21wbGV0ZTogZmFsc2UgfSk7XG59KTtcblxudGVzdCgnaXMgY29tcGxldGUgZm9yIGNyZWF0ZSByZXR1cm5zIHRydWUgd2hlbiByZXBsaWNhIGlzIGFjdGl2ZScsIGFzeW5jICgpID0+IHtcbiAgY29uc3QgZGVzY3JpYmVUYWJsZU1vY2sgPSBzaW5vbi5mYWtlLnJlc29sdmVzKHtcbiAgICBUYWJsZToge1xuICAgICAgUmVwbGljYXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZ2lvbk5hbWU6ICdldS13ZXN0LTInLFxuICAgICAgICAgIFJlcGxpY2FTdGF0dXM6ICdBQ1RJVkUnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIFRhYmxlU3RhdHVzOiAnQUNUSVZFJyxcbiAgICB9LFxuICB9KTtcblxuICBBV1MubW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGRlc2NyaWJlVGFibGVNb2NrKTtcblxuICBjb25zdCBkYXRhID0gYXdhaXQgaXNDb21wbGV0ZUhhbmRsZXIoY3JlYXRlRXZlbnQpO1xuXG4gIGV4cGVjdChkYXRhKS50b0VxdWFsKHsgSXNDb21wbGV0ZTogdHJ1ZSB9KTtcbn0pO1xuXG50ZXN0KCdpcyBjb21wbGV0ZSBmb3IgZGVsZXRlIHJldHVybnMgdHJ1ZSB3aGVuIHJlcGxpY2EgaXMgZ29uZScsIGFzeW5jICgpID0+IHtcbiAgY29uc3QgZGVzY3JpYmVUYWJsZU1vY2sgPSBzaW5vbi5mYWtlLnJlc29sdmVzKHtcbiAgICBUYWJsZToge1xuICAgICAgUmVwbGljYXM6IFtcbiAgICAgICAge1xuICAgICAgICAgIFJlZ2lvbk5hbWU6ICdldS13ZXN0LTEnLFxuICAgICAgICAgIFJlcGxpY2FTdGF0dXM6ICdBQ1RJVkUnLFxuICAgICAgICB9LFxuICAgICAgXSxcbiAgICAgIFRhYmxlU3RhdHVzOiAnQUNUSVZFJyxcbiAgICB9LFxuICB9KTtcblxuICBBV1MubW9jaygnRHluYW1vREInLCAnZGVzY3JpYmVUYWJsZScsIGRlc2NyaWJlVGFibGVNb2NrKTtcblxuICBjb25zdCBkYXRhID0gYXdhaXQgaXNDb21wbGV0ZUhhbmRsZXIoe1xuICAgIC4uLmNyZWF0ZUV2ZW50LFxuICAgIFJlcXVlc3RUeXBlOiAnRGVsZXRlJyxcbiAgfSk7XG5cbiAgZXhwZWN0KGRhdGEpLnRvRXF1YWwoeyBJc0NvbXBsZXRlOiB0cnVlIH0pO1xufSk7XG4iXX0=