# Python SCALE Codec Library
#
# Copyright 2018-2020 openAware BV (NL).
# This file is part of Polkascan.
#
# Polkascan is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Polkascan is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Polkascan. If not, see <http://www.gnu.org/licenses/>.

import datetime
import unittest

from scalecodec.metadata import MetadataDecoder

from scalecodec.base import ScaleDecoder, ScaleBytes, RemainingScaleBytesNotEmptyException, \
    InvalidScaleTypeValueException, RuntimeConfiguration
from scalecodec.type_registry import load_type_registry_preset


class TestScaleTypes(unittest.TestCase):

    @classmethod
    def setUpClass(cls):
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("default"))
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("kusama"))
        RuntimeConfiguration().set_active_spec_version_id(1045)

        metadata_v10_hex = "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"

        cls.metadata_decoder = MetadataDecoder(ScaleBytes(metadata_v10_hex))
        cls.metadata_decoder.decode()

    def test_compact_u32(self):
        obj = ScaleDecoder.get_decoder_class('Compact<u32>', ScaleBytes("0x02093d00"))
        obj.decode()
        self.assertEqual(obj.value, 1000000)

    def test_compact_u32_1byte(self):
        obj = ScaleDecoder.get_decoder_class('Compact<u32>', ScaleBytes("0x18"))
        obj.decode()
        self.assertEqual(obj.value, 6)

    def test_compact_u32_remaining_bytes(self):
        obj = ScaleDecoder.get_decoder_class('Compact<u32>', ScaleBytes("0x02093d0001"))
        self.assertRaises(RemainingScaleBytesNotEmptyException, obj.decode)

    def test_compact_u32_invalid(self):
        obj = ScaleDecoder.get_decoder_class('Compact<u32>', ScaleBytes("0x"))
        self.assertRaises(InvalidScaleTypeValueException, obj.decode)

    def test_u16(self):
        obj = ScaleDecoder.get_decoder_class('u16', ScaleBytes("0x2efb"))
        obj.decode()
        self.assertEqual(obj.value, 64302)

    def test_i16(self):
        obj = ScaleDecoder.get_decoder_class('i16', ScaleBytes("0x2efb"))
        obj.decode()
        self.assertEqual(obj.value, -1234)

    def test_compact_bool_true(self):
        obj = ScaleDecoder.get_decoder_class('bool', ScaleBytes("0x01"))
        obj.decode()
        self.assertEqual(obj.value, True)

    def test_compact_bool_false(self):
        obj = ScaleDecoder.get_decoder_class('bool', ScaleBytes("0x00"))
        obj.decode()
        self.assertEqual(obj.value, False)

    def test_compact_bool_invalid(self):
        obj = ScaleDecoder.get_decoder_class('bool', ScaleBytes("0x02"))
        self.assertRaises(InvalidScaleTypeValueException, obj.decode)

    def test_vec_accountid(self):
        obj = ScaleDecoder.get_decoder_class(
            'Vec<AccountId>',
            ScaleBytes("0x0865d2273adeb04478658e183dc5edf41f1d86e42255442af62e72dbf1e6c0b97765d2273adeb04478658e183dc5edf41f1d86e42255442af62e72dbf1e6c0b977")
        )
        obj.decode()
        self.assertListEqual(obj.value, [
            '0x65d2273adeb04478658e183dc5edf41f1d86e42255442af62e72dbf1e6c0b977',
            '0x65d2273adeb04478658e183dc5edf41f1d86e42255442af62e72dbf1e6c0b977'
        ])

    def test_validatorprefs_struct(self):
        obj = ScaleDecoder.get_decoder_class('ValidatorPrefsLegacy', ScaleBytes("0x0c00"))
        obj.decode()
        self.assertEqual(obj.value, {'unstakeThreshold': 3, 'validatorPayment': 0})

    def test_implied_struct(self):
        obj = ScaleDecoder.get_decoder_class('(Compact<u32>,Compact<u32>)', ScaleBytes("0x0c00"))
        obj.decode()
        self.assertEqual(obj.value, {"col1": 3, "col2": 0})

    def test_address(self):
        obj = ScaleDecoder.get_decoder_class(
            'Address',
            ScaleBytes("0xff1fa9d1bd1db014b65872ee20aee4fd4d3a942d95d3357f463ea6c799130b6318")
        )
        obj.decode()
        self.assertEqual(obj.value, '1fa9d1bd1db014b65872ee20aee4fd4d3a942d95d3357f463ea6c799130b6318')

    def test_moment(self):
        obj = ScaleDecoder.get_decoder_class('Compact<Moment>', ScaleBytes("0x03d68b655c"))
        obj.decode()
        self.assertEqual(obj.value, datetime.datetime(2019, 2, 14, 15, 40, 6))

    def test_balance(self):
        obj = ScaleDecoder.get_decoder_class('Compact<Balance>', ScaleBytes("0x130080cd103d71bc22"))
        obj.decode()
        self.assertEqual(obj.value, 2503000000000000000)

    def test_type_registry(self):
        # Example type SpecificTestType only define in type registry 'default'
        self.assertRaises(NotImplementedError, ScaleDecoder.get_decoder_class, 'SpecificTestType', ScaleBytes("0x01000000"))

        RuntimeConfiguration().update_type_registry(load_type_registry_preset("test"))

        obj = ScaleDecoder.get_decoder_class('SpecificTestType', ScaleBytes("0x06000000"))
        obj.decode()
        self.assertEqual(obj.value, 6)

    def test_type_registry_overloading(self):
        # Type BlockNumber defined as U32 in type registry 'kusama'
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("kusama"))

        obj = ScaleDecoder.get_decoder_class('BlockNumber', ScaleBytes("0x0000000000000001"))
        self.assertRaises(RemainingScaleBytesNotEmptyException, obj.decode)

        # Type BlockNumber changed to U64 in type registry 'test'
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("test"))

        obj = ScaleDecoder.get_decoder_class('BlockNumber', ScaleBytes("0x0000000000000001"))
        obj.decode()
        self.assertEqual(obj.value, 72057594037927936)

    def test_unknown_decoder_class(self):
        self.assertRaises(NotImplementedError, ScaleDecoder.get_decoder_class, 'UnknownType123', ScaleBytes("0x0c00"))

    def test_dynamic_set(self):
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("default"))

        obj = ScaleDecoder.get_decoder_class('WithdrawReasons', ScaleBytes("0x0100000000000000"))
        obj.decode()

        self.assertEqual(obj.value, ["TransactionPayment"])

        obj = ScaleDecoder.get_decoder_class('WithdrawReasons', ScaleBytes("0x0300000000000000"))
        obj.decode()

        self.assertEqual(obj.value, ["TransactionPayment", "Transfer"])

        obj = ScaleDecoder.get_decoder_class('WithdrawReasons', ScaleBytes("0x1600000000000000"))
        obj.decode()

        self.assertEqual(obj.value, ["Transfer", "Reserve", "Tip"])

    def test_set_value_type_u32(self):
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("default"))

        # Create set type with u32
        RuntimeConfiguration().update_type_registry({
            "types": {
                "CustomU32Set": {
                    "type": "set",
                    "value_type": "u32",
                    "value_list": {
                        "Value1": 1,
                        "Value2": 2,
                        "Value3": 4,
                        "Value4": 8,
                        "Value5": 16
                    }
                }
            }
        })

        obj = ScaleDecoder.get_decoder_class('CustomU32Set', ScaleBytes("0x0100000000000000"))
        self.assertRaises(RemainingScaleBytesNotEmptyException, obj.decode)

        obj = ScaleDecoder.get_decoder_class('CustomU32Set', ScaleBytes("0x01000000"))
        obj.decode()

        self.assertEqual(obj.value, ["Value1"])

        obj = ScaleDecoder.get_decoder_class('CustomU32Set', ScaleBytes("0x03000000"))
        obj.decode()

        self.assertEqual(obj.value, ["Value1", "Value2"])

        obj = ScaleDecoder.get_decoder_class('CustomU32Set', ScaleBytes("0x16000000"))
        obj.decode()

        self.assertEqual(obj.value, ["Value2", "Value3", "Value5"])

    def test_box_call(self):
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("default"))

        scale_value = ScaleBytes("0x0400ff6e57561de4b4e63f0af8bf336008252a9597e5cdcb7622c72de4ff39731c5402070010a5d4e8")

        obj = ScaleDecoder.get_decoder_class('Box<Call>', scale_value, metadata=self.metadata_decoder)
        value = obj.decode()

        self.assertEqual(value['call_function'], 'transfer')
        self.assertEqual(value['call_module'], 'Balances')
        self.assertEqual(value['call_args'][0]['value'], '0x6e57561de4b4e63f0af8bf336008252a9597e5cdcb7622c72de4ff39731c5402')
        self.assertEqual(value['call_args'][1]['value'], 1000000000000)

    def test_parse_subtype(self):
        RuntimeConfiguration().update_type_registry(load_type_registry_preset("default"))

        obj = ScaleDecoder.get_decoder_class('(BalanceOf, Vec<(AccountId, Data)>)')

        self.assertEqual(obj.type_mapping[0][1], "BalanceOf")
        self.assertEqual(obj.type_mapping[1][1], "Vec<(AccountId, Data)>")

        obj = ScaleDecoder.get_decoder_class('Vec<UncleEntryItem<BlockNumber, Hash, AccountId>>')

        self.assertEqual(obj.sub_type, "UncleEntryItem<BlockNumber, Hash, AccountId>".lower())


    # TODO make type_index in Metadatadecoder and add tests if all types are supported
