"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.EventsRuleToStepFunction = void 0;
const events = require("@aws-cdk/aws-events");
const defaults = require("@aws-solutions-constructs/core");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("@aws-solutions-constructs/core");
class EventsRuleToStepFunction extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the EventsRuleToStepFunction class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {EventsRuleToStepFunctionProps} props - user provided props for the construct
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        [this.stateMachine, this.stateMachineLogGroup] = defaults.buildStateMachine(this, props.stateMachineProps);
        // Create an IAM role for Events to start the State Machine
        const eventsRole = new iam.Role(this, 'EventsRuleRole', {
            assumedBy: new iam.ServicePrincipal('events.amazonaws.com')
        });
        // Grant the start execution permission to the Events service
        this.stateMachine.grantStartExecution(eventsRole);
        // Setup the Events target
        const stateMachine = {
            bind: () => ({
                id: '',
                arn: this.stateMachine.stateMachineArn,
                role: eventsRole
            })
        };
        // Defaults props for the Events
        const defaultEventsRuleProps = defaults.DefaultEventsRuleProps([stateMachine]);
        // Override the defaults with the user provided props
        const eventsRuleProps = core_2.overrideProps(defaultEventsRuleProps, props.eventRuleProps, true);
        // Create the Events Rule for the State Machine
        this.eventsRule = new events.Rule(this, 'EventsRule', eventsRuleProps);
        // Deploy best practices CW Alarms for State Machine
        this.cloudwatchAlarms = defaults.buildStepFunctionCWAlarms(this, this.stateMachine);
    }
}
exports.EventsRuleToStepFunction = EventsRuleToStepFunction;
//# sourceMappingURL=data:application/json;base64,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