/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
export declare enum DynamoConsumedCapacity {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation, together with ConsumedCapacity for each table and secondary index that was accessed.
     */
    INDEXES = "INDEXES",
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    TOTAL = "TOTAL",
    /**
     * No ConsumedCapacity details are included in the response.
     */
    NONE = "NONE"
}
/**
 * Determines whether item collection metrics are returned.
 */
export declare enum DynamoItemCollectionMetrics {
    /**
     * If set to SIZE, the response includes statistics about item collections, if any, that were modified during the operation.
     */
    SIZE = "SIZE",
    /**
     * If set to NONE, no statistics are returned.
     */
    NONE = "NONE"
}
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed.
 */
export declare enum DynamoReturnValues {
    /**
     * Nothing is returned.
     */
    NONE = "NONE",
    /**
     * Returns all of the attributes of the item.
     */
    ALL_OLD = "ALL_OLD",
    /**
     * Returns only the updated attributes.
     */
    UPDATED_OLD = "UPDATED_OLD",
    /**
     * Returns all of the attributes of the item.
     */
    ALL_NEW = "ALL_NEW",
    /**
     * Returns only the updated attributes.
     */
    UPDATED_NEW = "UPDATED_NEW"
}
/**
 * Class to generate projection expression.
 */
export declare class DynamoProjectionExpression {
    private expression;
    /**
     * Adds the passed attribute to the chain.
     *
     * @param attr Attribute name.
     */
    withAttribute(attr: string): DynamoProjectionExpression;
    /**
     * Adds the array literal access for passed index.
     *
     * @param index array index.
     */
    atIndex(index: number): DynamoProjectionExpression;
    /**
     * converts and return the string expression.
     */
    toString(): string;
}
/**
 * Represents the data for an attribute.
 *
 * Each attribute value is described as a name-value pair.
 * The name is the data type, and the value is the data itself.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_AttributeValue.html
 */
export declare class DynamoAttributeValue {
    /**
     * Sets an attribute of type String.
     *
     * For example:  "S": "Hello"
     * Strings may be literal values or as JsonPath
     */
    static fromString(value: string): DynamoAttributeValue;
    /**
     * Sets a literal number.
     *
     * For example: 1234
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumber(value: number): DynamoAttributeValue;
    /**
     * Sets an attribute of type Number.
     *
     * For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberFromString(value: string): DynamoAttributeValue;
    /**
     * Sets an attribute of type Binary.
     *
     * For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     *
     * @param value base-64 encoded string.
     */
    static fromBinary(value: string): DynamoAttributeValue;
    /**
     * Sets an attribute of type String Set.
     *
     * For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    static fromStringSet(value: string[]): DynamoAttributeValue;
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    static fromNumberSet(value: number[]): DynamoAttributeValue;
    /**
     * Sets an attribute of type Number Set.
     *
     * For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     *
     * Numbers may be expressed as literal strings or as JsonPath
     */
    static numberSetFromStrings(value: string[]): DynamoAttributeValue;
    /**
     * Sets an attribute of type Binary Set.
     *
     * For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    static fromBinarySet(value: string[]): DynamoAttributeValue;
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    static fromMap(value: {
        [key: string]: DynamoAttributeValue;
    }): DynamoAttributeValue;
    /**
     * Sets an attribute of type Map.
     *
     * For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     *
     * @param value Json path that specifies state input to be used.
     */
    static mapFromJsonPath(value: string): DynamoAttributeValue;
    /**
     * Sets an attribute of type List.
     *
     * For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    static fromList(value: DynamoAttributeValue[]): DynamoAttributeValue;
    /**
     * Sets an attribute of type Null.
     *
     * For example:  "NULL": true
     */
    static fromNull(value: boolean): DynamoAttributeValue;
    /**
     * Sets an attribute of type Boolean.
     *
     * For example:  "BOOL": true
     */
    static fromBoolean(value: boolean): DynamoAttributeValue;
    /**
     * Sets an attribute of type Boolean from state input through Json path.
     *
     * For example:  "BOOL": true
     *
     * @param value Json path that specifies state input to be used.
     */
    static booleanFromJsonPath(value: string): DynamoAttributeValue;
    /**
     * Represents the data for the attribute.
     *
     * Data can be
     * i.e. "S": "Hello"
     */
    readonly attributeValue: any;
    private constructor();
    /**
     * Returns the DynamoDB attribute value.
     */
    toObject(): any;
}
